<?php
require_once 'iplan/orm/ORMDefinition.php';
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORMObject.php';
require_once 'iplan/database/Connection.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Este objeto se crea indicando cu�l es la clase "pivote", con cuyo nombre recupera la definici�n desde el ORM.
 * 
 * Las condiciones de filtrado se especifican a trav�s de llamados a filterBy() y el conjunto de resultados se ordena seg�n el orden de llamados a orderBy().
 * 
 * Es posible segmentar el conjunto de resultados usando los pares de funciones: from()/to() � from()/top(), seg�n se quieran indicar fila de inicio y fin o fila de inicio y cantidad de registros respectivamente.
 * 
 * El resultado de la consulta se obtiene llamando a find().
 */
class ORMQuery {
  /**
   * @var string el nombre de la clase pivote de la consulta
   */
  public $pivotClass;

  /**
   * @var int el n�mero de fila del primer registro del resultado.
   */
  public $from;

  /**
   * @var int la cantidad de filas a recuperar.
   */
  public $count;

  /**
   * @var ORMDefinition la definici�n de la clase pivote
   */
  private $definition;

  /**
   * @var ORM el manejador de objetos desde el cual se toman las definiciones
   */
  public $orm;

  /**
   * @var array un arreglo con los nombres de columna a recuperar, si es que se invoc� a select()
   */
  public $columns;

  /**
   * @var array un arreglo con las condiciones indicadas por los llamados a filterBy()
   * El formato del arreglo es:
   * $conditions['filters'][$field][$operator][0..n]=array($value, $other, $connector);
   * $conditions['subqueries'][$field][0..n]=ORMQuery();
   */
  public $conditions;

  /**
   * @var array un arreglo con los ordenamientos establecidos por los llamados a orderBy()
   */
  public $sorts;

  /**
   * @var boolean determina si el usuario ha o no especificado columnas personalizadas para el resultado
   */
  public $uses_custom_columns;

  /**
   * @var boolean indica si se han especificado atributos de agrupaci�n
   */
  public $uses_groups;

  /**
   * Crea un objeto de consulta que retorna elementos del tipo de la clase indicada.
   * 
   * @param string $class la clase de objetos que retornar� la consulta.
   * @param ORM $orm el objeto ORM al que se ata la consulta.
   * @return ORMQuery un objeto con los criterios de una consulta.
   */
  public function __construct($class, &$orm)
  {
    // Bouml preserved body begin 00096C05
    $this->orm=$orm;
    $this->pivotClass=$class;
    $this->definition=$orm->getDefinition($class);
    $this->uses_custom_columns = false;
    $this->uses_groups = false;
    return $this;
    // Bouml preserved body end 00096C05
  }

  /**
   * Establece la cantidad de filas a saltear del conjunto de resultados antes de considerarla v�lida como respuesta.
   * 
   * @param int $row la fila a partir de la cual deseamos que se nos retornen resultados.
   * 
   * @return ORMQuery La consulta modificada. from() siempre debe llamarse antes que to() para calcular el diferencial.
   */
  public function from($row)
  {
    // Bouml preserved body begin 00098705
    if (isset($this->from) && isset($this->count)) {
        $tmpCount = $this->count;
        $tmpCount += ($this->from - $row);
        if ($tmpCount > 0) $this->count = $tmpCount;
    }
    $this->from = $row;
    return this;
    // Bouml preserved body end 00098705
  }

  /**
   * Establece el m�ximo n�mero de fila que quiere recuperarse. Debe ser mayor al valor prefijado en from().
   * 
   * @param int $row el n�mero de fila m�xima del conjunto de resultado.
   * 
   * @return ORMQuery La consulta con el nuevo l�mite aplicado. Debe considerarse que si no se llam� a from() previamente no se va a calcular el diferencial.
   */
  public function to($row)
  {
    // Bouml preserved body begin 00098785
    if (isset($this->from) && isset($this->count)) {
        $tmpTo = $this->from+$this->count;
        $tmpTo += ($row-$tmpTo);
        if ($tmpCount > 0) $this->count = $tmpCount;
    } else {
        $this->count = $row-$this->from;
    }
    return this;
    // Bouml preserved body end 00098785
  }

  /**
   * Establece cuantas filas se deben recuperar del conjunto de resultados.
   * 
   * @param int $limit la cantidad de filas a recuperar
   * 
   * @return ORMQuery El objeto Query modificado.
   */
  public function top($limit)
  {
    // Bouml preserved body begin 00098805
    $this->count = $limit;
    return this;
    // Bouml preserved body end 00098805
  }

  /**
   * Ordena el conjunto de resultados por la columna indicada.
   * 
   * @param string $attribute el atributo por el cual se desea ordenar el resultado. No se puede ordenar por atributos relaci�n, ni por los atributos de los elementos de las relaciones o de los atributos entidad.
   * @param string $order puede ser 'ASC' o 'DESC'
   * 
   * @return ORMQuery La consulta modificada como se indic�.
   */
  public function orderBy($attribute, $order = 'ASC')
  {
    // Bouml preserved body begin 00098685
    if ($pointPos = strpos($attribute, '.')) {
        $def = $this->definition->getFieldDefinition(substr($attribute, 0, $pointPos));
    } else {
        $def = $this->definition->getFieldDefinition($attribute);
    }
    $this->sorts[$def['fieldName']]=$def['fieldName'] . " $order";
    return $this;
    // Bouml preserved body end 00098685
  }

  /**
   * Establece una condici�n de b�squeda para la consulta.
   * 
   * @param string $attribute el campo sobre el cual se especifica la condici�n
   * @param int $operator la operaci�n de comparaci�n
   * @param mixed $value el valor utilizado para la comparaci�n
   * @param mixed $other otro valor en caso de ser necesario
   * @param string $connector la palabra 'AND' o 'OR' seg�n se quiera utilizar para concatenar la condici�n
   * 
   * @return ORMQuery la consulta con la condici�n incorporada.
   */
  public function filterBy($attribute, $operator, $value, $other = "", $connector = 'AND')
  {
    // Bouml preserved body begin 00098605

       //Valido si hay subquerys (una subquery se da cuando se filtra por attributo1.attributo2=valor
      if ($pointPos = strpos($attribute, '.')) {
         $currentField=substr($attribute, 0,$pointPos);
         if (self::$debug) echo "<br>Detectado: $attribute, generando subquery para $currentField";
         $targetClassDefinition = $this->definition->getFieldDefinition($currentField);
         if ($targetClassDefinition === false) {
             $targetClassDefinition = $this->definition->getRelationDefinition($currentField);
         }
         if ($targetClassDefinition != false) {
             if (self::$debug) echo "<br>         new Query(".$targetClassDefinition['class'].")";
             if (self::$debug) echo "<br>             filterBy(".substr($attribute,$pointPos+1).", $operator, $value, $other, $connector)<br>";
             $this->conditions['subqueries'][$currentField][]=$this->orm->query($targetClassDefinition['class'])
                                                                         ->filterBy(substr($attribute,$pointPos+1), $operator, $value, $other, $connector);
         } else {
             throw new UnknowAttribute("El atributo $attribute no está definido en la clase ".$this->pivotClass);
         }
      } else {
        $this->conditions['filters'][$attribute][strtoupper($operator)][]=array($value, $other, $connector);
      }
      return $this;
    // Bouml preserved body end 00098605
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra clase, utilizando para el emparejamiento las condiciones dadas e incorporando al conjunto de resultados los atributos indicados.
   * @param string $attribute el atributo que determina la relaci�n con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $attributes los atributos nuevos a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados
   */
  public function joinWith($attribute, $alias, $attributes, $conditions)
  {
    // Bouml preserved body begin 000C8A85
    // Bouml preserved body end 000C8A85
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra consulta. Debido a que no se pueden inferir los resultados de la consulta tan f�cilmente como con el joinWith, a esta funci�n se le deben pasar los nombres de columnas devueltas por el query para hacer el emparejamiento y para definir nuevos campos para el conjunto de resultados.
   * 
   * @param ORMQuery $query la query con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $fields los campos nuevos (con denominaci�n completa) a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados tambi�n con denominaci�n completa
   */
  public function joinBy($query, $alias, $fields, $conditions)
  {
    // Bouml preserved body begin 000C8A05
    // Bouml preserved body end 000C8A05
  }

  /**
   * Indica cu�les atributos se desea recuperar. Este m�todo es privado y s�lo para uso interno, dado que si se quiere obtener Objetos se debe usar find() o findOne() y, por el contrario, si se desea obtener valores de columna use select().
   */
  protected function recover($attributes)
  {
    // Bouml preserved body begin 000C5285
    $this->columns=$attributes;
    return $this;
    // Bouml preserved body end 000C5285
  }

  /**
   * Retorna el c�digo a ejecutar para resolver la query en formato codificado.
   * 
   * @return string
   */
  public function build()
  {
    // Bouml preserved body begin 0009A285
    $db = $this->orm->getDatabase();
    $filters = "";
    if (self::$debug) {
        self::$level++;
        if (self::$level == 1) echo "<pre>";
        echo $this->echoTab ().$this->pivotClass."-&gt;build();";
    }
    if (isset($this->conditions['filters'])) {
        if (self::$debug) echo $this->echoTab()."Hay filtros";
        foreach($this->conditions['filters'] as $attribute => $attributeFilter) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            foreach($attributeFilter as $operator=>$values) {
                foreach($values as $data) {
                    list($value, $other, $connector) = $data;
                    if (self::$debug) echo $this->echoTab()."   Filtrando por atributo $attribute $operator $value $other\n";
                    switch (strtolower($operator)) {
                        case 'between':
                            $filters[] = $db->makeCondition($operator, $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                            break;
                        default:
                            $filters[] = $db->makeCondition($operator, $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                    }
                    
                }
            }
        }
    }

    $keyDef = $this->definition->getFieldDefinition('id');
    if (is_null($this->columns)) {
        if (self::$debug) echo $this->echoTab()."Creando query = sql_filter(".$keyDef['table'].", array(".$keyDef['fieldName']."), '',filters)";
        $query = $db->sql_filter($keyDef['table'], array($keyDef['fieldName']), "", $filters);
    } else {
        if (self::$debug) echo $this->echoTab()."Creando query = sql_filter(".$keyDef['table'].", this->columns, '',filters)";
        $query = $db->sql_filter($keyDef['table'], $this->columns, "", $filters);
    }
    if (self::$debug) {
        echo $this->echoTab();
        var_dump($query);
    }

    if (isset($this->conditions['subqueries'])) {
        $myIdDef = $this->definition->getFieldDefinition('id');
        foreach($this->conditions['subqueries'] as $attribute=>$arrQuerys) {
            $relDef = $this->definition->getRelationDefinition($attribute);
            if (!$relDef) { //Si no es una relación (según su definición) puede ser un atributo "entidad"
                $relDef = $this->definition->getFieldDefinition($attribute);
                //Si no es relación ni atributo tiro un error
                if (!$relDef) throw new UnknowAttribute ("No se encuentra la definición de $attribute en la clase ".$this->pivotClass);
            }
            //Para el caso de las relaciones MxN hay que hacer un caso donde se haca un select de la tabla intermedia y luego en la tabla nativa
            if (self::$debug) echo $this->echoTab()."switch(".$relDef['type'].")<br>";
            switch($relDef['type']) {
                case ORM_RELATION_TYPE::OneToMany:
                case ORM_RELATION_TYPE::OneToOne:
                    //Los casos de 1:N y 1:1 son más sencillos... del lado del 1: siempre se busca que el campo
                    //id en la propia tabla esté IN los que retorna el subquery. Por demás está decir que si en
                    //una definición estámos indicando una relación 1:N es porque ESTAMOS parados en el lado del
                    //1: de lo contrario estaríamos diciendo que es un atributo entidad ó, en muy raras ocasiones,
                    //1:1 (en cuyo caso invariablemente estamos del lado del 1:)
                    if (self::$debug) {
                        echo $this->echoTab()."   caso OneToMany | OneToOne";
                        echo $this->echoTab()."   add_condition(query, ".$myIdDef['table'].", ".$relDef['fieldName'].", IN, subQuery)<br>";
                    }
                    foreach($arrQuerys as $subQuery) {
                        if (self::$debug) {
                            echo $this->echoTab()."      query = ".unpack("H*", $query)."<br>";
                            echo $this->echoTab()."      subquery = ".unpack("H*", $subQuery->build())."<br>";
                        }
//ANTES DE RECOVER()                        $query = $db->sql_add_condition($query, $myIdDef['table'], $relDef['fieldName'], 'IN', $subQuery->build());
                        $query = $db->sql_add_condition($query, $myIdDef['table'], $myIdDef['fieldName'], 'IN', $subQuery->recover(array($relDef['fieldName']))->build());
                    }
                    break;
                case ORM_TYPES::ORM_ENTITY:
                    //Es muy muy similar al 1:1 pero con la diferencia de que en vez de buscar que mi id esté
                    //entre los resultados del $subQuery, lo que verifico es que un campo arbitrario de mi tabla
                    //esté entre dichos resultados.
                    if (self::$debug)  {
                        echo $this->echoTab()."   caso ORM_ENTITY";
                        echo $this->echoTab()."   add_condition(query, ".$myIdDef['table'].", ".$relDef['fieldName'].", IN, subQuery)<br>";
                    }
                    foreach($arrQuerys as $subQuery) {
                        $subQueryCoded = $subQuery->build();
                        $query = $db->sql_add_condition($query, $myIdDef['table'], $relDef['fieldName'], 'IN', $subQueryCoded);
                    }
                    break;
                case ORM_RELATION_TYPE::ManyToMany:
                    if (self::$debug) echo $this->echoTab()."   caso ManyToMany";
                    foreach($arrQuerys as $subQuery) {
                        //Hago una query para traer el id relacionado a $query (el que se va a comparar con IN desde $query)
                        //Reseña: si es M:N tenemos ya hecha la $subQuery que filtra la tabla destino por la condición dada.
                        //        Ahora tenemos que vincular la tabla intermedia con los "id" que retorna dicha $subQuery;
                        //        para ello creamos una $subQuery2 sobre la tabla intermedia donde vamos a "recuperar" el
                        //        campo que contiene la clave final con el que vamos a hacer el IN de la clase pivot...
                        $subQuery2 = $db->sql_filter($relDef['table'], array($relDef['fieldName']));
                        if (self::$debug) echo $this->echoTab()."   creando consulta secundaria: sql_filter(".$relDef['table'].", ". $relDef['fieldName'].");";
                        //     ...pero para filtrar los ids que nos interesan de la tabla intermedia tenemos que ver que
                        //        sólo se recuperen los ids que se condigan con la condición de $subQuery. Para ello le
                        //        decimos que nos agregue la condición de que el campo asociado al id de la otra entidad
                        //        esté dentro de los resultados de $subQuery...
                        $subQuery2 = $db->sql_add_condition($subQuery2, $relDef['table'], $relDef['associatedFieldName'], 'IN', $subQuery->build());
                        //     ...finalmente de ese resultado le hacemos el IN a la query original
                        $query = $db->sql_add_condition($query, $myIdDef['table'], $myIdDef['fieldName'], 'IN', $subQuery2);
                        if (self::$debug) echo $this->echoTab()."   agregada restricci&oacute;n M:N : sql_add_condition(query, ".$myIdDef['table'].", ". $myIdDef['fieldName'].", IN,". "subquery);";
                    }
                    break;
                default:
                    throw new UnknowORM("No se encontró un tipo adecuado en la definición de $attribute en ".$this->pivotClass);
                    break;
            }
            if (self::$debug) echo $this->echoTab()."END switch(".$relDef['type'].")<br>";
        }
    }
    if (self::$debug) {
        if (self::$level == 1) echo "</pre>";
        self::$level--;
    }
    return $query;
    // Bouml preserved body end 0009A285
  }

  /**
   * Recupera el conjunto de resultados seg�n los par�metros configurados en el el objeto.
   * 
   * @return array Un arreglo con todos los elementos que respondieron positivamente a las condiciones indicadas.
   */
  public function find()
  {
    // Bouml preserved body begin 00098885
    $db = $this->orm->getDatabase();
    $query = $this->build();
    if (self::$debug) {
        $conn = $db->getConnection();
        $ps = $conn->prepareStatement("SELECT PKG_WEB_ACCESSOR.DECRYPT(:salida) as salida FROM DUAL");
        $conn->bind($ps, ":salida", $query,-1, Connection::T_BINARY);
        $resQuery = $conn->execute($ps);
        $data = $conn->fetchAll($ps);
        var_dump($data);
    }
    $result = $db->sql_execute($query, $this->sorts, $this->from, $this->count);
    if ($result !== false) {
        $keyDef = $this->definition->getFieldDefinition('id');
        foreach($result as $res) {
            $objs[] = $this->orm->load($this->pivotClass, $res[$keyDef['fieldName']]);
        }
        return $objs;
    } else {
        return false;
    }
    // Bouml preserved body end 00098885
  }

  /**
   * Busca el primer elemento de la consulta.
   * 
   * @return ORMObject el objeto en cuesti�n
   */
  public function findOne()
  {
    // Bouml preserved body begin 0009D785
    $res = $this->find();
    if ($res === false)
        return false;
    else
        return $res[0];
    // Bouml preserved body end 0009D785
  }

  /**
   * Transforma la consulta indicando que en vez de objetos, lo que se pretende obtener es el conjunto de campos.
   * 
   * @param array $attributes un arreglo con la lista de las columnas que se quieren obtener. Por el momento solamente se obtienen campos de la primer tabla de la clase "pivot".
   * 
   * Nota 30/03/2011: ahora se pueden traer campos de cualquier nivel de la consulta :)
   * 
   * @return array Un arreglo con los valores de las columnas.
   */
  public function select($attributes)
  {
    // Bouml preserved body begin 00098905
    foreach($attributes as $attribute) {
        $def=$this->definition->getFieldDefinition($attribute);
        $this->columns[] = $def['fieldName']. " AS $attribute";
    }
    $db = $this->orm->getDatabase();
    $query = $this->build();
    $result = $db->sql_execute($query, $this->sorts, $this->from, $this->count);
    if ($result != false) {
        return $result;
    } else {
        throw new UnknowORM('Fallo al ejecutar la consulta sobre '.$this->pivotClass);
    }
    // Bouml preserved body end 00098905
  }

  /**
   * @return string Imprime un cierto n�mero de espacios acorde al $level
   */
  public function echoTab()
  {
    // Bouml preserved body begin 0009F105
    echo str_pad("\n", 5 * self::$level, " ", STR_PAD_RIGHT);
    // Bouml preserved body end 0009F105
  }

  /**
   * @var int $level utilizado para fines de debug
   */
  static $level = 0;

  /**
   * @var boolean $debug indica si se debe hacer una salida de depuraci�n de los pasos de la consulta
   */
  static $debug = false;

  /**
   * Funci�n temporal s�lo para fines de debug. Decifra la cadena retornada por "build()".
   */
  public function decrypt(&$coded_sql)
  {
    // Bouml preserved body begin 000BEB05
    $db = $this->orm->getDatabase();
    $conn = $db->getConnection();
    $ps = $conn->prepareStatement("SELECT PKG_WEB_ACCESSOR.DECRYPT(:salida) as salida FROM DUAL");
    $conn->bind($ps, ":salida", $coded_sql,-1, Connection::T_BINARY);
    $resQuery = $conn->execute($ps);
    return $conn->fetchAll($ps);
    // Bouml preserved body end 000BEB05
  }

  /**
   * Establece una condici�n de b�squeda para la consulta.
   * 
   * @param string $attribute el campo sobre el cual se especifica la condici�n
   * @param int $operator la operaci�n de comparaci�n
   * @param mixed $value el valor utilizado para la comparaci�n
   * @param mixed $other otro valor en caso de ser necesario
   * @param string $connector la palabra 'AND' o 'OR' seg�n se quiera utilizar para concatenar la condici�n
   * 
   * @return ORMQuery la consulta con la condici�n incorporada.
   */
  public function filterBy2($attribute, $operator, $value, $other = "", $connector = 'AND')
  {
    // Bouml preserved body begin 000CA605
    if ($pointPos = strpos($attribute, '.')) {
       $atts = explode('.', $attribute);

       $currentSlot = &$this->conditions['filters'];
       $i = 1;
       foreach($atts as $key => $att) {
           if (count($atts) == $i++) {
              $currentSlot = &$currentSlot['locals'][$att];
           } else {
              $currentSlot = &$currentSlot['subquerys'][$att];
           }
       }
       $currentSlot[strtoupper($operator)][]=array($value, $other, $connector);
    } else {
      $this->conditions['filters']['locals'][$attribute][strtoupper($operator)][]=array($value, $other, $connector);
    }
    return $this;
    // Bouml preserved body end 000CA605
  }

  /**
   * Ordena el conjunto de resultados por la columna indicada.
   * 
   * @param string $attribute el atributo por el cual se desea ordenar el resultado. No se puede ordenar por atributos relaci�n, ni por los atributos de los elementos de las relaciones o de los atributos entidad.
   * @param string $order puede ser 'ASC' o 'DESC'
   * 
   * @return ORMQuery La consulta modificada como se indic�.
   */
  public function orderBy2($attribute, $order = 'ASC')
  {
    // Bouml preserved body begin 000CA685
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['sorts'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['sorts'][]=$attribute;
        }
    }
    return $this;
    // Bouml preserved body end 000CA685
  }

  /**
   * Retorna el c�digo a ejecutar para resolver la query en formato codificado.
   * 
   * @param array $conditions las condiciones a analizar �, si es null, toma las globales
   * @param int $alias_counter un numero que se a�ade al alias de cada tabla
   * @return string el SQL codificado de la instrucci�n
   */
  public function build2()
  {
    // Bouml preserved body begin 000CA705
    $db = $this->orm->getDatabase();
    $conditions = $this->conditions['filters'];
    $alias_counter = 1;
    $localFilters = "";
    $conditions['alias']="t$alias_counter";
    $conditions['alias.']=$conditions['alias'].".";
    $groups = array();
    $sorts  = array();

    //Creo el select principal con los locals
    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $attribute => $filters) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            foreach($filters as $operator => $valueSets) {
                foreach($valueSets as $valueSet) {
                    list($value, $other, $connector) = $valueSet;
                    switch ($operator) {
                        case 'BETWEEN':
                            $localFilters[] = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                            break;
                        default:
                            $localFilters[] = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                    }

                }
            }
        }
    }
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            $sorts[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

    $keyDef = $this->definition->getFieldDefinition('id');
    if (!isset($conditions['columns'])) {
        $query = $db->sql_filter($keyDef['table'], array($keyDef['fieldName']), $conditions['alias'], $localFilters);
    } else {
        foreach($conditions['columns'] as $att) {
            $colDef = $this->definition->getFieldDefinition($att);
            $colDefs[]=$colDef['fieldName'];
        }
        $query = $db->sql_filter($keyDef['table'], $colDefs, $conditions['alias'], $localFilters);
    }

    //Creo los subquerys
    $alias_counter++;
    if (isset($conditions['subquerys'])) {
        foreach($conditions['subquerys'] as $attribute => $subconditions) {
            list($query, $subSorts, $subGroups) = $this->build2_subquery($query, $this->definition, $attribute, $conditions['subquerys'][$attribute], $conditions['alias'], $alias_counter);
            $sorts = array_merge($sorts, $subSorts);
            $groups = array_merge($groups, $subGroups);
        }
    }
    if (count($sorts)==0)  $sorts = null;
    if (count($groups)==0) $groups = null;
    else {
        $query = $db->sql_set_from($query, $groups); //Reemplazo las columnas del SELECT
    }
    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA705
  }

  public function build2_subquery(&$query, $pivotDefinition, $attribute, &$conditions, $parent_alias, &$alias_counter)
  {
    // Bouml preserved body begin 000CA885
    $keyDef = $pivotDefinition->getFieldDefinition('id');
    $db = $this->orm->getDatabase();

    $myAliasCounter = $alias_counter++;
    $conditions['alias'] = "t$myAliasCounter";
    $conditions['alias.'] = $conditions['alias'].".";
    $sorts = array();
    $groups = array();

    $join_conditions = null;
    $relDefinition = $pivotDefinition->getRelationDefinition($attribute);
    if ($relDefinition === false) $relDefinition = $pivotDefinition->getFieldDefinition($attribute);

    $newPivotDefinition = $this->orm->getDefinition($relDefinition['class']);

    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $NxM_alias = $alias_counter++;
        $theOtherSideKey = $newPivotDefinition->getFieldDefinition('id');
        $join_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$relDefinition['associatedFieldName'], $conditions['alias.'].$theOtherSideKey['fieldName']);
    } else {
        $join_conditions[] = $db->makeCondition('=', $conditions['alias.']. $relDefinition['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
    }

    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $att => $filters) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($att);
            foreach($filters as $operator => $valueSets) {
                foreach($valueSets as $valueSet) {
                    list($value, $other, $connector) = $valueSet;
                    switch ($operator) {
                        case 'BETWEEN':
                            $join_conditions[] = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                            break;
                        default:
                            $join_conditions[] = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                    }

                }
            }
        }
    }
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($attribute);
            $sorts[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

    if (isset($conditions['subquerys']))
        foreach($conditions['subquerys'] as $att => $subconditions) {
            list($query, $subSorts, $subGroups) = $this->build2_subquery($query, $newPivotDefinition, $att, $conditions['subquerys'][$att], $conditions['alias'], $alias_counter);
            $sorts = array_merge($sorts, $subSorts);
            $groups = array_merge($groups, $subGroups);
        }

    $newPivotKeyField = $newPivotDefinition->getFieldDefinition('id');
    if (isset($conditions['columns']))
        foreach($conditions['columns'] as $att) {
            $colDef = $newPivotDefinition->getFieldDefinition($att);
            $colDefs[]=$colDef['fieldName'];
        }

    $query = $db->sql_join_table($query, $newPivotKeyField['table'], $conditions['alias'], (isset($conditions['columns'])?$colDefs:null), $join_conditions);

    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $newPivotKeyField = $newPivotDefinition->getFieldDefinition('id');
        $NxM_conditions = array();
        $NxM_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$keyDef['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
        $query = $db->sql_join_table($query, $relDefinition['table'], "t$NxM_alias", null, $NxM_conditions);

    }

    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA885
  }

  /**
   * Recupera el conjunto de resultados seg�n los par�metros configurados en el el objeto.
   * 
   * @param int $from indica desde qu� elemento se deben recuperar los resultados.
   * @param int $count indica cu�ntos elementos de resultado se deben recuperar.
   * @return array Un arreglo con todos los elementos que respondieron positivamente a las condiciones indicadas.
   */
  public function find2($from = null, $count = null)
  {
    // Bouml preserved body begin 000CA785
    $db = $this->orm->getDatabase();
    if (!is_null($from))  $this->from  = $from;
    if (!is_null($count)) $this->count = $count;

//    $query = $this->build2();
    list ($query, $sorts, $groups) = $this->build2();
    
    $result = $db->sql_execute2($query, $groups, $sorts, $this->from, $this->count);
    if ($result !== false) {
        if ($this->uses_custom_columns) {
            return $result;
        } else {
            $keyDef = $this->definition->getFieldDefinition('id');
            foreach($result as $res) {
                $objs[] = $this->orm->load($this->pivotClass, $res[$keyDef['fieldName']]);
            }
            return $objs;
        }
    } else {
        return false;
    }
    // Bouml preserved body end 000CA785
  }

  /**
   * Define qu� atributos recuperar del conjunto de resultados.
   * 
   * @param string $attribute un atributo a ser devueltos por la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function attributes($attribute)
  {
    // Bouml preserved body begin 000CA805
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['columns'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['columns'][]=$attribute;
        }
    }
    $this->uses_custom_columns = true;
    return $this;
    // Bouml preserved body end 000CA805
  }

  /**
   * Define qu� atributos deben ser agrupados una vez obtenido el conjunto de resultados masivo.
   * 
   * @param string $attribute un atributo por el cual se debe agrupar el resultado de la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function groupBy($attribute)
  {
    // Bouml preserved body begin 000CC285
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['groups'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['groups'][]=$attribute;
        }
    }
    $this->uses_groups = true;
    $this->uses_custom_columns = true; //Esto se pone porque si se agrupa por X criterio no tiene sentido devolver objetos: ¿cuál devuelvo?
    return $this;
    // Bouml preserved body end 000CC285
  }

}
?>