<?php
require_once 'iplan/database/Connection.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Concrecin de una conexin Oracle
 */
class OracleConnection extends Connection {
  private $host;

  private $port;

  private $schema;

  private $user;

  private $password;

  /**
   * Un arreglo que permite un acceso rpido a los tipos de datos nativos
   */
  private static $oraTypes = array(
	Connection::T_BLOB => SQLT_BLOB,
	Connection::T_BINARY =>  SQLT_BIN,
	Connection::T_BINARY_FILE => SQLT_BFILEE,
	Connection::T_CBLOB => SQLT_CLOB,
	Connection::T_CHAR => SQLT_CHR,
	Connection::T_CHARACTER_FILE => SQLT_CFILEE,
	Connection::T_CURSOR => SQLT_RSET,
	Connection::T_CUSTOM => SQLT_NTY,
	Connection::T_LONG => SQLT_LNG,
	Connection::T_INT => SQLT_INT,
	Connection::T_VARCHAR=>SQLT_CHR
);

  private static $oraTransactionModes = array(Connection::TR_DEFAULT=>OCI_DEFAULT,
         Connection::TR_AUTO_COMMIT=>OCI_COMMIT_ON_SUCCESS,
	 Connection::TR_NO_AUTO_COMMIT=>OCI_DEFAULT);

  private $session_mode;

  /**
   * Almacena temporalmente la conexin
   */
  private $connection = null;

  /**
   * Crea una instancia del objeto en base a los datos especificados
   */
  public function __construct($host, $port, $schema, $user, $password, $session_mode = OCI_DEFAULT)
  {
    // Bouml preserved body begin 00036385
    if (!extension_loaded('oci8')) die("No se ha cargado la extensión OCI8.\n");
    $this->host    = $host;
    if ($port==-1) {
        $port=1521;
    }
    $this->port    = $port;
    $this->schema  = $schema;
    $this->user    = $user;
    $this->password= $password;
    $this->session_mode=$session_mode;
    return $this;
    // Bouml preserved body end 00036385
  }

  /**
   * Cierra la conexin antes de destruir el objeto
   */
  public function __destruct()
  {
if ($this->isConnected()) {
    $this->close();
}
  }

  /**
   * Recupera la cantidad de filas afectadas (modificadas) por la ltima operacin
   */
  public function affectedRows($query)
  {
    // Bouml preserved body begin 00039805
    return oci_num_rows($query);
    // Bouml preserved body end 00039805
  }

  /**
   * Inicia una transaccin
   */
  public function beginTransaction()
  {
    // Bouml preserved body begin 00032885
    $statement = oci_parse($this->connection,"SAVEPOINT StartTransaction");
    return oci_execute($statement, OracleConnection::$oraTransactionModes[Connection::TR_NO_AUTO_COMMIT]);
    // Bouml preserved body end 00032885
  }

  public function bind($preparedStatement, $name, &$value, $size = -1, $type = Connection::T_VARCHAR, $element_type = Connection::T_VARCHAR)
  {
    // Bouml preserved body begin 0003B205
    if ($type != Connection::T_ARRAY)
        return ocibindbyname($preparedStatement, $name, $value, $size, OracleConnection::$oraTypes[$type]);
    else
        return oci_bind_array_by_name($preparedStatement, $name, $value,count($value), $size, OracleConnection::$oraTypes[$element_type]);
    // Bouml preserved body end 0003B205
  }

  /**
   * Cierra la conexin si est abierta
   */
  public function close()
  {
    // Bouml preserved body begin 00032985
    $result = oci_close($this->connection);
    if ($result) $this->connection=null;
    return $result;
    // Bouml preserved body end 00032985
  }

  /**
   * Hace un commit de los cambios hasta el momento
   */
  public function commit()
  {
    // Bouml preserved body begin 00032A05
    return oci_commit($this->connection);
    // Bouml preserved body end 00032A05
  }

  /**
   * Establece la conexin con los parmetros que posee registrados en la clase
   */
  public function connect()
  {
    // Bouml preserved body begin 00032A85
    $strConnectionString ="(DESCRIPTION = (ADDRESS = (PROTOCOL = TCP)(HOST =$this->host)(PORT = $this->port))
                            (CONNECT_DATA =(SERVER = DEDICATED) (SERVICE_NAME = $this->schema)))";
    $connection= $this->connection = oci_connect($this->user, $this->password, $strConnectionString, 'AL32UTF8');
    return $connection;
    // Bouml preserved body end 00032A85
  }

  /**
   * Crea una collection basada en el tipo de datos indicado
   * 
   * @param string $param el nombre del tipo de datos asociado. Generalmente es un TYPE personalizado.
   * @param string $schema el nombre de la base de datos (su esquema)
   * @return resource|false un puntero al recurso o false en caso de problemas
   */
  public function createCollection($type, $schema = NULL)
  {
    // Bouml preserved body begin 0007DE05

    /* Sitios de errores
     * http://marc.info/?l=php-cvs&m=118591630928253
     * http://bugs.php.net/bug.php?id=44113
     * http://pecl.php.net/package-info.php?package=oci8&version=1.2.5
     */
    if (is_null($schema))
        return oci_new_collection($this->connection, $type, $this->schema);
    else
        return oci_new_collection($this->connection, $type, $schema);
    // Bouml preserved body end 0007DE05
  }

  /**
   * Crea un cursor sobre la conexin especificada
   */
  public function createCursor()
  {
    // Bouml preserved body begin 0003CC85
    return oci_new_cursor($this->connection);
    // Bouml preserved body end 0003CC85
  }

  /**
   * Ejecuta el sql indicado y retorna lo que devuelva
   */
  public function execute($statement, $mode = Connection::TR_DEFAULT)
  {
    // Bouml preserved body begin 00032B05
    $result = null;
    if (is_string($statement)) {
        $statement = oci_parse($this->connection, $statement);
    }

    if ($mode==Connection::TR_DEFAULT) {
        try {
            $result= oci_execute($statement);
        }  catch (ErrorException $e) {
            krumo($this->lastError($statement));
            $result = false;
        }
    } else {
        $result = oci_execute($statement, OracleConnection::$oraTransactionModes[$mode]);
    }
    return $result;
    // Bouml preserved body end 00032B05
  }

  /**
   * Ejecuta una funcin en la base de datos y retorna el valor.
   * Es posible, indicar la variable en la cual queremos bindear el resultado, su tipo y tamao en caso de ser necesario. Si la misma no se especifica se utiliza un valor interno y se retorna una copia del valor del resultado.-
   */
  public function executeFunction($name, &$params, &$result = null, $type = Connection::T_VARCHAR, $size = -1)
  {
    // Bouml preserved body begin 0003CB85

    if ($type == Connection::T_CURSOR) {
        $result = $this->createCursor();
    }

    $strParams ="";//Preparo el string de los parámetros
    foreach($params as $key=>$value) {
        $strParams .= ":param$key, ";
    }
    $strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
    $strSQL = "begin :data := $name$strParams; end;";
    $preparedStatement = $this->prepareStatement($strSQL);
    if ($preparedStatement===false) {
        return false;
    }
    
    //Bindeo el resultado con lo que me pasaron o algo propio
    if (oci_bind_by_name($preparedStatement,":data",$result,$size,OracleConnection::$oraTypes[$type])) {
        //Luego los resultados
        foreach($params as $key => $value) {
            if (!oci_bind_by_name($preparedStatement, ":param$key", $params[$key])) {
                return false;
            }
        }
        if ($this->execute($preparedStatement)) {
            if ($type == Connection::T_CURSOR) {
                $this->execute($result);
            }
            return $result;
        } else {
            return false;
        }
    } else {
        return false;
    }
    // Bouml preserved body end 0003CB85
  }

  /**
   * Ejecuta el procedimiento bindeando los valores de los parmetros en el orden en que se encuentren
   */
  public function executeProcedure($name, &$params)
  {
    // Bouml preserved body begin 00037E05
    $strParams ="";//Preparo el string de los parámetros
    foreach($params as $key=>$value) {
        $strParams .= ":param$key, ";
    }
    $strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
    $strSQL = "begin $name$strParams; end;";
    $preparedStatement = $this->prepareStatement($strSQL);
    if ($preparedStatement===false) { return false; }
    //Bindeo los parámetros
    foreach($params as $key => $value) {
        if (!ocibindbyname($preparedStatement, ":param$key", $params[$key])) {
            return false;
        }
    }
    return $this->execute($preparedStatement);
    // Bouml preserved body end 00037E05
  }

  /**
   * Restorna una fila del conjunto de resultados en un arreglo indexado como se indique en $mode y avanza una posicin el cursor del $resource
   */
  public function fetch($resource, $mode = Connection::M_ASSOCIATIVE)
  {
    // Bouml preserved body begin 00032D05
    if ($mode==Connection::M_ASSOCIATIVE) {
        return oci_fetch_array($resource, OCI_ASSOC+OCI_RETURN_NULLS);
    } else {
        return oci_fetch_array($resource, OCI_NUM+OCI_RETURN_NULLS);
    }
    // Bouml preserved body end 00032D05
  }

  /**
   * Retorna un arreglo con los registros recuperados de la consulta. Se provee la facilidad de indicar que se saltee tantos registros como se indiquen en $offest y que se recuperen slo una cierta cantidad de ellos; se una -1 en caso de querer recuperar todas las filas disponibles desde $offset.
   * En forma predeterminada retorna un arreglo asociativo, lo cual puede cambiarse con el parmetro $mode.
   */
  public function fetchAll($resource, $mode = Connection::M_ASSOCIATIVE, $offset = 0, $count = -1)
  {
    // Bouml preserved body begin 00032D85
    if ($mode==Connection::M_ASSOCIATIVE) {
        $flags=OCI_FETCHSTATEMENT_BY_ROW+OCI_ASSOC;
    } else {
        $flags=OCI_FETCHSTATEMENT_BY_ROW+OCI_NUM;
    }
    if (oci_fetch_all($resource, $result, $offset, $count, $flags)) {
        return $result;
    } else {
        return false;
    }
    // Bouml preserved body end 00032D85
  }

  /**
   * Libera un recurso previamente asignado.
   */
  public function free(&$resource)
  {
    // Bouml preserved body begin 00032E05
    return oci_free_statement($resource);
    // Bouml preserved body end 00032E05
  }

  /**
   * Retorna true si la conexin est activa o false en caso contrario
   */
  public function isConnected()
  {
    // Bouml preserved body begin 00032E85

    if (isset($this->connection)) {
        return !is_null($this->connection) && ($this->connection != false);
    } else return false;
    // Bouml preserved body end 00032E85
  }

  /**
   * Retorna un texto que describe el ltimo error que sucedi en la conexin. Si se desea se puede acompaar del handler que retorno la funcin que fall o de lo contrario se retorna el ltimo error que se dio en la conexin.
   */
  public function lastError($over = -1)
  {
    // Bouml preserved body begin 00032F05
    if ($over == -1) {
        $data = oci_error($this->connection);
    } else {
        $data = oci_error($over);
    }

    if ($data == false) {
        return false;
    } else {
        return $data['message'];
    }
    // Bouml preserved body end 00032F05
  }

  /**
   * Retorna el cdigo del ltimo error que se present en la conexin o en el handler del parmetro
   */
  public function lastErrorCode($over = -1)
  {
    // Bouml preserved body begin 00032F85
    if ($over == -1) {
        $data = oci_error($this->connection);
    } else {
        $data = oci_error($over);
    }

    if ($data != false) {
        return $data['code'];
    } else {
        return false;
    }
    // Bouml preserved body end 00032F85
  }

  /**
   * Salta al siguiente conjunto de resultados
   */
  public function nextResultset($handler)
  {
    // Bouml preserved body begin 00033005
    return false;
    // Bouml preserved body end 00033005
  }

  /**
   * Retorna el nmero de columnas del handler del query especificado
   */
  public function numColumns($query)
  {
    // Bouml preserved body begin 00033085
    return oci_num_fields($query);
    // Bouml preserved body end 00033085
  }

  /**
   * Retorna el nmero de filas que devolvi el query especificado
   */
  public function numRows($query)
  {
    // Bouml preserved body begin 00033105
    if (oci_statement_type($query)=="SELECT") {
        return 0;
    } else {
        return oci_num_rows($query);
    }
    // Bouml preserved body end 00033105
  }

  /**
   * Retorna un handler a un prepared statement
   */
  public function prepareStatement($sql)
  {
    // Bouml preserved body begin 00033185
    return oci_parse($this->connection, $sql);
    // Bouml preserved body end 00033185
  }

  /**
   * Ejecuta la instruccin SQL y retorna el resultado.
   */
  public function query($sql)
  {
    // Bouml preserved body begin 00033205
    $statement = oci_parse($this->connection, $sql);
    oci_execute($statement, OracleConnection::$oraTransactionModes[Connection::TR_NO_AUTO_COMMIT]);
    return $statement;
    // Bouml preserved body end 00033205
  }

  /**
   * Si el motor de la base de datos lo soporta, hace un rollback de la ltima transaccin
   */
  public function rollback()
  {
    // Bouml preserved body begin 00033285
    //return oci_rollback($this->connection);
    $statement = oci_parse($this->connection, "ROLLBACK TO SAVEPOINT StartTransaction");
    return oci_execute($statement);
    // Bouml preserved body end 00033285
  }

  /**
   * Retorna la informacin de status del motor
   */
  public function status()
  {
    // Bouml preserved body begin 00033305
    return false;
    // Bouml preserved body end 00033305
  }

}
?>