//#////////////////////////////////////////////////////////////////////////////
//# Copyright (c) 2002 Interstar technologies inc. Permission to use
//# this work for any purpose must be obtained in writing from INTERSTAR
//# TECHNOLOGIES, 5835 Verdun Ave, Suite 302, Montreal, Quebec, H4H 1M1 Canada
//#////////////////////////////////////////////////////////////////////////////

/**
\file XmUser.java
    
\since 2002/10/03

\author Silvain Laplante

*/

import com.interstarinc.xmedius.*;
import java.util.*;

/**
 *  Test program
 */

public class XmUser extends XmBase
{
    /// Private Constants
    private static final int LIST_ALL      = 0;
    private static final int LIST_ONE      = 1;
    private static final int ADD           = 2;
    private static final int MODIFY        = 3;
    private static final int ADD_OR_MODIFY = 4;
    private static final int REMOVE        = 5;
    private static final int REMOVE_FAX    = 8;
    private static final int HELP          = 6;
    private static final int MODIFY_PASS   = 7;

    /// Private Attributes
    private int m_mode              = LIST_ALL;
    private boolean m_bReadFromFile = false;
    private String m_param          = "";


    /// PUBLIC INTERFACE /////////////////////////////////////////////////////

    /// Constructor
    public XmUser()
    {}

    public void execute(String workDescription)
    {
        // Transform workDescription into args
        // TODO
        String[] args = null;

        doWork(args);
    }

    /// main
    public static void main(String[] args)
    {
        XmUser worker = new XmUser();
        worker.doWork(args);
    }


    /// PRIVATE INTERFACE ////////////////////////////////////////////////////

    private void doWork(String[] args)
    {
        System.out.println("XmUser - Add, modify, remove users");
        System.out.println("------------------------------------------------------------\n");

        processArgs(args);

        System.out.println("---------------------------------------------------------END\n");
    }

    private void processArgs(String[] args)
    {
        XmBase.ParsedCommandLine commandLine = null;
        try
        {
            commandLine = XmBase.getParsedCommandLine(args);
        }
        catch (XmediusException ex)
        {
            outputError(ex.toString());
            return;
        }

        // Init private attributes with args
        if (!parseArgs(commandLine.remainingArgs)) {
            return;
        }

        if (m_mode == HELP)
        {
            outputUsage();
            return;
        }

        // Open input files if needed
        java.io.BufferedReader inputFile = null;
        
        if (m_bReadFromFile) {
            inputFile = getBufferedFileReader(m_param);
            if (inputFile == null) {
                return;
            }
        }

        if (!validateLoginInfo(commandLine, true)) {
            return;
        }

        Session session = openSession(commandLine);
        if (session == null) {
            return;
        }

        try
        {
            commandLine.targetSite = getSiteGuidFromSiteName(session, commandLine.targetSite);

            ListNodeId userNodeId = SiteListNodeId.create(SiteListNodeId.Name.USERS, commandLine.targetSite);
            ListNodeId profileNodeId = SiteListNodeId.create(SiteListNodeId.Name.PROFILES, commandLine.targetSite);

            ListNode users = ListNode.getNode(session, userNodeId);
            ListNode profiles = null;

            switch (m_mode)
            {
            case LIST_ALL:
                profiles = ListNode.getNode(session, profileNodeId);
                listAll(users, profiles);
                break;
            case LIST_ONE:
                profiles = ListNode.getNode(session, profileNodeId);
                listOne(users, profiles, m_param);
                break;
            case ADD:
                profiles = ListNode.getNode(session, profileNodeId);
                addUsers(users, profiles, inputFile, m_param);
                break;
            case MODIFY:
                profiles = ListNode.getNode(session, profileNodeId);
                modifyUsers(users, profiles, inputFile, m_param, false);
                break;
            case MODIFY_PASS:
                profiles = ListNode.getNode(session, profileNodeId);
                modifyPassUsers(users, profiles, inputFile, m_param);
                break;
            case ADD_OR_MODIFY:
                profiles = ListNode.getNode(session, profileNodeId);
                modifyUsers(users, profiles, inputFile, m_param, true);
                break;
            case REMOVE:
                removeUsers(users, inputFile, m_param);
                break;
            case REMOVE_FAX:
                removeUsersFax(users, inputFile, m_param);
                break;

            }
        }
        catch (XmediusException ex)
        {
            outputError(ex.getMessage());
        }
        
        // Close file
        if (inputFile != null) {
            try {
                inputFile.close();
            }
            catch (java.io.IOException ex) {
            }
        }
    }

    //# parseArgs #///////////////////////////////////////////////////////////
    private boolean parseArgs(String[] args)
    {
        boolean bNeedParam = false;
        
        for (int i=0;i<args.length;i++)
        {
            if ( (args[i].length() > 0) && (args[i].charAt(0) == '-') )
            {
                if (args[i].equalsIgnoreCase("-l"))
                {
                    m_mode = LIST_ALL;
                    bNeedParam = false;
                }
                else if (args[i].equalsIgnoreCase("-i"))
                {
                    m_mode = LIST_ONE;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-a"))
                {
                    m_mode = ADD;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-m"))
                {
                    m_mode = MODIFY;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-mp"))
                {
                    m_mode = MODIFY_PASS;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-am"))
                {
                    m_mode = ADD_OR_MODIFY;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-r"))
                {
                    m_mode = REMOVE;
                    bNeedParam = true;
                }
                else if (args[i].equalsIgnoreCase("-rf"))
                {
                    m_mode = REMOVE_FAX;
                    bNeedParam = true;
                }

                else if (args[i].equalsIgnoreCase("-f"))
                {
                    m_bReadFromFile = true;
                }
                else if (args[i].equalsIgnoreCase("-h"))
                {
                    m_mode = HELP;
                    bNeedParam = false;
                }
                else
                {
                    outputWarning("Invalid option: " + args[i]);
                }
            }
            else
            {
                m_param = args[i];
            }
        }
        
        if ( (bNeedParam) && (m_param.equals("")) )
        {
            if (m_bReadFromFile)
            {
                outputError("Missing file name");
            }
            else
            {
                outputError("Missing user description");
            }
            return false;
        }
        
        return true;
    }
    
    //# listAll #/////////////////////////////////////////////////////////////
    private void listAll(ListNode users, ListNode profiles)
    {
        outputInformation("--> Getting the list all users");
        
        int count = 0;
        
        // Load the profile list
        ProfileList profileList = new ProfileList();
        if (profileList.load(profiles) == false)
        {
            outputError("Failed to load the profile list");
            return;
        }
        
        PropertyDef[] props = new PropertyDef[3];
        props[0] = PropertyDef.USERS_SMTPADDRESS;
        props[1] = PropertyDef.USERS_PROFILEID;
        props[2] = PropertyDef.USERS_NTACCOUNT;
        
        try
        {
            View view = users.getView(props);
            
            long ids[] = view.recordIds();
            
            for (int i=0;i<ids.length;i++)
            {
                PropertyValue[] values = view.values(ids[i]);
                
                String profileName = profileList.findName(values[1].stringValue());
                if (profileName == null)
                {
                    profileName = values[1].stringValue();
                }
                
                StringBuffer buf = new StringBuffer();
                buf.append(values[0].stringValue());
                buf.append(",");
                buf.append(profileName);
                buf.append(",");
                buf.append(values[2].stringValue());
                
                outputInformation("user: " + buf.toString());
                
                count++;
            }
            
            if (count < 2)
            {
                outputInformation("count: " + count);
            }
            else
            {
                outputInformation("count: " + count);
            }
        }
        catch (XmediusException ex)
        {
            outputError(ex.getMessage());
        }
    }
    
    //# listOne #/////////////////////////////////////////////////////////////
    private void listOne(ListNode users, ListNode profiles,
                         String userDescription)
    {
        Tokenizer tokenizer = new Tokenizer(userDescription);
            
        String smtp = tokenizer.getNextToken("");
        
        outputInformation("--> Getting information on user " + smtp);
        
        // Load the profile list
        ProfileList profileList = new ProfileList();
        if (profileList.load(profiles) == false)
        {
            outputError("Failed to load the profile list");
            return;
        }
        
        PropertyDef[] props = new PropertyDef[4];
        props[0] = PropertyDef.USERS_SMTPADDRESS;
        props[1] = PropertyDef.USERS_PROFILEID;
        props[2] = PropertyDef.USERS_NTACCOUNT;
        props[3] = PropertyDef.USERS_PASSWORD;
        
        try
        {
            View view = users.getView(props);
            
            long ids[] = view.recordIds();
            
            boolean bFound = false;
            
            for (int i=0;i<ids.length;i++)
            {
                PropertyValue[] values = view.values(ids[i]);
                
                if (values[0].stringValue().equalsIgnoreCase(smtp))
                {
                    String profileName = profileList.findName(values[1].stringValue());
                    if (profileName == null)
                    {
                        profileName = values[1].stringValue();
                    }
                    
                    StringBuffer buf = new StringBuffer();
                    buf.append(values[0].stringValue());
                    buf.append(",");
                    buf.append(profileName);
                    buf.append(",");
                    buf.append(values[2].stringValue());
                    buf.append(",");
                    buf.append(values[3].stringValue());
                
                    outputInformation(buf.toString());
                    
                    bFound = true;
                    break;
                }
            }
            
            if (!bFound)
            {
                outputWarning("User not found");
            }
        }
        catch (XmediusException ex)
        {
            outputError(ex.getMessage());
        }
    }

    //# removeUsers #/////////////////////////////////////////////////////////
    private void removeUsers(ListNode users, java.io.BufferedReader inputFile,
                             String userDescription)
    {
        if (inputFile == null)
        {
            outputInformation("--> Removing one user");
        }
        else
        {
            outputInformation("--> Removing users from file: " + userDescription);
        }
        
        // Load the user list
        UserMap userList = new UserMap();
        if (userList.load(users) == false)
        {
            outputError("Failed to load the user list");
            return;
        }

        int count = 0;
        boolean done = false;

        try
        {
            while (!done)
            {
                String line = null;
                
                if (inputFile != null)
                {
                    line = inputFile.readLine();
                }
                else
                {
                    line = userDescription;
                    done = true;
                }
                
                if (line != null)
                {
                    Tokenizer tokenizer = new Tokenizer(line);
                        
                    String smtp = tokenizer.getNextToken(null);
                   
                    if (smtp != null)
                    {
                        /// Find the id corresponding to the smtp address
                        long id = userList.find(smtp);
                        
                        if (id != -1)
                        {
                            try
                            {
                                users.deleteRecord(id);
                                outputInformation(smtp + " has been removed");
                                userList.remove(smtp);
                                count++;
                            }
                            catch (XmediusException ex)
                            {
                                outputError("An error occured while removing " 
                                             + smtp + " : " + ex.getMessage());
                            }
                        }
                        else
                        {
                            outputWarning("User not found. " + smtp + " has not been removed.");
                        }
                    }
                }
                else
                {
                    done = true;
                }
            }
        }
        catch (java.io.IOException ex)
        {
            outputError("An error occured while reading the input file");
        }
        
        if (count < 2)
        {
            outputInformation("<-- " + count + " user has been removed");
        }
        else
        {
            outputInformation("<-- " + count + " users have been removed");
        }
    }
   

    private void removeUsersFax(ListNode users, java.io.BufferedReader inputFile,
                             String userDescription)
    {
        outputInformation("--> Removing one user");
        
        // Load the user list
        
        int count = 0;
        boolean done = false;

           String line = null;
                
           line = userDescription;
           done = true;
           try {     
                if (line != null)
                {
                    Tokenizer tokenizer = new Tokenizer(line);
                        
                    String fax_number = tokenizer.getNextToken(null);
                   
                    if ((fax_number != null) && (!fax_number.equals("")))
                    {
                        /// Find the id corresponding to the smtp address
                        PropertyDef[] props = new PropertyDef[1];
                        props[0] = PropertyDef.USERS_FAX;
                        View view = users.getView(props);

                        long[] ids = view.recordIds();
                        long id = -1;
                        for (int i=0; i<ids.length; i++) {
                            PropertyValue[] values = view.values(ids[i]);
                            if (values[0].stringValue().equals(fax_number))
                                id = ids[i];
                        }

                        if (id != -1)
                        {
                            //try
                            //{
                            users.deleteRecord(id);
                            outputInformation(fax_number + " has been removed");
                                //userList.removeFax(faxNumber);
                            count++;
                            //}
                            //catch (XmediusException ex)
                            //{
                            //    outputError("An error occured while removing " 
                            //                 + fax_number + " : " + ex.getMessage());
                            //}
                        }
                        else
                        {
                            outputWarning("User not found. " + fax_number + " has not been removed.");
                        }
                    }
                }
            } catch (Exception e) {
                outputError("Ha ocurrido un error al procesar la baja" + e.getMessage());
            }
        outputInformation("<-- " + count + " user has been removed");
    }


    //# addUsers #////////////////////////////////////////////////////
    private void addUsers(ListNode users, ListNode profiles,
                          java.io.BufferedReader inputFile,
                          String userDescription)
    {
        if (inputFile == null)
        {
            outputInformation("--> Adding one user");
        }
        else
        {
            outputInformation("--> Adding users from file: " + userDescription);
        }
        
        // Load the profile list
        ProfileList profileList = new ProfileList();
        if (profileList.load(profiles) == false)
        {
            outputError("Failed to load the profile list");
            return;
        }

        // Load the user list
        UserMap userList = new UserMap();
        if (userList.load(users) == false)
        {
            outputError("Failed to load the user list");
            return;
        }

        PropertyValue[] values = new PropertyValue[5];
        values[0] = new PropertyValue(PropertyDef.USERS_SMTPADDRESS);
        values[1] = new PropertyValue(PropertyDef.USERS_PROFILEID);
        values[2] = new PropertyValue(PropertyDef.USERS_NTACCOUNT);
        values[3] = new PropertyValue(PropertyDef.USERS_PASSWORD); // WEB Password
        values[4] = new PropertyValue(PropertyDef.USERS_FAX);

        boolean done = false;
        int count = 0;
        
        try
        {
            while (!done)
            {
                String line = null;
                
                if (inputFile != null)
                {
                    line = inputFile.readLine();
                }
                else
                {
                    line = userDescription;
                    done = true;
                }
                
                if (line != null)
                {
                    Tokenizer tokenizer = new Tokenizer(line);
                        
                    String smtp = tokenizer.getNextToken(null);
                    String profileName = tokenizer.getNextToken("");
                    String account = tokenizer.getNextToken("");
                    String password = tokenizer.getNextToken("");
                    String fax = tokenizer.getNextToken("");

                    if (smtp != null)
                    {
                        /// Find the id corresponding to the smtp address
                        long id = userList.find(smtp);
                        
                        if (id == -1)
                        {
                            // Validate profile
                            String profileId = profileList.findId(profileName);
                            if (profileId != null)
                            {
                                values[0].stringValue(smtp);
                                values[1].stringValue(profileId);
                                values[2].stringValue(account);
                                values[3].stringValue(password);
                                values[4].stringValue(fax);

                                try
                                {
                                    id = users.addRecord(values);
                                    userList.add(smtp, id);
                                    outputInformation(smtp + " has been added");
                                    count++;
                                }
                                catch (XmediusException ex)
                                {
                                    outputError("An error occured while adding " 
                                                + smtp + " : " + ex.getMessage());
                                }
                            }
                            else
                            {
                                outputError("Unknown profile: '" + profileName + "'. " + smtp + " has not been added.");
                            }
                        }
                        else
                        {
                            outputWarning("User already exists. " + smtp + " has not been added.");
                        }
                    }
                }
                else
                {
                    done = true;
                }
            }
        }
        catch (java.io.IOException ex)
        {
            outputError("An error occured while reading the input file");
        }
        
        if (count < 2)
        {
            outputInformation("<-- " + count + " user has been added");
        }
        else
        {
            outputInformation("<-- " + count + " users have been added");
        }
    }
    
    //# modifyUsers #/////////////////////////////////////////////////////////
    private void modifyUsers(ListNode users, ListNode profiles,
                             java.io.BufferedReader inputFile,
                             String userDescription, boolean bAddIfNotExist)
    {
        if (inputFile == null)
        {
            if (bAddIfNotExist)
            {
                outputInformation("--> Adding or modifying one user");
            }
            else
            {
                outputInformation("--> Modifying one user");
            }
        }
        else
        {
            if (bAddIfNotExist)
            {
                outputInformation("--> Adding or modifying users from file: " + userDescription);
            }
            else
            {
                outputInformation("--> Modifying users from file: " + userDescription);
            }
        }
        
        // Load the profile list
        ProfileList profileList = new ProfileList();
        if (profileList.load(profiles) == false)
        {
            outputError("Failed to load the profile list");
            return;
        }

        // Load the user list
        UserMap userList = new UserMap();
        if (userList.load(users) == false)
        {
            outputError("Failed to load the user list");
            return;
        }
        
        PropertyValue[] addValues = new PropertyValue[5];
        addValues[0] = new PropertyValue(PropertyDef.USERS_SMTPADDRESS);
        addValues[1] = new PropertyValue(PropertyDef.USERS_PROFILEID);
        addValues[2] = new PropertyValue(PropertyDef.USERS_NTACCOUNT);
        addValues[3] = new PropertyValue(PropertyDef.USERS_PASSWORD); // WEB Password
        addValues[4] = new PropertyValue(PropertyDef.USERS_FAX);

        int addedCount = 0;
        int modifiedCount = 0;
        boolean done = false;
            
        try
        {
            while (!done)
            {
                String line = null;
                
                if (inputFile != null)
                {
                    line = inputFile.readLine();
                }
                else
                {
                    line = userDescription;
                    done = true;
                }
                
                if (line != null)
                {
                    Tokenizer tokenizer = new Tokenizer(line);
                        
                    String smtp = tokenizer.getNextToken(null);
                    String profileName = tokenizer.getNextToken(null);
                    String account = tokenizer.getNextToken(null);
                    String password = tokenizer.getNextToken(null);
                    String fax = tokenizer.getNextToken(null);
                    
                    if (smtp != null)
                    {
                        /// Find the id corresponding to the smtp address
                        long id = userList.find(smtp);
                        
                        if (id != -1)
                        {
                            // Modify it
                            if (profileName != null)
                            {
                                String profileId = profileList.findId(profileName);
                                if (profileId != null)
                                {
                                    int nbProp = 1;
                                    
                                    if (account != null)
                                    {
                                        nbProp++;
                                    }
                                    if (password != null)
                                    {
                                        nbProp++;
                                    }
                                    
                                    PropertyValue[] modValues = new PropertyValue[nbProp];
                                    
                                    nbProp = 0;
                                    
                                    modValues[nbProp] = new PropertyValue(PropertyDef.USERS_PROFILEID);
                                    modValues[nbProp].stringValue(profileId);
                                    nbProp++;
                                    
                                    if (account != null)
                                    {
                                        modValues[nbProp] = new PropertyValue(PropertyDef.USERS_NTACCOUNT);
                                        modValues[nbProp].stringValue(account);
                                        nbProp++;
                                    }
                                    if (password != null)
                                    {
                                        modValues[nbProp] = new PropertyValue(PropertyDef.USERS_PASSWORD);
                                        modValues[nbProp].stringValue(password);
                                        nbProp++;
                                    }
                                    
                                    try
                                    {
                                        users.setValues(id, modValues);
                                        outputInformation(smtp + " has been modified");
                                        modifiedCount++;
                                    }
                                    catch (XmediusException ex)
                                    {
                                        outputError("An error occured while modifying " 
                                                     + smtp + " : " + ex.getMessage());
                                    }
                                }
                                else
                                {
                                    outputError("Unknown profile: '" + profileName + "'. " + smtp + " has not been modified.");
                                }
                            }
                        }
                        else if (bAddIfNotExist)
                        {
                            if (profileName == null) {
                                profileName = "";
                            }

                            String profileId = profileList.findId(profileName);
                            if (profileId != null)
                            {
                                addValues[0].stringValue(smtp);
                                addValues[1].stringValue(profileId);
                                addValues[2].stringValue( (account == null) ? "":account);
                                addValues[3].stringValue( (password == null) ? "":password);
                                addValues[4].stringValue( (fax == null) ? "":fax);
                    
                                try
                                {
                                    id = users.addRecord(addValues);
                                    userList.add(smtp, id);
                                    outputInformation(smtp + " has been added");
                                    addedCount++;
                                }
                                catch (XmediusException ex)
                                {
                                    outputError("An error occured while adding " 
                                                + smtp + " : " + ex.getMessage());
                                }
                            }
                            else
                            {
                                outputError("Unknown profile: '" + profileName + "'. " + smtp + " has not been added.");
                            }
                        }
                        else
                        {
                            outputWarning("User not found. " + smtp + " has not been modified.");
                        }
                    }
                }
                else
                {
                    done = true;
                }
            }
        }
        catch (java.io.IOException ex)
        {
            outputError("An error occured while reading the input file");
        }
        
        if (modifiedCount < 2)
        {
            outputInformation("<-- " + modifiedCount + " user has been modified");
        }
        else
        {
            outputInformation("<-- " + modifiedCount + " users have been modified");
        }
        
        if (bAddIfNotExist)
        {
            if (addedCount < 2)
            {
                outputInformation("<-- " + addedCount + " user has been added");
            }
            else
            {
                outputInformation("<-- " + addedCount + " users have been added");
            }
        }
    }

    private void modifyPassUsers(ListNode users, ListNode profiles,
                                java.io.BufferedReader inputFile,
                                String userDescription)
    {
        if (inputFile == null)
        {
            outputInformation("--> Modifying one user");
        }
        else
        {
            outputInformation("--> Modifying users from file: " + userDescription);
        }
        
        // Load the profile list
        ProfileList profileList = new ProfileList();
        if (profileList.load(profiles) == false)
        {
            outputError("Failed to load the profile list");
            return;
        }

        // Load the user list
        UserMap userList = new UserMap();
        if (userList.load(users) == false)
        {
            outputError("Failed to load the user list");
            return;
        }
        
        PropertyValue[] addValues = new PropertyValue[2];
        addValues[0] = new PropertyValue(PropertyDef.USERS_SMTPADDRESS);
        addValues[1] = new PropertyValue(PropertyDef.USERS_PASSWORD); // WEB Password

        int addedCount = 0;
        int modifiedCount = 0;
        boolean done = false;
            
        try
        {
            while (!done)
            {
                String line = null;
                
                if (inputFile != null)
                {
                    line = inputFile.readLine();
                }
                else
                {
                    line = userDescription;
                    done = true;
                }
                
                if (line != null)
                {
                    Tokenizer tokenizer = new Tokenizer(line);
                        
                    String old_smtp = tokenizer.getNextToken(null);
                    String new_smtp = tokenizer.getNextToken(null);
                    String password = tokenizer.getNextToken(null);
                    
                    if (old_smtp != null)
                    {
                        /// Find the id corresponding to the smtp address
                        long id = userList.find(old_smtp);
                        
                        if (id != -1)
                        {
                            int nbProp = 0;
                              
                            if ((new_smtp != null) && (!new_smtp.equals("")))
                            {
                                nbProp++;
                            }
                            if ((password != null) && (!password.equals("")))
                            {
                                nbProp++;
                            }
                                
                            PropertyValue[] modValues = new PropertyValue[nbProp];
                                 
                            nbProp = 0;
                                    
                            if ((new_smtp != null) && (!new_smtp.equals("")))
                            {
                                modValues[nbProp] = new PropertyValue(PropertyDef.USERS_SMTPADDRESS);
                                modValues[nbProp].stringValue(new_smtp);
                                nbProp++;
                            }
                            if ((password != null) && (!password.equals("")))
                            {
                                modValues[nbProp] = new PropertyValue(PropertyDef.USERS_PASSWORD);
                                modValues[nbProp].stringValue(password);
                                nbProp++;
                            }
                                    
                            try
                            {
                                users.setValues(id, modValues);
                                outputInformation(old_smtp + " has been modified");
                                modifiedCount++;
                            }
                            catch (XmediusException ex)
                            {
                                outputError("An error occured while modifying " 
                                             + old_smtp + " : " + ex.getMessage());
                            }
                        }
                        else
                        {
                            outputWarning("User not found. " + old_smtp + " has not been modified.");
                        }
                    }
                    else
                    {
                        outputWarning("User not specidiedd");
                    }
                }
                else
                {
                    done = true;
                }
            }
        }
        catch (java.io.IOException ex)
        {
            outputError("An error occured while reading the input file");
        }
        
        if (modifiedCount < 2)
        {
            outputInformation("<-- " + modifiedCount + " user has been modified");
        }
        else
        {
            outputInformation("<-- " + modifiedCount + " users have been modified");
        }
        
    }


    
    private void outputUsage()
    {
        outputInformation("Usages:");
        outputInformation("");

        outputInformation("List all users                 : XmUser {LoginInfo} -l");
        outputInformation("Get information on one user    : XmUser {LoginInfo} -i \"smtpAddress\"");
        outputInformation("Add one user                   : XmUser {LoginInfo} -a \"smtpAddress,profileName[,assocNtAccount[,webPassword[,fax]]]\"");
        outputInformation("Add one or more users          : XmUser {LoginInfo} -a -f \"fileName\"");
        outputInformation("Add or modify a user           : XmUser {LoginInfo} -am \"smtpAddress,profileName[,assocNtAccount[,webPassword[,fax]]]\"");
        outputInformation("Add or modify one or more users: XmUser {LoginInfo} -am -f \"fileName\"");
        outputInformation("Modify a user                  : XmUser {LoginInfo} -m \"smtpAddress,profileName[,assocNtAccount[,webPassword[,fax]]]\"");
        outputInformation("Modify a user/passowod         : XmUser {LoginInfo} -mp \"smtpAddress,[newSmtpAddress,[,webPassword]]\"");
        outputInformation("Modify one or more users       : XmUser {LoginInfo} -m -f \"fileName\"");
        outputInformation("Remove a user                  : XmUser {LoginInfo} -r \"smtpAddress\"");
        outputInformation("Remove one or more users       : XmUser {LoginInfo} -r -f \"fileName\"");
        outputInformation("Remove a user by fax number    : XmUser {LoginInfo} -rf \"fax\"");
        outputInformation("");

        String[] usages = GetLoginUsageSite();
        String prefix = "Where {LoginInfo} is: ";
        for (int i=0;i<usages.length;i++)
        {
            outputInformation(prefix + usages[i]);
            prefix = "                  or: ";
        }
    }


    //# NESTED CLASS #////////////////////////////////////////////////////////
        
    // Nested class ProfileList
    private class ProfileList
    {
        // Private Attributes
        private HashMap<String,String> m_nameToIdMap = null;  // key = UPPERCASE(profileName), value = profileId
        private HashMap<String,String> m_idToNameMap = null;  // key = profileId, value = profileName
        
        // Ctor
        public ProfileList()
        {
            m_nameToIdMap = null;
            m_idToNameMap = null;
        }        
        
        // load
        public boolean load(ListNode profiles)
        {
            if (profiles != null)
            {
                PropertyDef[] props = new PropertyDef[1];
                props[0] = PropertyDef.PROFILES_NAME;
        
                try
                {
                    View view = profiles.getView(props);
            
                    long ids[] = view.recordIds();
                    
                    HashMap<String,String> nameToIdMap = new HashMap<String,String>(ids.length);
                    HashMap<String,String> idToNameMap = new HashMap<String,String>(ids.length);
            
                    for (int i=0;i<ids.length;i++)
                    {
                        PropertyValue[] values = view.values(ids[i]);
                        
                        nameToIdMap.put(values[0].stringValue().toUpperCase(),
                                        Long.toString(ids[i]));
                        
                        idToNameMap.put(Long.toString(ids[i]),
                                        values[0].stringValue());
                    }
                    
                    m_nameToIdMap = nameToIdMap;
                    m_idToNameMap = idToNameMap;
                    return true;
                }
                catch (XmediusException ex)
                {}
            }
            return false;
        }
        
        // findId
        public String findId(String profileName)
        {
            if ( (profileName != null) && (m_nameToIdMap != null) )
            {
                return m_nameToIdMap.get(profileName.toUpperCase());
            }
            return null;
        }
        
        // findName
        public String findName(String profileId)
        {
            if ( (profileId != null) && (m_idToNameMap != null) )
            {
                return m_idToNameMap.get(profileId);
            }
            return null;
        }
        
    } //...class ProfileList
    
    // Nested class UserMap
    private class UserMap
    {
        // Private Attributes
        private HashMap<String,Long> m_map = null;  // key = UPPERCASE(smtpAddress), value = Long(id)
        
        // Ctor
        public UserMap()
        {
            m_map = null;
        }

        // load
        public boolean load(ListNode users)
        {
            if (users != null)
            {
                PropertyDef[] props = new PropertyDef[1];
                props[0] = PropertyDef.USERS_SMTPADDRESS;
                
                try
                {
                    View view = users.getView(props);
                
                    long[] ids = view.recordIds();
                    
                    HashMap<String,Long> map = new HashMap<String,Long>(ids.length);
                
                    for (int i=0;i<ids.length;i++)
                    {                    
                        PropertyValue[] values = view.values(ids[i]);
                        
                        String smtp = values[0].stringValue();
                        
                        if (smtp.equals(""))
                        {
                            users.deleteRecord(ids[i]);
                        }
                        else
                        {
                            map.put(smtp.toUpperCase(), new Long(ids[i]));
                        }
                    }
                    
                    m_map = map;
                    return true;
                }
                catch (XmediusException ex)
                {}
            }
            return false;
        }
        
        // contains
        public boolean contains(String userName)
        {
            if ( (userName != null) && (m_map != null) )
            {
                return m_map.containsKey(userName.toUpperCase());
            }
            return false;
        }
        
        // add
        public void add(String userName, long id)
        {
            if ( (userName != null) && (m_map != null) )
            {
                m_map.put(userName.toUpperCase(), new Long(id));
            }
        }
        
        // remove
        public void remove(String userName)
        {
            if ( (userName != null) && (m_map != null) )
            {
                m_map.remove(userName.toUpperCase());
            }
        }

        // find
        public long find(String userName)
        {
            if ( (userName != null) && (m_map != null) )
            {
                Long id = (Long) m_map.get(userName.toUpperCase());
                
                if (id != null)
                {
                    return id.longValue();
                }
            }
            return -1;
        }
        
    } //...class UserMap
}
//#///////////////////////////////////////////////////////////////END OF FILE//
