//#////////////////////////////////////////////////////////////////////////////
//# Copyright (c) 2002 Interstar technologies inc. Permission to use
//# this work for any purpose must be obtained in writing from INTERSTAR
//# TECHNOLOGIES, 5835 Verdun Ave, Suite 302, Montreal, Quebec, H4H 1M1 Canada
//#////////////////////////////////////////////////////////////////////////////

/**
\file XmBase.java
    
\since 2002/10/04

\author Silvain Laplante

*/

import java.io.*;
import java.nio.charset.*;
import java.util.StringTokenizer;
import java.util.Vector;

import com.interstarinc.xmedius.*;

public class XmBase
{
    //# Public Constants
    static public final String LOGIN_NAME      = "-LoginName";
    static public final String LOGIN_PWD       = "-LoginPwd";
    static public final String LOGIN_SERVERS   = "-LoginServers";
    static public final String LOGIN_SITE      = "-LoginSite";
    static public final String TARGET_SITE     = "-Site";


    /// PUBLIC INTERFACE /////////////////////////////////////////////////////

    //# Constructor
    public XmBase()
    {}

    public void execute(String workDescription)
    {}


    /// PROTECTED INTERFACE //////////////////////////////////////////////////

    //# outputError
    protected void outputError(String message)
    {
        outputMessage("ERROR  : ",  message);
    }

    //# outputWarning
    protected void outputWarning(String message)
    {
        outputMessage("WARNING: ",  message);
    }

    //# outputInformation
    protected void outputInformation(String message)
    {
        outputMessage("INFO   : ",  message);
    }

    //# getBufferedFileReader (return can be null)
    protected java.io.BufferedReader getBufferedFileReader(String fileName)
    {
        java.io.BufferedReader reader = null;
        
        try
        {
        	FileInputStream fis = new FileInputStream(fileName);
        	Charset cs = Charset.forName("UTF-8");
        	CharsetDecoder cd = cs.newDecoder();
        	InputStreamReader isr = new InputStreamReader(fis, cd);
            reader = new java.io.BufferedReader(isr);
        }
        catch (java.io.FileNotFoundException ex )
        {
            outputError("Failed to open " + fileName);
        }

        return reader;
    }

    //# getBufferedFileWriter (return can be null)
    protected java.io.BufferedWriter getBufferedFileWriter(String fileName)
    {
        java.io.BufferedWriter writer = null;
        try
        {
        	FileOutputStream fos = new FileOutputStream(fileName);
        	Charset cs = Charset.forName("UTF-8");
        	CharsetEncoder ce = cs.newEncoder();
        	OutputStreamWriter osr = new OutputStreamWriter(fos, ce);
            writer = new java.io.BufferedWriter(osr);
        }
        catch (java.io.FileNotFoundException ex)
        {
            outputError("Failed to open " + fileName);
        }        

        return writer;
    }

    //# getParsedCommandLine
    static protected ParsedCommandLine getParsedCommandLine(String[] args)
        throws com.interstarinc.xmedius.XmediusException
    {
        ArgVector argVec = new ArgVector();

        // Add hardcoded command line
        argVec.add(XmConstants.LOGIN_STRING);

        // Add runtime command line.  Runtime values will override hardcoded values
        argVec.add(args);

        return createParsedCommandLine(argVec.toStringArray());
    }
    
    protected boolean validateLoginInfo(ParsedCommandLine commandLine, boolean site)
    {
        if (commandLine == null) {
            return false;
        }

        boolean valid = true;
        boolean targetSiteRequired = site;

        if (commandLine.loginName == null)
        {
            valid = false;
            StringBuffer buf = new StringBuffer();
            buf.append("Missing login name. Use ");
            buf.append(LOGIN_NAME);
            buf.append(" \"your administrator name\" at the command line.");
            outputError(buf.toString());
        }

        if (commandLine.loginPassword == null)
        {
            valid = false;
            StringBuffer buf = new StringBuffer();
            buf.append("Missing login password. Use ");
            buf.append(LOGIN_PWD);
            buf.append(" \"your password\" at the command line.");
            outputError(buf.toString());
        }

        if (targetSiteRequired && (commandLine.targetSite == null))
        {
            valid = false;
            StringBuffer buf = new StringBuffer();
            buf.append("Missing target site. Use ");
            buf.append(TARGET_SITE);
            buf.append(" \"site name\" at the command line if login as a system administrator.");
            outputError(buf.toString());

            buf = new StringBuffer();
            buf.append("Missing target site. Use ");
            buf.append(LOGIN_SITE);
            buf.append(" \"site name\" at the command line if login as a site administrator.");
            outputError(buf.toString());
        }

        if ( (commandLine.loginServers == null) || (commandLine.loginServers.length == 0) )
        {
            valid = false;
            StringBuffer buf = new StringBuffer();
            buf.append("Missing login servers. Use ");
            buf.append(LOGIN_SERVERS);
            buf.append(" \"server1[,server2]\"");
            outputError(buf.toString());
        }

        return valid;
    }

    protected Session openSession(ParsedCommandLine commandLine)
    {
        if ( (commandLine == null) ||
             (commandLine.loginName == null) ||
             (commandLine.loginPassword == null) ||
             (commandLine.loginServers == null) ||
             (commandLine.loginServers.length == 0) )
        {
            return null;
        }

        try
        {
            boolean site = (commandLine.loginSite != null);

            if (site)
            {
                // Resolve guid if loginSite is a site name
                commandLine.targetSite = commandLine.loginSite
                    = getSiteGuidFromSiteName(commandLine.loginServers, commandLine.loginSite);

                return Session.open(commandLine.loginSite,
                                    commandLine.loginName,
                                    commandLine.loginPassword,
                                    commandLine.loginServers);
            }
            else
            {
                return Session.open(commandLine.loginName,
                                    commandLine.loginPassword,
                                    commandLine.loginServers);
            }
        }
        catch (XmediusException ex)
        {
            outputError(ex.getMessage());
        }
        return null;
    }

    static protected String[] GetLoginUsageSystem()
    {
        String[] usages = new String[1];

        StringBuffer buf = new StringBuffer();
        buf.append(LOGIN_NAME);
        buf.append(" \"system_admin_name\" ");
        buf.append(LOGIN_PWD);
        buf.append(" \"pwd\" ");
        buf.append(LOGIN_SERVERS);
        buf.append(" \"server1[,server2]\"");

        usages[0] = buf.toString();

        return usages;
    }

    static protected String[] GetLoginUsageSite()
    {
        String[] usages = new String[2];

        StringBuffer buf = new StringBuffer();
        buf.append(LOGIN_NAME);
        buf.append(" \"system_admin_name\" ");
        buf.append(LOGIN_PWD);
        buf.append(" \"pwd\" ");
        buf.append(TARGET_SITE);
        buf.append(" \"target_site_name\" ");
        buf.append(LOGIN_SERVERS);
        buf.append(" \"server1[,server2]\"");

        usages[0] = buf.toString();

        buf = new StringBuffer();
        buf.append(LOGIN_NAME);
        buf.append(" \"site_admin_name\" ");
        buf.append(LOGIN_PWD);
        buf.append(" \"pwd\" ");
        buf.append(LOGIN_SITE);
        buf.append(" \"site_guid\" ");
        buf.append(LOGIN_SERVERS);
        buf.append(" \"server1[,server2]\"");

        usages[1] = buf.toString();

        return usages;
    }

    /**
    Returns the site GUID associated to the given siteName or the
    given siteName itself if it is not found within the list
    */
    public String
    getSiteGuidFromSiteName(Session session, String siteName)
        throws com.interstarinc.xmedius.XmediusException
    {
        if (isAGuid(siteName)) {
            return siteName;
        }

        SiteResolver resolver = SiteResolver.create(session);
        return resolver.getIdentifier(siteName);
    }

    public String
    getSiteGuidFromSiteName(String[] servers, String siteName)
        throws com.interstarinc.xmedius.XmediusException
    {
        if (isAGuid(siteName)) {
            return siteName;
        }

        SiteResolver resolver = SiteResolver.create(servers);
        return resolver.getIdentifier(siteName);
    }


    /// PRIVATE INTERFACE ////////////////////////////////////////////////////

    //# outputMessage
    private void outputMessage(String level, String message)
    {
        System.out.println(level + message);
    }

    //# isAGuid, returns true if the given String represents a GUID
    static private boolean
    isAGuid(String guid)
    {
        boolean result = true;

        result = ( 38 == guid.length() );

        for(int i=0; result && i < guid.length(); i++)
        {
            char c = guid.charAt(i);

            switch(i)
            {
            case 0:
                result = ( '{' == c );
                break;
            case 9:
            case 14:
            case 19:
            case 24:
                result = ( '-' == c );
                break;
            case 37:
                result = ( '}' == c );
                break;
            default:
                result = ( '0' <= c && '9' >= c ) ||
                         ( 'A' <= c && 'F' >= c ) ||
                         ( 'a' <= c && 'f' >= c );
                break;
            }
        }
        return result;
    }

    //# createParsedCommandLine
    static private ParsedCommandLine createParsedCommandLine(String[] args)
        throws com.interstarinc.xmedius.XmediusException
    {
        java.util.Vector<String> servers = new java.util.Vector<String>();
        java.util.Vector<String> remainingArgs = new java.util.Vector<String>();

        ParsedCommandLine result = new ParsedCommandLine();

        ArgWalker walker = new ArgWalker(args);

        String arg = walker.next();

        while (arg != null)
        {
            if (arg.equalsIgnoreCase(LOGIN_NAME)) {
                result.loginName = walker.next();
            }
            else if (arg.equalsIgnoreCase(LOGIN_PWD)) {
                result.loginPassword = walker.next();
            }
            else if (arg.equalsIgnoreCase(LOGIN_SERVERS)) 
            {
                servers.clear();

                arg = walker.next();
                if (arg != null)
                {
                    String server;
                    StringBuffer buf = new StringBuffer("");

                    int i=0;
                    int imax = arg.length();
                    while (i<imax)
                    {
                        char c = arg.charAt(i);

                        switch (c)
                        {
                        case ' ':
                            break;
                        case ',':
                            server = buf.toString();
                            if (!server.equals("")) {
                                servers.add(server);
                                buf = new StringBuffer("");
                            }
                            break;
                        default:
                            buf.append(c);
                            break;
                        }

                        i++;
                    }
                    
                    server = buf.toString();
                    if (!server.equals("")) {
                        servers.add(server);
                    }
                }
            }
            else if (arg.equalsIgnoreCase(LOGIN_SITE)) {
                result.loginSite = walker.next();
            }
            else if (arg.equalsIgnoreCase(TARGET_SITE)) {
                result.targetSite = walker.next();
            }
            else {
                remainingArgs.add(arg);
            }

            arg = walker.next();
        }

        if (result.targetSite == null) {
            result.targetSite = result.loginSite;
        }

        if (!servers.isEmpty()) {
            result.loginServers = servers.toArray(new String[0]);
        }

        if (!remainingArgs.isEmpty()) {
            result.remainingArgs = remainingArgs.toArray(new String[0]);
        }
        else {
            result.remainingArgs = new String[0];
        }

        //outputCommandLine(result);
        return result;
    }

    //# outputCommandLine
    static private void outputCommandLine(ParsedCommandLine commandLine)
    {
        if (commandLine == null) {
            return;
        }

        String str = commandLine.loginName;
        System.out.println("loginName: " + ((str != null) ? str:"null"));

        str = commandLine.loginPassword;
        System.out.println("loginPassword: " + ((str != null) ? str:"null"));

        str = commandLine.loginSite;
        System.out.println("loginSite: " + ((str != null) ? str:"null"));

        str = commandLine.targetSite;
        System.out.println("targetSite: " + ((str != null) ? str:"null"));

        StringBuffer buf = new StringBuffer("servers: ");
        String[] strs = commandLine.loginServers;
        if (strs == null)
        {
            buf.append("null");
        }
        else
        {
            if (strs.length > 0)
            {
                buf.append(strs[0]);
            }

            for (int i=1;i<strs.length;i++)
            {
                buf.append(", ");
                buf.append(strs[i]);
            }
        }
        System.out.println(buf.toString());

        buf = new StringBuffer("remainingArgs: ");
        strs = commandLine.remainingArgs;
        if (strs == null)
        {
            buf.append("null");
        }
        else
        {
            if (strs.length > 0)
            {
                buf.append(strs[0]);
            }

            for (int i=1;i<strs.length;i++)
            {
                buf.append(", ");
                buf.append(strs[i]);
            }
        }
        System.out.println(buf.toString());
    }


    /// Nested class ParsedCommandLine
    static public class ParsedCommandLine
    {
        public String loginName = null;
        public String loginPassword = null;
        public String loginSite = null;
        public String[] loginServers = null;

        public String targetSite = null;

        public String[] remainingArgs = null;

        public ParsedCommandLine()
        {}
    }

    /// Nested class ArgVector
    static private class ArgVector
    {
        private java.util.Vector<String> m_args;

        /// PUBLIC INTERFACE //////////////////////////////////////////////////
        public ArgVector()
        {
            m_args = new java.util.Vector<String>();
        }

        public void add(String[] args)
        {
            if (args == null) {
                return;
            }

            for (int i=0;i<args.length;i++) {
                m_args.add(args[i]);
            }
        }

        public void add(String commandLine)
        {
            if (commandLine == null) {
                return;
            }

            StringBuffer buf = new StringBuffer("");

            boolean inarg = false;
            boolean inquote = false;
            boolean cutnow = false;

            int i=0;
            int imax = commandLine.length();
            while (i<imax)
            {
                char c = commandLine.charAt(i);

                switch (c)
                {
                case ' ':
                    if (inquote) {
                        buf.append(c);
                    }
                    else if (inarg) {
                        cutnow = true;
                    }
                    break;

                case '"':
                    if (inquote) {
                        cutnow = true;
                    }
                    else if (inarg){
                        cutnow = true;
                    }
                    else {
                        inquote = true;
                    }
                    break;

                default:
                    inarg = true;
                    buf.append(c);
                    break;
                }

                if (cutnow)
                {
                    m_args.add(buf.toString());
                    buf = new StringBuffer("");

                    inarg = false;
                    inquote = false;
                    cutnow = false;
                }

                i++;
            }

            if (inarg) {
                m_args.add(buf.toString());
            }
        }

        public String[] toStringArray()
        {
            return m_args.toArray(new String[0]);
        }
    }

    /// Nested class ArgWalker
    static private class ArgWalker
    {
        private String[] m_args = null;
        private int m_index = -1;

        /// PUBLIC INTERFACE //////////////////////////////////////////////////
        public ArgWalker(String[] Args)
        {
            if (Args != null)
            {
                m_args = new String[Args.length];

                for (int i=0;i<Args.length;i++) {
                    m_args[i] = Args[i];
                }
            }
            else {
                m_args = new String[0];
            }
        }

        public String next()
        {
            m_index++;

            while (m_index < m_args.length)
            {
                String next = m_args[m_index];

                if (next != null) {
                    return next;
                }

                m_index++;
            }

            return null;
        }
    }

    /// Nested class Tokenizer
    protected static class Tokenizer
    {
        private Vector<String> m_tokens;
        private int m_next;

        public Tokenizer(String ToTokenize)
        {
            Vector<String> temp = new Vector<String>();
            if (ToTokenize.equals(""))
            {
                temp.add("");
            }
            else
            {
                StringTokenizer tok = new StringTokenizer(ToTokenize, ",", true);

                while (tok.hasMoreTokens())
                {
                    temp.add(tok.nextToken());
                }
            }

            m_tokens = getTokens(temp);
            m_next = 0;
        }

        public String getNextToken(String defaultToken)
        {
            if (m_tokens.size() > m_next)
            {
                String token = (String) m_tokens.elementAt(m_next);
                m_next++;
                return token;
            }

            return defaultToken;
        }

        private Vector<String> getTokens(Vector<String> rawTokens)
        {
            Vector<String> tokens = new Vector<String>();

            int size = rawTokens.size();
            for (int i=0;i<size;i++)
            {
                String temp = (String) rawTokens.elementAt(i);
                if (temp.equals(","))
                {
                    if (0 == i)
                    {
                        tokens.add("");
                    }
                    else
                    {
                        String before = (String) rawTokens.elementAt(i-1);
                        if (before.equals(","))
                        {
                            tokens.add("");
                        }
                    }

                    if ((size-1) == i)
                    {
                        tokens.add("");
                    }
                }
                else
                {
                    tokens.add(temp);
                }
            }

            return tokens;
        }
    }
}
//#///////////////////////////////////////////////////////////////END OF FILE//