<?php
require_once 'iplan/web/Form.php';
require_once 'iplan/security/ApplicationContext.php';
require_once 'iplan/orm/ORM.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
class FormCondition {
  /**
   * @var Form el formulario que se debe mostrar en caso que se cumpla la condici�n
   */
  private $form;

  /**
   * @var boolean indica si el formulario es un formulario final o no.
   */
  private $final;

  /**
   * @var array un arreglo donde se almacenan las condiciones a verificar para mostrar el formulario.
   */
  private $tests;

  /**
   * @var boolean un valor booleano que indica si este formulario debe ser comiteado con �xito antes del siguiente paso.
   */
  private $needSave;

  /**
   * @var string el nombre de la regla
   */
  private $ruleName;

  /**
   * @var Closure la funci�n de inicializaci�n del formulario
   */
  private $initFunction;

  /**
   * @var string la clase del formulario a instanciar
   */
  private $class;

  /**
   * Crea un objeto que es capaz de decidir si debe o no mostrar un formulario en base a un conjunto de condiciones.
   * 
   * @param string $ruleName el nombre de la regla
   * @param string $class la clase del formulario que se debe exhibir cuando se cumplen las condiciones.
   * @param boolean $final un valor booleano que indica si se trata de un formulario final o no.
   * @param boolean $needSave indica si este formulario requiere realizar una operaci�n de "save" exitosa antes de continuar.
   * @param array $tests un arreglo donde cada �tem tiene la forma ('componente'=>'valor') y se emplea para determinar si se debe o no mostrar el formulario.
   * @param Closure $initFunction la funci�n de inicializaci�n
   * 
   * @return ComposedForm Una instancia de ComposedForm.
   */
  public function __construct($ruleName, $class, $final, $needSave, $tests, $initFunction = null)
  {
    // Bouml preserved body begin 00123385
	$this->ruleName = $ruleName;
	$this->final=$final;
	$this->tests=$tests;
	$this->class=$class;
	$this->needSave = $needSave;
	$this->initFunction = $initFunction;
	return $this;
    // Bouml preserved body end 00123385
  }

  /**
   * Indica si la condici�n es para un formulario final.
   * 
   * @return boolean Devuelve TRUE si el formulario es el definitivo y FALSE si se trata de un formulario para otra instancia de selecci�n.
   */
  public function isFinal()
  {
    // Bouml preserved body begin 00124F05
	return $this->final;
    // Bouml preserved body end 00124F05
  }

  /**
   * Devuelve el Form de la condici�n. El primer llamado que se hace crea el form, los sucesivos pueden omitir dicho par�metro.
   * 
   * @param ApplicationContext $context el contexto en el cual se instanciar� el formulario.
   * @param ORM $orm el manejador de objetos a emplear para la inicializaci�n.
   * 
   * @return Form El formulario de la condici�n.
   */
  public function getForm(&$context = null, &$orm = null)
  {
    // Bouml preserved body begin 00124F85
	if (is_null($this->form)) {
		$class = $this->class;
		if (is_subclass_of($this->class, 'ABMForm'))
			$this->form = new $class($orm, $context);
		else
			$this->form = new $class();
		$this->form->setPrefix($this->ruleName);
	} else {
		if (!is_null($context))
			$this->form->setContext($context);
		if (is_subclass_of($this->class, 'ABMForm') && !is_null($orm))
			$this->form->setORM($orm);
	}
	return $this->form;
    // Bouml preserved body end 00124F85
  }

  /**
   * Evalua si el formulario debe o no ser mostrado.
   * 
   * @param ApplicationContext $context el contexto en el que se evaluan los tests.
   * @param array $status un arreglo que refleja el estado de los otros formularios.
   * @param ORM $orm el manejador de objetos. Es requerido si se trata de ABMForms.
   * 
   * @return boolean Un TRUE si el formulario debe ser mostrado y un FALSE sino.
   */
  public function test(&$context, &$status, &$orm = null)
  {
    // Bouml preserved body begin 00123405
	if (is_null($this->form)) $this->getForm($context, $orm);
	$this->form->setContext($context);
	$pass = true;
	if (is_array($this->tests) && (count($this->tests)>0)) {
		foreach ($this->tests as $name=>$referenceValue) {
			if (is_string($name) && !is_numeric($name)) {
				if ($this->form->getMethod()=='POST') {
					$value = $context->getParam($name, null, 'PG');
				} else {
					$value = $context->getParam($name, null, 'GP');
				}
			} else {
				if (is_numeric($name) && strpos($referenceValue, '.')) {
					$value = $status[$referenceValue];
					$referenceValue=true;//Tiene que ser verdadero
				} else {
					throw new Exception("FormCondition::test() => No se puede evaluar $referenceValue");
				}
			}
			if (is_null($value)) $value = $this->form->getValueOf($name);
			$pass = $pass && ($value == $referenceValue);
		}
	}
	return $pass;
    // Bouml preserved body end 00123405
  }

  /**
   * Devuelve la configuraci�n de usuario de invocar al m�todo save.
   * 
   * @return boolean Un TRUE si el form require una operaci�n de guardado exitosa antes de continuar, FALSE sino...
   */
  public function needSave()
  {
    // Bouml preserved body begin 0012DA85
	return $this->needSave;
    // Bouml preserved body end 0012DA85
  }

  /**
   * Retorna el nombre de regla configurado.
   * 
   * @return string El nombre de la regla que representa el objeto.
   */
  final public function getRuleName()
  {
    return $this->ruleName;
  }

  /**
   * @return Closure la funci�n de inicializaci�n del form
   */
  final public function getInitFunction()
  {
    return $this->initFunction;
  }

  /**
   * Resetea la condici�n como para ser evaluada de cero.
   * @return FormCondition
   */
  public function clear()
  {
    // Bouml preserved body begin 00135305
	$this->form = null;
	return $this;
    // Bouml preserved body end 00135305
  }

  /**
   * Inicializa el formulario en base al resultado del formulario previo
   * @param ApplicationContext $context el contexto en el que se desarrolla la acci�n
   * @param Form $previusForm el formulario de la regla que se evalu� previamente en forma exitosa o el default.
   * @return boolean Devuelve TRUE si todo salio bien y FALSE sino.
   */
  public function init(&$context, &$previusForm)
  {
    // Bouml preserved body begin 00135385
	if (is_null($this->form)) throw new Exception('No se puede inicializar un form que no existe');
	$func = $this->initFunction; 
	return $func($context, $this->form, $previusForm);
    // Bouml preserved body end 00135385
  }

}
?>