<?php
require_once 'iplan/security/ApplicationContext.php';
require_once 'iplan/web/Component.php';
require_once 'iplan/web/HiddenComponent.php';
require_once 'iplan/orm/ORMObject.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
class Form {
  /**
   * Constate que representa un formulario est�ndar sin ning�n agregado autom�tico de nada. Es el predeterminado.
   */
  const FORM_CUSTOM = "FORM_CUSTOM";

  /**
   * Constate que representa un formulario est�ndar que s�lo muestra un bot�n para volver atr�s.
   */
  const FORM_BACK = "FORM_BACK";

  /**
   * Constate que representa un formulario con los botones "Aceptar" y "Cancelar"
   */
  const FORM_OK_CANCEL = "FORM_OK_CANCEL";

  /**
   * Constate que representa un formulario con los botones "Sí" y "No"
   */
  const FORM_YES_NO = "FORM_YES_NO";

  /**
   * Constate que representa un formulario con los botones "Guardar" y "Cancelar"
   */
  const FORM_SAVE_BACK = "FORM_SAVE_BACK";

  /**
   * @var string el identificador del formulario
   */
  protected $id;

  /**
   * @var string el valor del atributo <form action="xxxx"
   */
  protected $action;

  /**
   * @var string el valor del atributo <form method="xxx"
   */
  protected $method;

  /**
   * @var string el t�tulo del formulario
   */
  protected $title;

  /**
   * @var string el tipo de codificaci�n a utilizar, puede adoptar 3 valores "application/x-www-form-urlencoded" (default), "multipart/form-data" y "text/plain"
   */
  protected $enctype;

  /**
   * @var ApplicationContext el contexto en el cual se ejecuta el Form. Esta variable solo adquiere valor al llamar a deploy().
   */
  protected $context;

  /**
   * @var string de existir indica cu�l es el div en el cual se debe volcar el resultado del formulario una vez enviado
   */
  protected $target;

  /**
   * @var string el template que se renderiza para mostrar el form
   */
  protected $formTemplate;

  /**
   * @var string el template predeterminado sobre el cual se muestran mensajes de �xito o error cuando no hay un formulario involucrado
   */
  protected $successTemplate;

  /**
   * @var boolean determina si el tag <form> debe o no escribirse. En forma predeterminada es TRUE, o sea, lo escribe
   */
  protected $openTag;

  /**
   * @var boolean determina si el tag </form> debe o no escribirse. En forma predeterminada es TRUE hace que se escriba
   */
  protected $closeTag;

  /**
   * @var string el prefijo usado para la generaci�n de nombres
   */
  protected $prefix;

  /**
   * @var string el sufijo usado para la generaci�n de nombres
   */
  protected $suffix;

  /**
   * @var string Un valor que indica si el formulario emplea alg�n set de botones est�ndar.
   */
  private $formType;

  /**
   * @var array la lista de componentes que han sido actualizados en respuesta a un evento. (Ver deploy())
   */
  private $updateds;

  /**
   * @var string el nombre de la función que se llamará en caso de que AJAX esté disponible. La firma de la función es function (form, validations, target)
   */
  private $ajaxAction = 'uws_submit';

  /**
   * @var array la lista de items del formulario
   */
  private $items;

  /**
   * Crea una instancia de Form.
   * 
   * @return Form
   */
  public function __construct()
  {
    // Bouml preserved body begin 00112B85
	$this->formTemplate = 'forms.xhtml';
	$this->successTemplate = 'lib/success.xhtml';
	$this->openTag = true;
	$this->closeTag = true;
	$this->enctype = "application/x-www-form-urlencoded";
	$this->target="desktop_center_content";
	$this->prefix="";
	$this->suffix="";
	$this->method="post";
	$this->formType = Form::FORM_CUSTOM;
	
	return $this;
    // Bouml preserved body end 00112B85
  }

  /**
   * Configura el t�tulo del formulario.
   * 
   * @param string $title el t�tulo que figurar� como encabezado del formulario
   * 
   * @return Form el formulario con el t�tulo configurado
   */
  public function setTitle($title)
  {
    // Bouml preserved body begin 000E8D85
    $this->title = $title;
    return $this;
    // Bouml preserved body end 000E8D85
  }

  /**
   * Recupera el valor de un componente del formulario.
   * 
   * @param string $name el nombre del componente. Es importante recordar que en los casos que no se especifica el name, se usa el valor del "maps".
   * 
   * @return mixed El valor actual del componente indicado.
   */
  public function getValueOf($name)
  {
    // Bouml preserved body begin 000F9385
	$name2 = $this->prefix.$name.$this->suffix;
	
	//Se hace necesario identificar si el componente es una fecha para compatibilizar formatos.
	$comp = $this->getComponent($name);
	$isDate = is_a($comp, 'DateComponent');
	$value = $defaultValue = null;
	if (is_null($this->context)) throw new Exception(" Error al invocar a Form::getValueOf(). No se puede acceder al valor del componente \"$name\" porque no hay un contexto configurado");
	
	
	if (!is_null($comp) && is_a($comp, "Component")) {
		switch (get_class($comp)) {
			case "CheckboxComponent":
				if ($comp->getChecked()) {
					$defaultValue = $comp->getValue();
				}
				break;
			default:
				$defaultValue = $comp->getValue();
				break;
		}
		
	}
    switch (strtoupper($this->method)) {
        case 'POST':
				$value = $this->context->getParam($name2, null, 'PG');
				if (is_null($value)) $value = $this->context->getParam($name, $defaultValue, 'PFG');
            break;
        case 'GET':
				$value = $this->context->getParam($name2, null, 'GP');
				if (is_null($value)) $value = $this->context->getParam($name, $defaultValue, 'GFP');
            break;
        default: throw new Exception ("class Form: Debe tener definido el method (POST/GET).");
    }
	/*if (is_a($comp, "MultiselectComponent")) {
		echo "[" . $comp->getName() . "]";
	}*/
	
	if (!is_null($value) && is_array($value)) {
		$value = array_diff($value, array(""));
		if (count($value) == 0)
			$value  = null;
	} elseif (!is_null($value) && $isDate) {
		$value = DateTime::createFromFormat($comp->getFormat(), $value);
	}
	return $value;
    // Bouml preserved body end 000F9385
  }

  /**
   * Establece el valor predeterminado de un componente del formulario.
   * 
   * @param string $name el nombre del componente que queremos configurar.
   * @param mixed $value el valor a asignarle
   * 
   * @return retorna el Form con el componente configurado.
   */
  public function setValueOf($name, $value)
  {
    // Bouml preserved body begin 000FCA85
    $comp = $this->getComponent($name);
	if ($comp !== null) {
		switch(true) {
			case is_a($comp, 'SelectComponent'):
			case is_a($comp, 'RadioComponent'):
					if (is_bool($value)) {
						$val = ($value)?1:0;
						$comp->setSelected($val);
					} else {
						$comp->setSelected($value);
					}
				break;
			case is_a($comp, 'CheckboxComponent'):
					$comp->setChecked($value==$comp->getValue());
				break;
			default:
				$comp->setValue($value);
		}
	}
    return $this;
    // Bouml preserved body end 000FCA85
  }

  public function setMethod($value)
  {
    $this->method = $value;
    return $this;
  }

  /**
   * Recupera el m�todo por el que se hace el submit del formulario.
   * 
   * @return string El m�todo, t�picamente GET, POST o null si no est� configurado.
   */
  final public function getMethod()
  {
    return $this->method;
  }

  public function setAction($value)
  {
    $this->action = $value;
    return $this;
  }

  public function setId($value)
  {
    $this->id = $value;
    return $this;
  }

  public function setEnctype($value)
  {
    $this->enctype = $value;
    return $this;
  }

  /**
   * Agrega un elemento al formulario.
   * 
   * @param Component $component el elemento a agregar
   * 
   * @return Form el formulario al que se le agreg� el elemento.
   */
  public function addComponent($component)
  {
    // Bouml preserved body begin 000E6D85
    $this->items[]=$component;
    return $this;
    // Bouml preserved body end 000E6D85
  }

  /**
   * Devuelve el componente indicado.
   * 
   * @param string $name el nombre del componente a recuperar. El name puede es el indicado por el usuario o el maps en los casos que no se configur� el name.
   * 
   * @return Component|array El componente solicitado o NULL sino existe. En el caso que se llame sin par�metros retorna un arreglo con todos los componentes del formulario en una sola lista.
   */
  public function &getComponent($name = null)
  {
    // Bouml preserved body begin 000F9405
	$ret = null;
	if ($name !== null) {
		if (is_array($this->items)) {
			foreach($this->items as $item) {
				if (($item->getName()==$name) || ($item->getMaps()==$name))
						return $item;
				elseif (is_a($item, 'Layout')) {
					$result = $item->getComponent($name);
					if (!is_null($result)) return $result;
				}
			}
		}
	} else {
		$aComponents = array();
		if (is_array($this->items)) {
			foreach($this->items as $item) {
				if (!is_a($item, 'Layout')) {
					$aComponents[] = $item;
				} else {
					$arr = $item->getComponent();
					if (is_array($arr)) {
						$aComponents = array_merge($aComponents, $arr);
					}
				}
			}
		}
		return ($aComponents);
	}
    return $ret;
    // Bouml preserved body end 000F9405
  }

  /**
   * Quita un componente del formulario.
   * @param string name el nombre del componente a quitar.
   * 
   * @result Form El formulario sin el componente.
   */
  public function removeComponent($name)
  {
    // Bouml preserved body begin 0012E485
	if (is_array($this->items) && (count($this->items)>0)) {
		for($i=0;$i<count($this->items);$i++) {
			if (is_a($this->items[$i], 'Layout')) {
				$this->items[$i]->removeComponent($name);
			} elseif($this->items[$i]->getName()==$name) {
				unset ($this->items[$i]);
				break;
			}
		}
	}
	return $this;
    // Bouml preserved body end 0012E485
  }

  final public function getFormTemplate()
  {
    return $this->formTemplate;
  }

  public function setFormTemplate($value)
  {
    $this->formTemplate = $value;
    return $this;
  }

  final public function getSuccessTemplate()
  {
    return $this->successTemplate;
  }

  public function setSuccessTemplate($value)
  {
    $this->successTemplate = $value;
  }

  /**
   * Recupera el target del Form.
   * 
   * @return string el target que se refrescar� al hacer el submit del form.
   */
  public function getTarget()
  {
    // Bouml preserved body begin 00109E05
    return $this->target;
    // Bouml preserved body end 00109E05
  }

  /**
   * Configura el al form para que refresque la capa (div) con el nombre indicado.
   * 
   * @param string $target el nombre del div que ser� refrescado con el fruto del submit del formulario.
   * 
   * @return Form el formulario configurado
   */
  public function setTarget($target)
  {
    // Bouml preserved body begin 00109E85
    $this->target = $target;
    return $this;
    // Bouml preserved body end 00109E85
  }

  final public function getContext()
  {
    return $this->context;
  }

  public function setContext($value)
  {
    $this->context = $value;
    return $this;
  }

  /**
   * Recupera el valor del flag de openTag
   * 
   * @return boolean retorna un TRUE si el form va a escribir el tag <form> y FALSE sino.
   */
  final public function getOpenTag()
  {
    return $this->openTag;
  }

  /**
   * Configura el flag de escritura del tag <form>.
   * 
   * @param boolean $value un TRUE para que escriba el tag (valor predeterminado) y un FALSE para que no lo escriba.
   * @return Form el formulario modificado.
   */
  public function setOpenTag($value)
  {
    $this->openTag = $value;
    return $this;
  }

  /**
   * Recupera el valor del flag de openTag
   * 
   * @return boolean retorna un TRUE si el form va escribir el </form> y FALSE sino.
   */
  final public function getCloseTag()
  {
    return $this->closeTag;
  }

  /**
   * Configura el flag de escritura del tag </form>.
   * 
   * @param boolean $value un TRUE para que escriba el tag (valor predeterminado) y un FALSE para que no lo escriba.
   * @return Form el formulario modificado.
   */
  public function setCloseTag($value)
  {
    $this->closeTag = $value;
    return $this;
  }

  /**
   * Recupera el prefijo configurado.
   * 
   * @return string Una cadena de texto con el prefijo configurado
   */
  final public function getPrefix()
  {
    return $this->prefix;
  }

  /**
   * Configura el prefijo de la generaci�n de nombres.
   * 
   * @param string $value una cadena de texto con el prefijo a anteponer a los nombres de componentes.
   * 
   * @return Form El formulario que ahora tiene el valor del prefijo configurado.
   */
  public function setPrefix($value)
  {
    $this->prefix = $value;
    return $this;
  }

  /**
   * Recupera el sufijo configurado.
   * 
   * @return string Una cadena de texto con el sufijo configurado
   */
  final public function getSuffix()
  {
    return $this->suffix;
  }

  /**
   * Configura el sufijo de la generaci�n de nombres.
   * 
   * @param string $value una cadena de texto con el sufijo a anteponer a los nombres de componentes.
   * 
   * @return Form El formulario que ahora tiene el valor del sufijo configurado.
   */
  public function setSuffix($value)
  {
    $this->suffix = $value;
    return $this;
  }

  /**
   * Realiza una verificaci�n sobre los par�metros de POST y GET (depende qu� se consigne en el Form) seg�n las definiciones de campos que se hayan realizado.
   * 
   * @return TRUE|array Devuelve un valor TRUE si todas las definiciones de componentes tienen una contraparte v�lida y acorde a las validaciones impuestas. En caso que algo falle retorna una matr�z con el siguiente formato:
   * 
   * array (
   * 	0 = > array (
   * 		Component, array("Error 1", "Error 2")
   * 		)
   * 	...
   * )
   */
  public function validate()
  {
    // Bouml preserved body begin 000FCA05
	$comps = $this->getComponent();
	$aRes = array();
	for ($i = 0; $i < count($comps); $i++) {
            $mps  = $comps[$i]->getMaps();
			$name = $comps[$i]->getName();
			if ($name == null) $name = $mps;
			if ($mps == null) $mps = $name;
            $vals = $comps[$i]->getValidate();
            //Si tiene validaciones...
            if (!is_null($vals)) {
                if (is_string($vals)) $vals = array($vals);
                //...las recorro, ejecuto y almaceno el resultado
                foreach($vals as $val) {
                    $valData = explode(" ", $val);
                    if (Validation::isComplexValidation($valData[0])) {
                        $oValida = Validation::getValidation($valData[0], $valData[1]);
                    } else {
                        $oValida = Validation::getValidation($valData[0]);
                    }
                    $res=$oValida->validate($this->getValueOf($name));
                    if ($res !== true)
                        $aRes[$mps][]=$res;
                }
            }
	}
	if (count($aRes) > 0)
		return $aRes;
	else
        return true;
    // Bouml preserved body end 000FCA05
  }

  /**
   * Carga en los componentes los valores que encuentren correspondencia con alguno de los par�metros pasados recibidos del usuario.
   * 
   * @return boolean Retorna TRUE si todo anduvo bien. FALSE de otra manera.
   */
  public function loadValues()
  {
    // Bouml preserved body begin 00106885
	$comps = $this->getComponent();
	for ($i = 0; $i < count($comps); $i++) {
		//setSelected
		$name = $comps[$i]->getName();
		switch(get_class($comps[$i])) {
			case 'SelectComponent':
			case 'RadioComponent':
				//if (is_a($comps[$i], "MultiselectComponent")) {
				//	$comps[$i]->setSelected($this->getValueOf($comps[$i]->getName()));
				//} elseif ($comps[$i]->getSelected() == "")
				$comps[$i]->setSelected($this->getValueOf($name));
				break;
			case 'ButtonComponent':
			case 'LabelComponent':
				//if ($comps[$i]->getValue() == "")
				//$comps[$i]->setValue($this->getValueOf($comps[$i]->getName()));
				break;
			case 'CheckboxComponent':
				$name2 = $this->prefix.$name.$this->suffix;
				$val = $this->context->getParam($name2);
				if ($val === null) {
					$val = $this->context->getParam($name);
				}
				$comps[$i]->setChecked($val != null);
				break;
			default:
				$comps[$i]->setValue($this->getValueOf($name));
		}
	}
    // Bouml preserved body end 00106885
  }

  /**
   * Retorna un arreglo la definici�n del objeto a fin de que sea f�cilmente interpretable por un Template de TWIG.
   * Debido a que no todos los componentes tienen el maps y el name obligatorio, se asume como regla que:
   * <ul>
   * 	<li><b>si tiene name</b>, se usa el name.</li>
   * 	<li><b>si no tiene name</b>, se usa el maps de la siguiente manera: $prefix.maps.$postfix</li>
   * </ul>
   * 
   * @param string $prefix el prefijo que emplea para generar los nombres cuando no existe el name.
   * @param string $suffix el sufijo que emplea para generar los nombres cuando no existe el name.
   * 
   * @return array El arreglo que representa el objeto y sus propiedades
   */
  public function toArray()
  {
    // Bouml preserved body begin 000E8E85
	
	//Para saber donde volver agrego un componente con la acción
	if (isset($this->context)) {
		if ($this->context->getParam('UWS_BACK_ACTION') !== null) {
			$this->addComponent(new HiddenComponent(null, 'UWS_BACK_ACTION', null, $this->context->getParam('UWS_BACK_ACTION')));
			$this->addComponent(new HiddenComponent(null, 'UWS_LIST_FILTER', null, $this->context->getParam('UWS_LIST_FILTER')));
		}
	}
	//TODO: acá habia un $this->loadValues(); pero no hay contexto a este punto.
	  
	$items = array();
    foreach($this->items as $item) {
        $items[] = $item->toArray($this->prefix, $this->suffix);
    }
	//krumo($items);
	//$addComponents ES UNA FUNCIÓN definida localmente como anónima...
	$addComponents = function ($list, $f, &$showForm) {
        $js_text = "";
        if (is_array($list)) {
            foreach($list as $item) {
                //Si es un layout recurso
                if ((isset($item['type'])) && ($item['type'] == 'layout')) {
                    if ($item['name']=='GridLayout') {
                        foreach($item['items'] as $row)
                            foreach($row as $element) {
								if ((isset($element['type'])) && ($element['type'] != 'hidden'))
									$showForm = true;
                                $js_text .= $f($element, $f, $showForm);
							}
                    } else {
                        $js_text .= $f($item['items'], $f, $showForm);
                    }
                //Si es un componente
                } else {
					if ((isset($item['type'])) && ($item['type'] != 'hidden')) {
						$showForm = true;
					}
					if ((isset($item['validate'])) && (is_array($item['validate']))) {
						$first=true;
						$close=false;
						foreach($item['validate'] as $validation=>$expression) {
							if (Validation::isComplexValidation($validation)) {
								if ($first) {
									$js_text .= "'".$item['name']."': {";
									$first=false;
									$close = true;
								}
							}
							$js_text .= "'$validation': '$expression',";
						}
						if ($close) $js_text = substr ($js_text, 0, strlen ($js_text)-1). "},";
					}
                }
            }
        }
        return $js_text;
    };
	//...y se usa ACÁ.
	$showForm = false;
    $js_text = $addComponents($items, $addComponents, $showForm);
    $js_text = '{' . substr ($js_text, 0, strlen ($js_text)-1) . '}';

	$result = array(
	'title'  => $this->title,
	'items'  => $items,
	'method' => $this->method,
	'action' => $this->action,
	'id'     => $this->id,
	'enctype'=> $this->enctype,
	'target' => $this->target,
	'formTemplate'    => $this->formTemplate,
	'messageTemplate' => $this->successTemplate,
	'js_validations'  => $js_text,
    'openTag'=> $this->openTag,
	'closeTag'=> $this->closeTag,
	'formType'=> $this->formType,
	'showForm' => $showForm,
	'ajaxFunction'=>$this->ajaxAction
  	);
    return $result;
    // Bouml preserved body end 000E8E85
  }

  /**
   * Configura los valores predeterminados del form con los atributos correspondientes del objeto pasado por par�metro, usando para ello las definiciones de los "maps" de cada componente.
   * El m�todo asume que todo est� bien definido. No contempla relaciones.
   * 
   * @param Object $object el objeto sobre el cual se deben basar los campos del formulario.
   * 
   * @return boolean En caso que todo salga bien devuelve TRUE y si sucedio un error FALSE.
   */
  public function getValuesFromObject(&$object)
  {
    // Bouml preserved body begin 0012E085
	$comps = $this->getComponent();
    for ($i = 0; $i < count($comps); $i++) {
		$mps = $comps[$i]->getMaps();
		if ($mps) {
			//Esta mapeado
			$name = $comps[$i]->getName();
			if ($name === null) $name = $mps;
			
			$attributes = explode('.', $mps);
			$tmpObj = $object;
			foreach($attributes as $attribute) {
				$strAtr= "get" . ucfirst($attribute);
				$value = $tmpObj->$strAtr();
				if (!is_null($value)) {
					$tmpObj=$value;
				} else break;
			}
			if (!is_null($value)) {
				if (!is_a($value, 'ORMCollection')) {
					$this->setValueOf($name, $value);
				} else {
					//Es una relacion
					$aIdcitos = array();
					if ((count($value)> 0)) {
						foreach ($value as $itemcin) {
							$aIdcitos[] = $itemcin->getId();
						}
					}
					$this->setValueOf($name, $aIdcitos);
				}
			}	
			
		}
    }
    // Bouml preserved body end 0012E085
  }

  /**
   * Convierte la actual representaci�n del formulario a un su representaci�n en URL de los par�metros. Se adjunta el par�metro $includeAction A FIN DE QUE ESTE CONTEMPLADO PARA SU FUTURA IMPLEMENTACI�N.
   * 
   * @param boolean $includeAction indica si se desea sacar una URL completa o s�lo la parte de los par�metros con valor.
   * @param string $prefix el prefijo de los nombres generados.
   * @param string $suffix el sufijo de los nombres generados.
   * @return string La URL o los par�metros (seg�n $onlyParams) en formato string
   */
  public function toURL($includeAction = false, $prefix = '', $suffix = '')
  {
    // Bouml preserved body begin 00131B05
	$str='';
	if ($includeAction) {
		throw new Exception('Form::toURL($includeAction=true) => no está implementada la generación de la acción en éste método<br/>');
	}
	if (is_array($this->items) && (count($this->items)>0)) {
		foreach($this->items as $comp) {
			$tmpStr = $comp->toURL($prefix, $suffix);
			if ($tmpStr != '')
				$str .= $tmpStr."&";
		}
		if ($str != '')
			return substr($str, 0, strlen($str)-1);
		else return '';
	} else {
		return $str;
	}
    // Bouml preserved body end 00131B05
  }

  /**
   * Solicita al formulario que realice la tarea para la que est� configurado.
   * 
   * @param ApplicationContext $context el contexto en el cual el Form debe hacer su trabajo. Puede omitirse si anteriormente se llam� a Form::setContext()
   * @param ORM $orm el ORM a usar
   * 
   * @return Renderable la pantalla que debe mostrar.
   */
  public function deploy(&$context = null)
  {
    // Bouml preserved body begin 0013BE85
	if (is_null($context)) {
		if (is_null($this->context))
				throw new Exception("No se ha asignado un contexto para desplegar el formulario.");
		else $context = $this->context;
	}
	
	$eventName = $this->getValueOf('UWS_EVENT');
	if (!is_null($eventName)) {
		$this->updateds=array();
		$target = $this->getValueOf('UWS_EVENT_TARGET');
		if (!is_null($target)) {
			$approved = $this->getValueOf('UWS_EVENT_VERIFIED');
			$approved = explode(',', $approved);
			if (count($approved) > 0) {
				$events=$this->getComponent($target)->getEvents($eventName);
				foreach($approved as $key) {
					$function = $events[$key]['function'];
					$params = $events[$key]['params'];
					if (is_string($function)) {
						if (is_callable(array($this,$function))) {
							$this->$function($context, $this, $params);
						} else {
							throw new Exception("Error al procesar el evento \"$eventName\" de $target, la función ".  get_class($this)."->$function() no existe");
						}
					} elseif(is_callable($function) || is_a($function, Closure)) {
						$function($context, $this, $params);
					} else {
						throw new Exception("Se intentó invocar una función inválida al procesar el evento \"$eventName\" de $target");
					}
				}
			}
		} else {
			throw new Exception("Se ha producido un evento \"$eventName\", sin un target");
		}
		$comps=array();
		foreach($this->updateds as $component=>$scrap)
			$comps[]=$this->getComponent($component)->toArray($this->prefix, $this->suffix);
	
		$options = array(
		'title'  => $this->title,
		'items'  => $comps,
		'method' => $this->method,
		'action' => $this->action,
		'id'     => $this->id,
		'enctype'=> $this->enctype,
		'target' => $this->target,
		'formTemplate'    => $this->formTemplate,
		'messageTemplate' => $this->successTemplate,
		'openTag'=> $this->openTag,
		'closeTag'=> $this->closeTag,
		'formType'=> $this->formType,
		);
		$context->set('options', $options);
		$context->set('noscrumbs', true);
		return new Renderable($this->formTemplate);
	}
	switch($this->formType) {
		case Form::FORM_OK_CANCEL:
			$url     = $context->getDomain()->getBaseURL()."request.php";
			$back    = $this->getValueOf('UWS_BACK_ACTION');
			if (!$back) {
				$back = "default";
			} else {
				$back    = $context->getActionByAlias($back);
				$back    = $back->getUrlCode();
			}
			$filters = urldecode(urldecode($this->getValueOf('UWS_LIST_FILTER')));
			$gl = new GridLayout(1, 2);
			$gl->addComponent(new ButtonComponent('Aceptar', 'submit'), 1, 1, 1, 1, 'right')
			   ->addComponent(new ButtonComponent('Cancelar', 'submit', "return uws_load('$url', 'action=$back$filters', 'desktop_center_content')"), 1, 2, 1, 1, 'left');
			$this->addComponent($gl)
				 ->addComponent(new HiddenComponent('', 'uwsSubmit', null, '1'));
			break;
		case Form::FORM_YES_NO:
			$url     = $context->getDomain()->getBaseURL()."request.php";
			$back    = $this->getValueOf('UWS_BACK_ACTION');
			if (!$back) {
				$back = "default";
			} else {
				$back    = $context->getActionByAlias($back);
				$back    = $back->getUrlCode();
			}
			$filters = urldecode(urldecode($this->getValueOf('UWS_LIST_FILTER')));
			$gl = new GridLayout(1, 2);
			$gl->addComponent(new ButtonComponent('Sí', 'submit'), 1, 1, 1, 1, 'right')
			   ->addComponent(new ButtonComponent('No', 'submit', "return uws_load('$url', 'action=$back$filters', 'desktop_center_content')"), 1, 2, 1, 1, 'left');
			$this->addComponent($gl)
				 ->addComponent(new HiddenComponent('', 'uwsSubmit', null, '1'));
			break;
		case Form::FORM_BACK:
			$url     = $context->getDomain()->getBaseURL()."request.php";
			$back    = $this->getValueOf('UWS_BACK_ACTION');
			if (!$back) {
				$back = "default";
			} else {
				$back    = $context->getActionByAlias($back);
				$back    = $back->getUrlCode();
			}
			$filters = urldecode(urldecode($this->getValueOf('UWS_LIST_FILTER')));
			$gl = new GridLayout(1, 1);
			$gl->addComponent(new ButtonComponent('Volver', 'submit', "return uws_load('$url', 'action=$back$filters', 'desktop_center_content')"), 1, 1, 1, 1, 'center');
			$this->addComponent($gl)
				 ->addComponent(new HiddenComponent('', 'uwsSubmit', null, '1'));
			break;
		case Form::FORM_SAVE_BACK:
			$url     = $context->getDomain()->getBaseURL()."request.php";
			$back    = $this->getValueOf('UWS_BACK_ACTION');
			if (!$back) {
				$back = "default";
			} else {
				$back    = $context->getActionByAlias($back);
				$back    = $back->getUrlCode();
			}
			$filters = urldecode(urldecode($this->getValueOf('UWS_LIST_FILTER')));
			$gl = new GridLayout(1, 2);
			$gl->addComponent(new ButtonComponent('Guardar', 'submit'), 1, 1, 1, 1, 'right')
			   ->addComponent(new ButtonComponent('Volver', 'submit', "return uws_load('$url', 'action=$back$filters', 'desktop_center_content')"), 1, 2, 1, 1, 'left');
			$this->addComponent($gl)
				 ->addComponent(new HiddenComponent('', 'uwsSubmit', null, '1'));
			break;
		default:
			break;
	}
	$context->set('options', $this->toArray());
	return new Renderable($this->formTemplate);
    // Bouml preserved body end 0013BE85
  }

  /**
   * Recupera el tipo de formulario.
   * @return String Una de las constantes predefinidas.
   */
  final public function getFormType()
  {
    return $this->formType;
  }

  /**
   * Configura el tipo de formulario. Por el momento esto afecta la botonera nada m�s.
   * 
   * @param string $value el tipo de formulario a representar.
   * 
   * @return Form el formulario modificado.
   */
  public function setFormType($value)
  {
    $this->formType = $value;
    return $this;
  }

  /**
   * Toma una serie de nombres de componentes que han sido actualizados recientemente.
   * 
   * @param string $component el nombre de componente, se puede pasar más de uno.
   * 
   * @return Form El objeto Form.
   */
  public function updated($component)
  {
    // Bouml preserved body begin 00196A05
	$numargs = func_num_args();
	for($i=0;$i < $numargs; $i++) {
		$this->updateds[func_get_arg($i)]=true;
	}
	$this->closeTag=true;
	return $this;
    // Bouml preserved body end 00196A05
  }

  /**
   * Configura la función que será invocada al enviar el formulario. La firma de dicha función es: function (form, validations, target)
   * 
   * @param string $function el nombre de la función
   * 
   * @return Form el formulario modificado
   */
  public function setAjaxAction($function)
  {
    // Bouml preserved body begin 001C8A05
	$this->ajaxAction = $function;
	return $this;
    // Bouml preserved body end 001C8A05
  }

  /**
   * Recupera el nombre de la función que se llama cuando se submitea el formulario.
   * 
   * @return string El nombre de la función.
   */
  public function getAjaxAction()
  {
    // Bouml preserved body begin 001C8A85
	return $this->ajaxAction;
    // Bouml preserved body end 001C8A85
  }

}
?>