<?php
require_once 'iplan/web/Form.php';
require_once 'iplan/web/FormCondition.php';
require_once 'iplan/security/ApplicationContext.php';
require_once 'iplan/orm/ORM.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Es una clase de utiler�a que permite hacer un juego de mostrar o no un formulario en base al resultado de otro.
 */
class ComposedForm {
  /**
   * @var string un identificador aleatorio que se remite con el formulario.
   */
  private $session_identifier;

  /**
   * @var Form el formulario que se muestra a�n cuando no hay condiciones
   */
  private $defaultForm;

  /**
   * @var array la lista de condiciones y sus respectivos formularios.
   */
  private $conditions;

  /**
   * Recupera el formulario principal
   * 
   * @return Form el formulario predeterminado
   */
  final public function getDefaultForm()
  {
    return $this->defaultForm;
  }

  /**
   * Configura el formulario predeterminado
   * 
   * @param Form $value el formulario a mostrar a�n cuando no haya condiciones cumplidas.
   * 
   * @return ComposedForm la instancia debidamente configurada
   */
  public function setDefaultForm($value)
  {
    $this->defaultForm = $value;
    return $this;
  }

  /**
   * Crea un objeto que es capaz de decidir si debe o no mostrar un formulario en base a un conjunto de condiciones.
   * Es de mencionar que si el form es un ABMForm entonces se asume que para pasar al siguiente formulario se requiere que �ste primero se haya GUARDADO con �xito. Es decir que Form::save() se ejecute correctamente.
   * @param string $rule el nombre de la regla. Posteriormente a su definici�n podr� ser utilizada por otras reglas a trav�s de la condici�n: <nombreRegla>.success o <nombreRegla>.fail
   * @param string $class la clase del formulario que se debe exhibir cuando se cumplen las condiciones
   * @param boolean $final un valor booleano que indica si se trata de un formulario final o no
   * @param array $tests un arreglo donde cada �tem tiene la forma ('componente'=>'valor') y se emplea para determinar si se debe o no mostrar el formulario
   * @param Closure $initFunction la funcion de inicializaci�n del form. Recibe el form anterior y el form a mostrar luego del loadValues()
   * 
   * @return ComposedForm La instancia de ComposedForm con la condici�n agregada.
   */
  public function addCondition($rule, $class, $final, $tests, $initFunction = null)
  {
    // Bouml preserved body begin 00123585
	if (!class_exists($class, false))
			throw new Exception("La clase $class no existe o no está disponible en el contexto actual.");
	
	if (is_subclass_of($class, 'ABMForm')) {
		$this->conditions[$rule]=new FormCondition($rule, $class, $final, true, $tests, $initFunction);
	} else {
		$this->conditions[$rule]=new FormCondition($rule, $class, $final, false, $tests, $initFunction);
	}
	return $this;
    // Bouml preserved body end 00123585
  }

  /**
   * Levanta la sesi�n para el Form activo.
   * 
   * @return array El arreglo con los status correspondientes a esta sesi�n.
   */
  public function prepareSession(&$context)
  {
    // Bouml preserved body begin 0012E185
	$this->session_identifier = $context->getParam('UWS_ComposedFormIdentifier', null);
	//Debug
	//echo "ID de sesión detectado: ".$this->session_identifier."<br>";
	if (is_null($this->session_identifier)) {
		$now = new DateTime();
		$this->session_identifier = md5($now->format('Y-m-d H:i:s')."_".rand());
	}
	
	$composedFormSession = $context->getFromSession('UWS_ComposedForms');
	$formsStatus = null;
	if (is_array($composedFormSession) && (count($composedFormSession)>0) && isset($composedFormSession[$this->session_identifier])) {
		$formsStatus = $composedFormSession[$this->session_identifier];
	}

	if (is_null($formsStatus) && is_array($composedFormSession) && (count($composedFormSession)>=50)) {
		//TODO: avisar que tiene 50 formularios abiertos y que no puede continuar hasta que no termine alguno.
		//     Dar un botón para limpiar todo con una advertencia bien en rojo.
		throw new Exception('No tengo template para avisar de 5+ forms abiertos!');
	}
	
	//...pero antes, si no existen status predefinidos, los creo todos en false
	if (is_null($formsStatus)) {
		foreach($this->conditions as $name => $form) {
			$formsStatus["$name.success"]=false;
		}
		$composedFormSession[$this->session_identifier]=$formsStatus;
		$context->setToSession('UWS_ComposedForms', $composedFormSession);
	}
	return $composedFormSession[$this->session_identifier];
    // Bouml preserved body end 0012E185
  }

  /**
   * Recupera un arreglo de los formularios que son v�lidos para mostrarse seg�n los par�metros del contexto y el status de la sesi�n.
   * @param ApplicationContext $context el contexto en el que se da la validaci�n
   * @param array $status el arreglo con los datos de los estados actuales de los formularios
   * @param ORM $orm el manejador de objetos pasado por par�metro. Requerido para los ABMForm.
   * 
   * @return array La lista de formularios que cumplen las condiciones actuales
   */
  public function getValidsForms(&$context, $status, &$orm = null)
  {
    // Bouml preserved body begin 0012E205
	//Obtengo los forms que cumplen las condiciones
	$conditionsOk = null;
	if (is_array($this->conditions) && (count($this->conditions)>0)) {
		foreach($this->conditions as $ruleName=>$condition) {
			if ($condition->test($context, $status, $orm))
				$conditionsOk[]=$condition;
		}
	}
	return $conditionsOk;
    // Bouml preserved body end 0012E205
  }

  /**
   * Dada una lista de condiciones las prepara para que se exhiban en cadena. Es decir configura los forms internos para abran o cierren los tags de "form" y para que incluyan un hidden con la informaci�n de sesi�n.
   * 
   * @param array $formConditions Un arreglo de condiciones.
   * 
   * @return array el arreglo de los formularios de las condiciones ya preparados.
   */
  public function prepareForms($formConditions)
  {
    // Bouml preserved body begin 0012E285
	//Preparo un Hidden que lo voy a sumar a cada form que muestre incluso el default
	$hiddenIdentifier = new HiddenComponent();
	$hiddenIdentifier->setName('UWS_ComposedFormIdentifier')->setValue($this->session_identifier);
	if (is_null($this->defaultForm->getComponent('UWS_ComposedFormIdentifier')))
		$this->defaultForm->addComponent($hiddenIdentifier);

	//Para cada form que cumple la condición los recorro y verifico si es final o no.
	//Si es final rompo el ciclo (sólo se admite 1 formulario "final".
	//Nótese que el tag <form> es abierto por el defaultForm y que por eso los forms
	//intermedios no lo generan y se van dejando también sin tag de cierre los forms
	//que quedan "sandwich". Sólo el último alternativo cierra el </form>.
	$conditionalsForms=null;
	if (!is_null($formConditions)) {
		$this->defaultForm->setCloseTag(false);
		for($m=0;$m < count($formConditions);$m++) {
			$condition = &$formConditions[$m];
			$form = $condition->getForm();
			$form->setOpenTag(false)->setCloseTag(true);
			if ($condition->isFinal()) {
				$form->addComponent($hiddenIdentifier);
				$conditionalsForms[]=$condition;
			} elseif ($condition->needSave()) {
				$form->addComponent($hiddenIdentifier);
				$conditionalsForms[]=$condition;
			} else {
				//$this->defaultForm->setCloseTag(false);
				//$form->setOpenTag(false);
				if (count($conditionalsForms)>0) {
					$conditionalsForms[count($conditionalsForms)-1]->getForm()->setCloseTag(false);
				}
				$conditionalsForms[]=$condition;
			}
		}
	}
	return $conditionalsForms;
    // Bouml preserved body end 0012E285
  }

  /**
   * Solicita al formulario que realice la tarea para la que est� configurado.
   * 
   * @param ApplicationContext $context el contexto en el cual el Form debe hacer su trabajo.
   * @param ORM $orm el ORM a usar
   * @return Renderable la pantalla que debe mostrar.
   */
  public function deploy(&$context, &$orm)
  {
    // Bouml preserved body begin 00123485
	
	//Me fijo si existe un identificador de sesión oculto o sino creo uno.

	$formsStatus = $this->prepareSession($context);
	$conditionsOk = $this->getValidsForms($context, $formsStatus, $orm);
//	//Debug
//	echo "Estado de reglas: <br/><ul>";
//	if (!is_null($formsStatus)) {
//		foreach($formsStatus as $ruleName=>$value) {
//			echo "<li>$ruleName=>".($value?'true':'false')."</li>";
//		}
//		echo "</ul>";
//	}
	$js_validations = "";
	$hiddenIdentifier = new HiddenComponent();
	$hiddenIdentifier->setName('UWS_ComposedFormIdentifier')->setValue($this->session_identifier);
	
	$this->defaultForm->addComponent($hiddenIdentifier)
					  ->setContext($context)
					  ->loadValues();
	$previusForm = $this->defaultForm;
	if (!is_null($conditionsOk)) {
		//Deshabilito el form default:
		foreach($this->defaultForm->getComponent() as $comp) {
			$comp->setDisabled(true);
			$hidden = new HiddenComponent();
			$class = get_class($comp);
			switch ($class) {
				case 'SelectComponent':
					$hidden->setName($comp->getName())
						   ->setValue($comp->getSelected());
					break;
				case 'MultiselectComponent':
					$hidden->setName($comp->getName().'[]')
						   ->setValue($comp->getValue());
					break;
				default:
					$hidden->setName($comp->getName())
						   ->setValue($comp->getValue());
			}
			$this->defaultForm->addComponent($hidden);
		}
		$conditionalsForms = $this->prepareForms($conditionsOk);
		$arrayForms = null;
		$first=false;$noErrors=false;
		for($k=0; $k < count($conditionalsForms);$k++) {
			if ($first) { $k=0;$first=false; }
			$form = $conditionalsForms[$k]->getForm($context, $orm);
			$form->loadValues();
			$conditionalsForms[$k]->init($context, $previusForm);
			if (is_a($form, 'ABMForm')) {
				if ($formsStatus[$form->getPrefix().'.success']) {
					$previusForm = $form;
					continue;
				}
				$renderable = $form->deploy($context, $orm);
				if ($renderable->getTemplate()==$form->getSuccessTemplate()) {
					if ($conditionalsForms[$k]->isFinal()) {
						$mySession = $context->getFromSession('UWS_ComposedForms');
						unset($mySession[$this->session_identifier]);
						$context->setToSession('UWS_ComposedForms', $mySession);
						return $renderable;
					} else {
						$mySession = $context->getFromSession('UWS_ComposedForms');
						$formsStatus[$form->getPrefix().".success"]=true;
						$mySession[$this->session_identifier]=$formsStatus;
						$context->setToSession('UWS_ComposedForms', $mySession);
						
						//Reevaluo las reglas:
						foreach($conditionsOk as $condition)
							$condition->clear();
						$conditionsOk = $this->getValidsForms($context, $formsStatus, $orm);
						$conditionalsForms = $this->prepareForms($conditionsOk);
						$first=true;$noErrors=true;
						continue;
					}
				} else {
					$mySession = $context->getFromSession('UWS_ComposedForms');
					$formsStatus[$form->getPrefix().".success"]=false;
					$mySession[$this->session_identifier]=$formsStatus;
					$context->setToSession('UWS_ComposedForms', $mySession);
					$options = $context->get('options');
					if ($noErrors) {
						$arrayForms[]=array('options'=>$options, 'errors'=>'');
					} else {
						$arrayForms[]=array('options'=>$options, 'errors'=>$context->getParam('errors', null, 'C'));
					}
					if ($options['js_validations'] != "{}") {
						if ($js_validations != "") {
							$js_validations = substr($js_validations, 0, strlen($js_validations)-1).",".substr($options['js_validations'],1);
						} else {
							$js_validations = $options['js_validations'];
						}
					}
					$context->reset('options')->reset('errors');
				}
			} else {
				$arrayForms[]=array('options'=>$conditionalsForms[$k]->toArray(), 'errors'=>null);
			}
		}
		$context->set('conditionalForms', $arrayForms);
	}
	
	$defaultFormOptions = $this->defaultForm->toArray();
	if ($js_validations != "") {
		if ($defaultFormOptions['js_validations'] != "{}") {
			$defaultFormOptions['js_validations']="{".substr($js_validations, 1, strlen($js_validations)-1).",".substr($defaultFormOptions['js_validations'],1);
		} else $defaultFormOptions['js_validations']=$js_validations;
	}
	$context->set('defaultForm', $defaultFormOptions);
	return new Renderable('composedForm.xhtml');
    // Bouml preserved body end 00123485
  }

}
?>