<?php
require_once 'iplan/web/TemplateInterface.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Es la base de la jerarquia de componentes HTML.
 * 
 * @method string getId()
 * @method Component setId()
 * @method string getName()
 * @method Component setName()
 * @method string getLabel()
 * @method Component setLabel()
 * @method string getStyle()
 * @method Component setStyle()
 * @method string getHint()
 * @method Component setHint()
 * @method string|array getValidate()
 * @method Component setValidate()
 * @method string getMaps()
 * @method Component setMaps()
 * @method string getValue()
 * @method Component setValue()
 * @method boolean getDisabled()
 * @method Component setDisabled()
 * @method int getSize()
 * @method Component setSize()
 * @method string getTemplate()
 * @method Component setTemplate()
 */
class Component implements TemplateInterface {
  /**
   * @var string Un símbolo que representa el evento "onBlur" de los componentes del navegador
   */
  const EVENT_BLUR = 'onblur';

  /**
   * @var string Un símbolo que representa el evento "onChange" de los componentes del navegador
   */
  const EVENT_CHANGE = 'onchange';

  /**
   * @var string Un símbolo que representa el evento "onClick" de los componentes del navegador
   */
  const EVENT_CLICK = 'onclick';

  /**
   * @var string Un símbolo que representa el evento "onDblClick" de los componentes del navegador
   */
  const EVENT_DBLCLICK = 'ondblclick';

  /**
   * @var string Un símbolo que representa el evento "onKeyDown" de los componentes del navegador
   */
  const EVENT_KEYDOWN = 'onkeydown';

  /**
   * @var string Un símbolo que representa el evento "onKeyPress" de los componentes del navegador
   */
  const EVENT_KEYPRESS = 'onkeypress';

  /**
   * @var string  Un símbolo que representa el evento "onKeyUp" de los componentes del navegador
   */
  const EVENT_KEYUP = 'onkeyup';

  /**
   * @var string Un símbolo que representa el evento "onFocus" de los componentes del navegador
   */
  const EVENT_FOCUS = 'onfocus';

  /**
   * @var string Un símbolo que representa el evento "onMouseDown" de los componentes del navegador
   */
  const EVENT_MOUSEDOWN = 'onmousedown';

  /**
   * @var string Un símbolo que representa el evento "onMouseMove" de los componentes del navegador
   */
  const EVENT_MOUSEMOVE = 'onmousemove';

  /**
   * @var string  Un símbolo que representa el evento "onMouseOut" de los componentes del navegador
   */
  const EVENT_MOUSEOUT = 'onmouseout';

  /**
   * @var string Un símbolo que representa el evento "onMouseOver" de los componentes del navegador
   */
  const EVENT_MOUSEOVER = 'onmouseover';

  /**
   * @var string Un símbolo que representa el evento "onMouseUp" de los componentes del navegador
   */
  const EVENT_MOUSEUP = 'onmouseup';

  /**
   * @var string Un símbolo que representa el evento "onReset" de los componentes del navegador
   */
  const EVENT_RESET = 'onreset';

  /**
   * @var string Un símbolo que representa el evento "onSelect" de los componentes del navegador
   */
  const EVENT_SELECT = 'onselect';

  /**
   * @var string Un símbolo que representa el evento "onSubmit" de los componentes del navegador
   */
  const EVENT_SUBMIT = 'onsubmit';

  /**
   * @var string el identificador con el que se crea el componente en HTML.
   */
  private $id;

  /**
   * @var string el nombre del par�metro que ser� remitido al script que procesa el formulario
   */
  private $name;

  /**
   * @var string el r�tulo que acompa�a al componente en pantalla, es opcional
   */
  private $label;

  /**
   * @var string El estilo personalizado que se aplica al componente (CSS)
   */
  private $style;

  /**
   * @var string descripci�n de la funci�n del componente o de como usarlo
   */
  private $hint;

  /**
   * @var array|string el conjunto de validaciones que debe aplicarse al componente
   */
  private $validate;

  /**
   * @var string opcional, es el atributo del que se toma o al que le asigna valor el componente
   */
  private $maps;

  /**
   * @var string el valor del dato que visualiza el componente
   */
  private $value;

  /**
   * @var bool indica si el componente est� deshabilitado
   */
  private $disabled;

  /**
   * @var int especifica el ancho en pantalla de un componente �, en el caso de los select, la cantidad de �tems a mostrar.
   */
  private $size;

  /**
   * @var string un template para poner los maps todos juntos en un solo "value"
   */
  private $template;

  /**
   * @var array La colección que almacena los eventos a los que responde el objeto
   */
  private $events = array();

  /**
   * Retorna un arreglo la definici�n del objeto a fin de que sea f�cilmente interpretable por un Template de TWIG.
   * Debido a que no todos los componentes tienen el maps y el name obligatorio, se asume como regla que:
   * <ul>
   * 	<li><b>si tiene name</b>, se usa el name.</li>
   * 	<li><b>si no tiene name</b>, se usa el maps de la siguiente manera: $prefix.maps.$postfix</li>
   * </ul>
   * 
   * @param string $prefix el prefijo que emplea para generar los nombres cuando no existe el name.
   * @param string $suffix el sufijo que emplea para generar los nombres cuando no existe el name.
   * 
   * @return array El arreglo que representa el objeto y sus propiedades
   */
  public function toArray($prefix = '', $suffix = '')
  {
    // Bouml preserved body begin 000E5205

	$value = $this->value;
	if (is_a($this->value, 'DateTime')) {
		$value = date_format($this->value, Utils::$FMT_DATETIME);
	}
	  
    //Se computa si es necesario calcular el nombre
    if (!$this->name) {
        if ($this->maps) {
            $name = $prefix.$this->maps.$suffix;
        } else {
            $name = null;//Este caso NO debería darse
        }
    } else {
		//En caso que el name esté configurado, verifico que tenga el prefix y suffix
		$startsWith=substr($this->name,0,strlen($prefix));
		$name = ($startsWith==$prefix?'':$prefix).$this->name;
		
		$endsWith =substr($this->name,(-1*strlen($suffix)));
		$name .= ($endsWith==$suffix)?'':$suffix;
    }
    
	if (is_string($this->validate))
		$this->validate = array($this->validate);
	
    //Se desglosan las validaciones complejas
    $validations = null;
    if (is_array($this->validate)) {
        for($i=0; $i < count($this->validate); $i++) {
			$vals = explode(" ", $this->validate[$i]);
			if ($vals[0] == 'regexp') {
				$firstSpace = strpos($this->validate[$i], ' ');
				$validations[$vals[0]]= substr($this->validate[$i], $firstSpace+1);
			} else {
				if (Validation::isComplexValidation($vals[0])) {
					$validations[$vals[0]]=$vals[1];
				} else {
					$validations[$vals[0]]="";
				}
			}
        }
    }
	
	$eventData = array();
	foreach($this->events as $event=>$functions) {
		foreach($functions as $key=>$data)
			$eventData[$event][$key]=$data['jsfunction'];
	}
	
    return array(
        'id'        =>(is_null($this->id))?($prefix.$this->name.$suffix):($prefix.$this->id.$suffix),
        'name'      =>$name,
        'label'     =>$this->label,
        'style'     =>$this->style,
        'hint'      =>$this->hint,
        'validate'  =>$validations,
        'maps'      =>$this->maps,
        'value'     =>$value,
        'disabled'  =>$this->disabled,
		'size'		=>$this->size,
		'template'	=>$this->template,
        'type'      =>'default',
		'events'	=>$eventData
    );
    // Bouml preserved body end 000E5205
  }

  /**
   * Uso este m�todo para hacer los setter y getter
   */
  public function __call($method, $params)
  {
    // Bouml preserved body begin 000E6C85
	$es = preg_match('/^(?P<operacion>(get|set|is){1})(?P<atributo>[A-Z_][a-zA-Z0-9_]*)$/', $method, $matches);
	if ($es) {
		$propertyName = $matches['atributo'];
		if(strlen($propertyName) > 0) $propertyName = lcfirst($propertyName);
		switch ($matches['operacion']) {
			case "get":
			case "is":
					return $this->$propertyName;
				break;
			case "set":
					$num_params = count($params);
					switch(true) {
						case $num_params == 1:
							$this->$propertyName = $params[0];
							break;
						case $num_params > 1: 
							$arr = array();
							for($i=0; $i < $num_params; $i++)
								$arr[]=$params[0];
							$this->$propertyName = $arr;
							break;
					}
					return $this;
				break;
		}		
	} else {
		throw new Exception("El m&eacute;todo ".get_class($this)."-&gt;$propertyName(...) no existe");
	}
    // Bouml preserved body end 000E6C85
  }

  /**
   * Retorna una versi�n string del nombre del componente y de su valor. S�lo sirve cuando ambos datos est�n definidos.
   * @param string $prefix el prefijo del nombre del elemento
   * @param string suffix el sufijo del nombre del elemento
   * @return string una porci�n de URL que representa el componente y su valor o el string vac�o si no tiene un value asignado o si no tiene un name.
   */
  public function toURL($prefix = '', $suffix = '')
  {
    // Bouml preserved body begin 00131B85
	if (!is_null($this->name) && !is_null($this->value) && ($this->value != '')) {
		if (is_array($this->value)) {
			if (count($this->value)>0) {
				$str = '';
				foreach($this->value as $value)
					$str .= $prefix.$this->name.$suffix."[]=$value&";
				return substr($str, 0, strlen($str)-1);
			} else {
				return '';
			}
		} else {
			return $prefix.$this->name.$suffix."=$this->value";
		}
	} else {
		return '';
	}
    // Bouml preserved body end 00131B85
  }

  /**
   * Crea un componente.
   * @param string $label el r�tulo del componente
   * @param string $name el nombre del componente
   * @param string $maps el valor al que mapea el componente
   * @param mixed $value el valor predeterminado del componente
   * @param string $template el template a utilizar para el renderizado
   * @return Component La instancia del componente inicializada.
   */
  public function __construct($label = null, $name = null, $maps = null, $value = null, $template = null)
  {
    // Bouml preserved body begin 00144D05
	if ((func_num_args() == 1)&& (is_array(func_get_arg(0)))) {
		$args = func_get_arg(0);
		foreach ($args as $attribute=>$value) {
			$this->$attribute=$args[$attribute];
		}
	} else {
		$this->label=$label;
		$this->name =$name;
		$this->maps =$maps;
		$this->value=$value;
		$this->template=$template;
	}
	return $this;
    // Bouml preserved body end 00144D05
  }

  /**
   * Indica que el componente es capaz de responder a un evento y que para el mismo es necesario ejecutar la función que se indica.
   * 
   * @param string $event el evento a capturar
   * @param string|Closure $function la función a ejecutar cuando se produzca el evento. Se puede pasar el nombre de la función como un string o definir la propia función como anónima. Los parámetros que recibirá son (ApplicationContext $context, Form $form, mixed $params).
   * @param mixed $params un valor (puede ser un array) que se pasará como parámetro adicional a la función
   * @param string $jsfunction una función de verificación de JS que será invocada para determinar si el llamado del evento tiene o no a lugar.
   * 
   * @return Renderable Devuelve el template que sirve de respuesta al pedido.
   */
  public function addEvent($event, $function, $params = null, $jsfunction = null)
  {
    // Bouml preserved body begin 00196885
	if($jsfunction === null) {
		$this->events[$event][]=array('function'=>$function, 'params'=>$params, 'jsfunction'=>'eventTrue');
	} else {
		$this->events[$event][]=array('function'=>$function, 'params'=>$params, 'jsfunction'=>$jsfunction);
	}
	return $this;
    // Bouml preserved body end 00196885
  }

  /**
   * Recupera la lista de eventos completa si no se pasa parámetro o de un evento determinado si se lo indica
   * @param string $event el evento del que se desea recuperar la configuración.
   * 
   * @return array|false La lista de eventos configurados para el componente. Si no hay ninguno retorna false.
   */
  public function getEvents($event = null)
  {
    // Bouml preserved body begin 00196985
	if (count($this->events)< 0) return false;
	if (is_null($event)) return $this->events;
	else if (isset($this->events[$event])) return $this->events[$event];
		 else return false;
    // Bouml preserved body end 00196985
  }

  /**
   * Desactiva la ejecución de un evento en particular.
   * 
   * @param string $event el evento a capturar
   * @param string|Closure $function la función a ejecutar cuando se produzca el evento. Se puede pasar el nombre de la función como un string o definir la propia función como anónima. Los parámetros que recibirá son (ApplicationContext $context, mixed $params).
   * 
   * @return Form el formulario modificado.
   */
  public function removeEvent($event, $function)
  {
    // Bouml preserved body begin 00198385
	throw new Exception('El método Component::removeEvent() no ha sido implementado aún');
    // Bouml preserved body end 00198385
  }

}
?>