<?php
require_once 'iplan/web/TemplateInterface.php';
require_once 'iplan/web/ActionInterface.php';
require_once 'iplan/security/ApplicationContext.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
class ActionButton implements TemplateInterface, ActionInterface {
  /**
   * @var string el c�digo de la acci�n del sistema
   */
  private $action;

  /**
   * @var string el modo de ejecuci�n de la acci�n: inline, popup, inline-popup
   */
  private $mode;

  /**
   * @var array un arreglo clave valor con los par�metros a pasar a la acci�n
   */
  private $params;

  /**
   * @var string el nombre del componente
   */
  private $caption;

  /**
   * @var string el �cono de la acci�n
   */
  private $icon;

  /**
   * @var string la t�cnica de refresco luego de ejecutada la acci�n: none, list, row
   */
  private $refresh;

  /**
   * @var string un template que se ejecuta para saber si la acción debe o no mostrarse.
   */
  private $condition;

  /**
   * @var string un mensaje que se muestra al usuario en ciertos casos (depende del modo de la acción)
   */
  private $message;

  /**
   * Este constructor puede usarse de modo habitual sin parámetros ó, en caso de querer aprovechar la configuración automática, es posible pasarle un contexto y una acción para que se configure sola.
   * @param ApplicationContext $context el contexto en el que se crea la acción.
   * @param string $action una cadena de texto de la forma "FacadeAlias.ActionAlias"
   * @param array $params los parámetros a mapear para invocar la accción.
   * @param string $icon el ícono de la acción
   * @param string $mode el modo de apertura o ejecución de la acción
   * @param string $refresh la forma en la que se debe refrescar la pantalla tras la ejecución de la acción
   * @param string $condition una condición que debe ser cumplida para que se visualice la acción. Se escribe como template de Twig.
   * @param string $message un mensaje a mostrar al usuario en ciertos modos.
   * 
   * @return ActionButton El botón creado, configurado o no dependiendo de si se le pasaron parámetros o no.
   */
  public function __construct(&$context = null, $actionStr = null, $params = null, $icon = null, $mode = null, $refresh = null, $condition = null, $message = null)
  {
    // Bouml preserved body begin 0013A385
	if (!is_null($context) && !is_null($actionStr)) {
		$action = $context->getActionByAlias($actionStr);
		if (!is_null($action)) {
			$this->caption = $action->getHint();
			$this->icon = $action->getIcon();
		}
		$this->action = $actionStr;
		$this->params = $params;
		$this->mode = $mode;
		$this->refresh = $refresh;
		$this->message = $message;
		$this->condition = $condition;
		if ($icon !==null)
			$this->icon=$icon;
	}
	return $this;
    // Bouml preserved body end 0013A385
  }

  /**
   * Configura el c�digo de la acci�n a ejecutar
   * 
   * @param string $action el c�digo de la acci�n
   * 
   * @return ActionInterface la acci�n modificada
   */
  public function setAction($action)
  {
    // Bouml preserved body begin 000EE005
    $this->action = $action;
    return $this;
    // Bouml preserved body end 000EE005
  }

  /**
   * El modo de ejecuci�n de la acci�n.
   * 
   * @param string $mode la forma en la cual la acci�n es invocada. Puede ser:
   * <ul><li>none, la acci�n se env�a a ejecutar y no requiere retorno al usuario</li>
   *     <li>inline, la acci�n se ejecuta como un link</li>
   *     <li>popup, el resultado de la acci�n se vuelca en un popup</li>
   *     <li>inline-popup, el resultado de la acci�n se devuelve en una ventana hecha con javascript y DOM</li>
   * </ul>
   */
  public function setMode($mode)
  {
    // Bouml preserved body begin 000EE085
    $this->mode=$mode;
    return $this;
    // Bouml preserved body end 000EE085
  }

  /**
   * Configura par�metros adicionales para la acci�n ejecutada.
   * 
   * @params string $params los par�metros a pasar a la acci�n
   * 
   * @return ActionInterface el objeto con los par�metros modificados
   */
  public function setParams($params)
  {
    // Bouml preserved body begin 000EE105
    $this->params = $params;
    return $this;
    // Bouml preserved body end 000EE105
  }

  /**
   * El �cono para la acci�n
   * @param string $icon el icono a configurar en la acci�n
   * @return ActionInterface el objeto configurado con el �cono seleccionado
   */
  public function setIcon($icon)
  {
    // Bouml preserved body begin 000EE185
    $this->icon=$icon;
    return $this;
    // Bouml preserved body end 000EE185
  }

  /**
   * Configura el esquema de refresco de la acci�n
   * 
   * @param string $refresh el modo de refresco de la acci�n
   * @return ActionInterface el objeto con el m�todo de refresco seleccionado
   */
  public function setRefresh($refresh)
  {
    // Bouml preserved body begin 000EE205
    $this->refresh=$refresh;
    return $this;
    // Bouml preserved body end 000EE205
  }

  /**
   * Retorna la acci�n configurada.
   * 
   * @return string el c�digo de la acci�n
   */
  public function getAction()
  {
    // Bouml preserved body begin 000F2285
    return $this->action;
    // Bouml preserved body end 000F2285
  }

  /**
   * Recupera el modo de ejecuci�n
   * 
   * @return string el modo de ejecuci�n de la acci�n
   */
  public function getMode()
  {
    // Bouml preserved body begin 000F2305
    return $this->mode;
    // Bouml preserved body end 000F2305
  }

  /**
   * Devuelve los par�metros adicionales.
   * 
   * @return string los par�metros adicionales que se hayan configurado en la acci�n
   */
  public function getParams()
  {
    // Bouml preserved body begin 000F2385
    return $this->params;
    // Bouml preserved body end 000F2385
  }

  /**
   * Devuelve la imagen de la acci�n.
   * 
   * @return string recupera la imagen de la acci�n
   */
  public function getIcon()
  {
    // Bouml preserved body begin 000F2405
    return $this->icon;
    // Bouml preserved body end 000F2405
  }

  /**
   * Recupera el m�todo de refresco de la acci�n
   * 
   * @return string recupera el m�todo de refresco de la acci�n
   */
  public function getRefresh()
  {
    // Bouml preserved body begin 000F2485
    return $this->refresh;
    // Bouml preserved body end 000F2485
  }

  /**
   * Retorna un arreglo la definici�n del objeto a fin de que sea f�cilmente interpretable por un Template de TWIG.
   * Debido a que no todos los componentes tienen el maps y el name obligatorio, se asume como regla que:
   * <ul>
   * 	<li><b>si tiene name</b>, se usa el name.</li>
   * 	<li><b>si no tiene name</b>, se usa el maps de la siguiente manera: $prefix.maps.$postfix</li>
   * </ul>
   * 
   * @param string $prefix el prefijo que emplea para generar los nombres cuando no existe el name.
   * @param string $postfix el sufijo que emplea para generar los nombres cuando no existe el name.
   * 
   * @return array El arreglo que representa el objeto y sus propiedades
   */
  public function toArray($prefix = '', $postfix = '')
  {
    // Bouml preserved body begin 000F2505
	if (is_string($this->params))
		$params = array($this->params);
	else $params = $this->params;
    return array(
        'action'    =>  $this->action,
		'mode'      =>  explode(" ", $this->mode),
        'icon'      =>  $this->icon,
        'params'    =>  $params,
        'refresh'   =>  $this->refresh,
        'caption'   =>  $this->caption,
		'message'   =>  $this->message,
		'condition' =>	$this->condition
    );
    // Bouml preserved body end 000F2505
  }

  final public function getCaption()
  {
    return $this->caption;
  }

  public function setCaption($value)
  {
    $this->caption = $value;
    return $this;
  }

}
?>