<?php



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Esta clase encapsula un template a ser mostrado por el framework.
 * 
 * Generalmente los m�todos de los Facade retornan Renderables's para indicarle a la Application qu� mostrar. Adem�s el renderable permite especificar a trav�s de su flag "downlodeable" si el contenido es para descarga directa o no.
 */
class Renderable {
  private $template = null;

  /**
   * @var boolean esta variable indica si el template es un listado o alg�n contenido descargable.
   */
  private $downlodeable = false;

  /**
   * @var array la lista de headers personalizados para el envio
   */
  private $headers = null;

  /**
   * @var string Un contenido alternativo al generado por el template
   */
  private $content = null;

  /**
   * @param string $templateName el nombre o alias del template definido en el Facade
   * @param boolean $downlodeable indica si el contenido devuelto tiene el objeto de ser descargado. En dicho caso se devolverá el template directamente sin agregarle contenido del index, ni nada.
   * 
   * @return Renderable Una clase capaz de renderizar el template.
   */
  public function __construct($templateName = null, $downlodeable = false)
  {
    // Bouml preserved body begin 000A2505
    $this->template = $templateName;
	$this->downlodeable = $downlodeable;
    // Bouml preserved body end 000A2505
  }

  final public function getTemplate()
  {
    return $this->template;
  }

  /**
   * Recupera el flag downlodeable.
   * 
   * @return boolean Devuelve un TRUE si el contenido est� pensado para descarga y FALSE sino.
   */
  public function isDownlodeable()
  {
    // Bouml preserved body begin 00126985
	return $this->downlodeable;
    // Bouml preserved body end 00126985
  }

  /**
   * Agrega un header que será remitido junto con la respuesta al cliente.
   * 
   * @param string $type el tipo o nombre del header
   * @param string $value el valor del header
   * 
   * @return Renderable El objeto actual con el header agregado.
   */
  public function addHeader($type, $value)
  {
    // Bouml preserved body begin 0019A685
	$this->headers[$type]=$value;
	return $this;
    // Bouml preserved body end 0019A685
  }

  /**
   * Retorna la lista de headers configurados para el pedido.
   * 
   * @return array La lista de los headers si existen o null en caso que no se haya definido ninguno.
   */
  public function getHeaders()
  {
    // Bouml preserved body begin 0019A705
	return $this->headers;
    // Bouml preserved body end 0019A705
  }

  /**
   * Establece in situ el contenido que se desea que devuelva la aplicación. Emplear esta opción significa anular la generación via template.
   * 
   * @param string $content el contenido a devolver.
   * 
   * @return Renderable La instancia con el contenido establecido y la generación de template anulada.
   */
  public function setContent($content)
  {
    // Bouml preserved body begin 0019A785
	$this->content = $content;
	$this->template = null;
	return $this;
    // Bouml preserved body end 0019A785
  }

  /**
   * Carga el contenido del renderable con el de un archivo del file system.
   * 
   * @param string $filename el nombre completo, incluyendo la ruta, del archivo que se desea retornar.
   * 
   * @return Renderable La instancia con el contenido actualizado al del archivo indicado y el sistema de generación via template anulado.
   */
  public function loadFile($filename)
  {
    // Bouml preserved body begin 0019A805
	if (file_exists($filename)) {
		$this->template = null;
		$this->content = file_get_contents($filename);
		return $this;
	} else throw new Exception("No se puede encontrar el archivo $filename");
    // Bouml preserved body end 0019A805
  }

  /**
   * Devuelve el contenido establecido por medio de las funciones setContent() o loadFile()
   * 
   * @return string El contenido especificado manualmente para el renderable
   */
  public function getContent()
  {
    // Bouml preserved body begin 0019A885
	return $this->content;
    // Bouml preserved body end 0019A885
  }

  /**
   * Configura el template del Renderable. Téngase en cuenta que llamar este método posterior a setContent() o loadFile() deshabilita la salida preconfigurada.
   * 
   * @param string $template El template de Twig a parsear como salida.
   * 
   * @return Renderable El template que genera la salida deseada.
   */
  public function setTemplate($template)
  {
    // Bouml preserved body begin 0019A905
	$this->template = $template;
	return $this;
    // Bouml preserved body end 0019A905
  }

  /**
   * Configura el Renderable indicando si es para descarga.
   * 
   * @param boolean $downlodeable un booleano que es true en caso que sea una descarga y false si no.
   * 
   * @return Renderable La instancia actual del Renderable con la configuración de descarga establecida.
   */
  public function setDownlodeable($downlodeable)
  {
    // Bouml preserved body begin 0019A985
	$this->downlodeable=$downlodeable;
	return $this;
    // Bouml preserved body end 0019A985
  }

}
?>