<?php
require_once 'iplan/security/User.php';
require_once 'iplan/security/AbstractManager.php';
require_once 'iplan/security/Provider.php';
require_once 'iplan/security/Domain.php';
require_once 'iplan/security/Application.php';
require_once 'iplan/security/Action.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Ésta clase determina contexto de ejecución en el cual se solicitó realizar una determinada tarea. Básicamente brinda información del proveedor de servicio, el sitio del proveedor desde el cual se está haciendo el pedido, el usuario que se autenticó en dicho sitio y los parámetros con los que se realizó la petición.
 * Idealmente un proceso no necesita nada más realizar validaciones internas y llevar a cabo la tarea encomendada, incluyendo cualquier información para log.
 * Empleando el contexto se posibilita además la interacción entre las instancias de los módulos habilitados.
 * El contexto está disponible tanto para el PHP, como para los templates.
 */
class ApplicationContext {
  /**
   * @var User el usuario que inició sesión en el dominio del Provider
   */
  private $user;

  /**
   * @param array $globals un arreglo donde se almacenan las variables de sesi�n y otros que se quieran pasar globalmente a trav�s de la sesi�n
   */
  private $globals;

  /**
   * @var array la lista de descendientes de AbstractManagers que tiene habilitado el usuario.
   */
  private $managers;

  /**
   * @var Provider el proveedor en el que est� logueado el usuario
   */
  private $provider;

  /**
   * @var Domain el dominio en el que está actuando el usuario
   */
  private $domain;

  /**
   * @var Application la aplicación a la que pertenece el contexto
   */
  private $application;

  /**
   * Crea un contexto de la aplicación, cargando las facilidades que el usuario tiene para el proveedor del servicio indicado en esa URL.
   * 
   * @param Application $application la aplicación que generó el contexto
   * @param User $user el usuario que se autenticó en el sistema
   * @param Provider $provider el proveedor de servicios al cual está accediendo
   * @param Domain $domain el dominio en el cual se autenticó el usuario
   * 
   * @return ApplicationContext Una instancia que configura un contexto de uso de la aplicación. En base a él los procesos y funciones de la aplicación web tienen información de la solicitud de servicio.
   */
  public function __construct(&$application, &$user, $provider, $domain)
  {
    // Bouml preserved body begin 000AE405
    $this->application = $application;
    $this->user = $user;
    $this->managers = array();
    $this->provider = $provider;
	$this->domain = $domain;
    return $this;
    // Bouml preserved body end 000AE405
  }

  /**
   * Guarda un valor en el contexto global. Éste valor luego puede ser recuperado desde el PHP o los templates.
   * 
   * @param string $name el nombre de la variable a crea o cuyo valor se quiere establecer
   * @param mixed $value el valor para dicha variable
   * 
   * @return ApplicationContext Tras la ejecución se retorna una referencia al mismo contexto, en el cual se ha configurado la variable de contexto.
   */
  public function set($name, $value)
  {
    // Bouml preserved body begin 000A4285
    if (is_null($value)) {
        if (isset ($this->globals[$name])) {
            unset($this->globals[$name]);
            $this->globals=array_values($this->globals);
        }
    } else {
        $this->globals[$name]=$value;
    }
	return $this;
    // Bouml preserved body end 000A4285
  }

  /**
   * Devuelve el valor de una variable de contexto.
   * 
   * @param string el nombre de la variable cuyo valor se quiere recuperar
   * 
   * @return mixed Ésta función devuelve el valor de la variable solicitada o null si no existe.
   */
  public function get($name)
  {
    // Bouml preserved body begin 000C1E85
	if (isset($this->globals[$name]))
		return $this->globals[$name];
	else return null;
    // Bouml preserved body end 000C1E85
  }

  /**
   * Desregistra una variable de contexto.
   * 
   * @param string $name el nombre de la variable a quitar
   * 
   * @return ApplicationContext El propio contexto en el cual se ha quitado la variable indicada, si es que existía.
   */
  public function reset($name)
  {
    // Bouml preserved body begin 000A4305
    unset($this->globals[$name]);
	$this->globals=array_values($this->globals);
	return $this;
    // Bouml preserved body end 000A4305
  }

  /**
   * Agrega un Facade al contexto del usuario.
   * @param string $alias el nombre con el cual se referirá al manager desde los templates
   * @param AbstractManager $manager el manager de pedidos o Facade a ser agregado al contexto
   * 
   * @return ApplicationContext El contexto ya configurado con el nuevo Facade.
   */
  public function addManager($alias, &$manager)
  {
    // Bouml preserved body begin 000AE485
    $this->managers[$alias] = $manager;
	return $this;
    // Bouml preserved body end 000AE485
  }

  /**
   * Recupera un manager o Facade del contexto de la aplicación.
   * 
   * @param string $alias el alias del manager
   * 
   * @return AbstractManager|FALSE Recupera el manager solicitado o devuelve FALSE si el manager no existe en el contexto actual. En dicho caso puede ser que el usuario no tuviera acceso a dicho módulo.
   */
  public function getManager($alias)
  {
    // Bouml preserved body begin 000B1A05
		if (isset($this->managers[$alias]))
			return $this->managers[$alias];
		else
			return false;
    // Bouml preserved body end 000B1A05
  }

  /**
   * Recupera un parámetro pasado al script por alguno de los medios habituales: $_POST, $_GET, $_REQUEST, $_SESSION, $_FILES o del contexto global. En términos ideales esas variables no deben ser usadas por los métodos y sólo deben obtener dichos valores a través de ésta función.
   * 
   * @param string $name el nombre del parámetro a recuperar
   * @param mixed $default el valor a devolver en caso que el parámetro no se encuentre
   * @param string $where un string que indica en donde buscar y en qué orden: P=POST, G=GET, R=REQUEST y C=Context (variables globales)
   * 
   * @return mixed El valor del primer valor que coincida con el nombre dado ó el valor predeterminado en caso de que no se encuente.
   * 
   * Un ejemplo de uso sería:
   * <code>
   * <?
   * function showContextData(ApplicationContext $context) {
   *     echo "Nombre de usuario: ".$context->getUser()->getUsername()."<br/>";
   *     echo "Logueado en: ".$context->getDomain()->getBaseURL()."<br/>";
   * }
   * ?>
   * </code>
   */
  public function getParam($name, $default = null, $where = 'PGCRF')
  {
    // Bouml preserved body begin 000B4E85
    $where = strtoupper($where);
    for($i=0; $i<strlen($where); $i++) {
        switch($where[$i]) {
            case 'P':
            case 'G':
            case 'F':
            case 'R': 
				if (!is_null($value= $this->application->getParam($name, null, $where[$i]))) return $value; 
				break;
            case 'C': if (isset ($this->globals[$name])) return $this->globals[$name]; break;
            default:
                throw new Exception("Error al recuperar parámetro $name en ".  strtoupper($where));
        }
    }
    return $default;
    // Bouml preserved body end 000B4E85
  }

  /**
   * Devuelve el usuario que se autenticó en el dominio de la aplicación.
   * 
   * @return User el usuario de la sesión actual.
   */
  final public function getUser()
  {
    return $this->user;
  }

  /**
   * Asigna el usuario. Este método sólo debe ser empleado para cambiar el usuario activo en acciones tales como el "login".
   * @param User $user el nuevo usuario del contexto
   * @return ApplicationContext El contexto modificado
   */
  public function setUser($user)
  {
    // Bouml preserved body begin 000C3805
    $this->user = $user;
    return $this;
    // Bouml preserved body end 000C3805
  }

  /**
   * Retorna el Provider del sitio actual. Del Provider activo dependen los perfiles, usuarios, acciones, facades y demás parámetros específicos a los que el usuario logueado se puede acceder en el sistema.
   * 
   * @return Provider El proveedor de servicio dueño del dominio actual
   */
  final public function getProvider()
  {
    return $this->provider;
  }

  /**
   * Retorna la acción en curso.
   * 
   * @return Action La acción que se la aplicación identificó como objeto de ejecución.
   */
  public function getCurrentAction()
  {
    // Bouml preserved body begin 0010A585
    return $this->application->getCurrentAction();
    // Bouml preserved body end 0010A585
  }

  /**
   * Convierte los datos del contexto en variables que pueda entender el template de Twig.
   * 
   * @return array Un arreglo con los objetos modelados según el estándar adoptado al estilo Twig.
   */
  public function toTwigArguments()
  {
    // Bouml preserved body begin 000B1A85
	$profs = array();
	$orm = $this->application->getORM();
	$aPerfiles = $orm->query("Profile")
					->filterBy("users.id", '=', $this->user->getId())
					->filterBy("domains.id", '=', $this->getDomain()->getId())
					->find()
					;
	if (($aPerfiles) && (count($aPerfiles) > 0)) {
		foreach($aPerfiles as $profile) {
			if (($profile->getWelcome() != '') && ($profile->getWelcome()!=null))
				$profs[]= array("template" => $profile->getWelcome(), "id" => $profile->getId(), "tiene_manual" => ($profile->getManual() != ""));
		}
	} else {
		$profs = array();
	}
	$this->set('templates_profile', $profs);
	$this->set('domain', $this->domain->getBaseURL());
	$urlInfo = parse_url($this->domain->getBaseURL());
	$this->set('dir_name', $urlInfo['path']);
    $array = array();
    if (is_array($this->globals)) {
        foreach($this->globals as $key => $value) {
            $array[$key] = $value;
        }
    }
    foreach($this->managers as $alias => $manager) {
        $array[$alias] = $manager;
		if (is_array($manager->__listActions()) && (count($manager->__listActions()) > 0 )) {
			foreach($manager->__listActions() as $action) {
				$array[$alias]->$action = "$alias.$action";
			}
		}
    }
	if (!is_null($this->application->getCurrentAction()))
		$array['CurrentAction']=$this->application->getCurrentAction()->getStringCommand();
    return $array;
    // Bouml preserved body end 000B1A85
  }

  /**
   * Guarda un valor en la sesión de usuario activa.
   * 
   * @param string $name el nombre de la variable de sesión cuyo valor se quiere establecer
   * @param mixed $value el valor de dicha variable
   * 
   * @return ApplicationContext El mismo contexto en el cual se efectuó la configuración de la variable.
   */
  public function setToSession($name, $value)
  {
    // Bouml preserved body begin 0012DB05
	$_SESSION[$name]=$value;
	return $this;
    // Bouml preserved body end 0012DB05
  }

  /**
   * Devuelve el valor de una variable de sesión.
   * 
   * @param string el nombre de la variable cuyo valor se quiere recuperar
   * 
   * @return mixed El valor de la variable solicitada si es que la misma existe.
   */
  public function getFromSession($name)
  {
    // Bouml preserved body begin 0012DB85
	if (isset($_SESSION[$name]))
		return $_SESSION[$name];
	else
		return null;
    // Bouml preserved body end 0012DB85
  }

  /**
   * Quita un valor de la sesión.
   * @param string $name el nombre de la variable a quitar
   * 
   * @return mixed Devuelve el valor de la variable solicitada o null si no existe.
   */
  public function resetFromSession($name)
  {
    // Bouml preserved body begin 0012DC05
	unset ($_SESSION[$name]);
    // Bouml preserved body end 0012DC05
  }

  /**
   * Recupera una acción (si es que está permitida) a través de su representación en string.
   * 
   * @param string $actionAlias una cadena de texto con la forma "FacadeAlias.ActionAlias"
   * 
   * @return Action La acción que responde a ese alias o null si la misma no existe o no está permitida.
   */
  public function getActionByAlias($actionAlias)
  {
    // Bouml preserved body begin 0013A485
	return $this->application->getActionByAlias($actionAlias);
    // Bouml preserved body end 0013A485
  }

  /**
   * Recupera el dominio en el cual se está ejecutando la aplicación (está dado por la URL a la que accedió para entrar al sistema)
   * 
   * @return Domain El objeto que representa al dominio en la aplicación
   */
  public function getDomain()
  {
    // Bouml preserved body begin 0019E185
	return $this->domain;
    // Bouml preserved body end 0019E185
  }

}
?>