<?php
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORMDefinition.php';
require_once 'iplan/orm/ORMObject.php';
require_once 'iplan/database/Connection.php';
require_once 'iplan/orm/ORMAttribute.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Este objeto se crea indicando cuál es la clase "pivote", con cuyo nombre recupera la definición desde el ORM.
 * 
 * Las condiciones de filtrado se especifican a través de llamados a filterBy() y el conjunto de resultados se ordena según el orden de llamados a orderBy().
 * 
 * Es posible segmentar el conjunto de resultados usando los pares de funciones: from()/to() ó from()/top(), según se quieran indicar fila de inicio y fin o fila de inicio y cantidad de registros respectivamente.
 * 
 * El resultado de la consulta se obtiene llamando a find().
 */
class ORMQuery {
  /**
   * @var ORM el manejador de objetos desde el cual se toman las definiciones
   */
  private $orm;

  /**
   * @var string el nombre de la clase pivote de la consulta
   */
  public $pivotClass;

  /**
   * @var ORMDefinition la definici�n de la clase pivote
   */
  private $definition;

  /**
   * @var int el n�mero de fila del primer registro del resultado.
   */
  public $from;

  /**
   * @var int la cantidad de filas a recuperar.
   */
  public $count;

  /**
   * @var array un arreglo con las condiciones indicadas por los llamados a filterBy()
   * El formato del arreglo es:
   * $conditions['filters'][$field][$operator][0..n]=array($value, $other, $connector);
   * $conditions['subqueries'][$field][0..n]=ORMQuery();
   */
  public $conditions;

  /**
   * @var boolean Indica si la consulta debe retornar tuplas �nicas
   */
  public $distinct = FALSE;

  /**
   * @var boolean determina si el usuario ha o no especificado columnas personalizadas para el resultado
   */
  public $uses_custom_columns;

  /**
   * @var boolean indica si se han especificado atributos de agrupaci�n
   */
  public $uses_groups;

  /**
   * @var array una lista de los atributos sobre los cuales se especific� que se haga LEFT JOIN mediante dontWorryAbout()
   */
  public $leftJoins;

  /**
   * @var int esta variable, si est� configurada, determina la cantidad de filas a recuperar simult�neamente.
   */
  private $maxFetchRows;

  /**
   * @var array la lista de grupos de filtros asociados.
   */
  public $associations;

  /**
   * @var int es el contador de asociaciones.
   */
  public $countAssociations;

  /**
   * @var int cuenta el n�mero de columna de ordenamiento por la que va. Es de uso interno. La idea es que si se llama varias veces a $query->orderBy(), el contador se sostenga entre los llamados.
   */
  public $sortsCount = 0;

  /**
   * Crea un objeto de consulta que retorna elementos del tipo de la clase indicada.
   * 
   * @param string $class la clase de objetos que retornar� la consulta.
   * @param ORM $orm el objeto ORM al que se ata la consulta.
   * @return ORMQuery un objeto con los criterios de una consulta.
   */
  public function __construct($class, &$orm)
  {
    // Bouml preserved body begin 00096C05
    $this->orm=$orm;
    $this->pivotClass=$class;
    $this->definition=$orm->getDefinition($class);
    $this->uses_custom_columns = false;
    $this->uses_groups = false;
    $this->distinct = false;
	$this->countAssociations=0;
    return $this;
    // Bouml preserved body end 00096C05
  }

  /**
   * Recupera la clase que se indicara como pivot de la consulta.
   * 
   * @return string el nombre de la clase pasado por par�metro en el llamdo al contructor de la instancia.
   */
  public function getClass()
  {
    // Bouml preserved body begin 00119C05
	return $this->pivotClass;
    // Bouml preserved body end 00119C05
  }

  /**
   * Devuelve el ORM al que pertenece la Query.
   * 
   * @return ORM el ORM que construy� la instancia de ORMQuery.
   */
  public function getORM()
  {
    // Bouml preserved body begin 00119C85
	return $this->orm;
    // Bouml preserved body end 00119C85
  }

  /**
   * Establece la cantidad de filas a saltear del conjunto de resultados antes de considerarla v�lida como respuesta.
   * 
   * @param int $row la fila a partir de la cual deseamos que se nos retornen resultados.
   * 
   * @return ORMQuery La consulta modificada. from() siempre debe llamarse antes que to() para calcular el diferencial.
   */
  public function from($row)
  {
    // Bouml preserved body begin 00098705
    if (isset($this->from) && isset($this->count)) {
        $tmpCount = $this->count;
        $tmpCount += ($this->from - $row);
        if ($tmpCount > 0) $this->count = $tmpCount;
    }
    $this->from = $row;
    return $this;
    // Bouml preserved body end 00098705
  }

  /**
   * Establece el m�ximo n�mero de fila que quiere recuperarse. Debe ser mayor al valor prefijado en from().
   * 
   * @param int $row el n�mero de fila m�xima del conjunto de resultado.
   * 
   * @return ORMQuery La consulta con el nuevo l�mite aplicado. Debe considerarse que si no se llam� a from() previamente no se va a calcular el diferencial.
   */
  public function to($row)
  {
    // Bouml preserved body begin 00098785
    if (isset($this->from) && isset($this->count)) {
        $tmpTo = $this->from+$this->count;
        $tmpTo += ($row-$tmpTo);
        if ($tmpCount > 0) $this->count = $tmpCount;
    } else {
        $this->count = ($row-$this->from)+1;
    }
    return $this;
    // Bouml preserved body end 00098785
  }

  /**
   * Establece cuantas filas se deben recuperar del conjunto de resultados.
   * 
   * @param int $limit la cantidad de filas a recuperar
   * 
   * @return ORMQuery El objeto Query modificado.
   */
  public function top($limit)
  {
    // Bouml preserved body begin 00098805
    $this->count = $limit;
    return $this;
    // Bouml preserved body end 00098805
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra clase, utilizando para el emparejamiento las condiciones dadas e incorporando al conjunto de resultados los atributos indicados.
   * @param string $attribute el atributo que determina la relaci�n con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $attributes los atributos nuevos a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados
   */
  public function joinWith($attribute, $alias, $attributes, $conditions)
  {
    // Bouml preserved body begin 000C8A85
	throw new Exception ("ORMQuery::joinWith() no está implementada aún");
    // Bouml preserved body end 000C8A85
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra consulta. Debido a que no se pueden inferir los resultados de la consulta tan f�cilmente como con el joinWith, a esta funci�n se le deben pasar los nombres de columnas devueltas por el query para hacer el emparejamiento y para definir nuevos campos para el conjunto de resultados.
   * 
   * @param ORMQuery $query la query con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $fields los campos nuevos (con denominaci�n completa) a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados tambi�n con denominaci�n completa
   */
  public function joinBy($query, $alias, $fields, $conditions)
  {
    // Bouml preserved body begin 000C8A05
	throw new Exception ("ORMQuery::joinBy() no está implementada aún");
    // Bouml preserved body end 000C8A05
  }

  /**
   * Busca el primer elemento de la consulta.
   * 
   * @return ORMObject el objeto en cuesti�n
   */
  public function findOne()
  {
    // Bouml preserved body begin 0009D785
    $res = $this->find(1, 1);
    if ($res === false)
        return false;
    else
        return $res[0];
    // Bouml preserved body end 0009D785
  }

  /**
   * @return string Imprime un cierto n�mero de espacios acorde al $level
   */
  public function echoTab()
  {
    // Bouml preserved body begin 0009F105
    echo str_pad("\n", 5 * self::$level, " ", STR_PAD_RIGHT);
    // Bouml preserved body end 0009F105
  }

  /**
   * @var int $level utilizado para fines de debug
   */
  static $level = 0;

  /**
   * @var boolean $debug indica si se debe hacer una salida de depuraci�n de los pasos de la consulta
   */
  static $debug = false;

  /**
   * Funci�n temporal s�lo para fines de debug. Decifra la cadena retornada por "build()".
   */
  public function decrypt(&$coded_sql)
  {
    // Bouml preserved body begin 000BEB05
    $db = $this->orm->getDatabase();
    $conn = $db->getConnection();
    $ps = $conn->prepareStatement("SELECT PKG_WEB_ACCESSOR.DECRYPT(:salida) as salida FROM DUAL");
	if ($db->isBoostEnabled()) {
		$conn->bind($ps, ":salida", $coded_sql,-1, Connection::T_INT);
	} else {
		$conn->bind($ps, ":salida", $coded_sql,-1, Connection::T_BINARY);
	}
    $resQuery = $conn->execute($ps);
	$fetched = $conn->fetchAll($ps);
    return $fetched;
    // Bouml preserved body end 000BEB05
  }

  /**
   * Establece una condici�n de b�squeda para la consulta.
   * 
   * @param string $attribute el campo sobre el cual se especifica la condici�n
   * @param int $operator la operaci�n de comparaci�n
   * @param mixed $value el valor utilizado para la comparaci�n
   * @param mixed $other otro valor en caso de ser necesario
   * @param string $connector la palabra 'AND' o 'OR' seg�n se quiera utilizar para concatenar la condici�n
   * 
   * @return ORMQuery la consulta con la condici�n incorporada.
   */
  public function filterBy($attribute, $operator, $value = "", $other = "", $connector = 'AND')
  {
    // Bouml preserved body begin 000CA605
	if (($this->associations !== null) && (count($this->associations)>0)) {
	  $idx = max(array_keys($this->associations));
	  list($associationId, $associationConnector) = $this->associations[$idx];
	} else {
	  $associationId=null;
	  $associationConnector=null;
	}
    if ($pointPos = strpos($attribute, '.')) {
       $atts = explode('.', $attribute);

       $currentSlot = &$this->conditions['filters'];
       $i = 1;
       foreach($atts as $key => $att) {
           if (count($atts) == $i++) {
              $currentSlot = &$currentSlot['locals'][$att];
           } else {
              $currentSlot = &$currentSlot['subquerys'][$att];
           }
       }
       $currentSlot[strtoupper($operator)][]=array($value, $other, $connector, $associationId, $associationConnector);
    } else {
      $this->conditions['filters']['locals'][$attribute][strtoupper($operator)][]=array($value, $other, $connector, $associationId, $associationConnector);
    }
    return $this;
    // Bouml preserved body end 000CA605
  }

  /**
   * Inicia un agrupamiento de condiciones. Todas los llamados a filterBy() que vengan luego del startGroup() ser�n considerados como un conjunto.
   * @param string $joinMode indica como se va a unir el conjunto con las siguiente condiciones.
   * @return ORMQuery La consulta con el grupo creado.
   */
  public function startGroup($joinMode = 'AND')
  {
    // Bouml preserved body begin 00146785
	$this->associations[$this->countAssociations] = array($this->countAssociations, $joinMode);
	$this->countAssociations++;
	return $this;
    // Bouml preserved body end 00146785
  }

  /**
   * Finaliza un grupo de condiciones previamente iniciado con startGroup().
   * 
   * @return ORMQuery La consulta con el grupo cerrado.
   */
  public function endGroup()
  {
    // Bouml preserved body begin 00146805
	unset($this->associations[count($this->associations)-1]);
	return $this;
    // Bouml preserved body end 00146805
  }

  /**
   * Ordena el conjunto de resultados por la columna indicada.
   * 
   * @param string $attribute el atributo por el cual se desea ordenar el resultado. No se puede ordenar por atributos relaci�n, ni por los atributos de los elementos de las relaciones o de los atributos entidad. Se puede indicar anteponiendo un espacio si es 'ASC' o 'DESC'
   * 
   * @return ORMQuery La consulta modificada como se indic�.
   */
  public function orderBy($attribute)
  {
    // Bouml preserved body begin 000CA685
	if ((func_num_args()==1) && (is_array($attribute)))
		$listOfAttributes = $attribute;
	else
		$listOfAttributes = func_get_args();
    foreach($listOfAttributes as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['sorts'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[$att]=$this->sortsCount++;
        } else {
          $this->conditions['filters']['sorts'][$attribute]=$this->sortsCount++;
        }
    }
    return $this;
    // Bouml preserved body end 000CA685
  }

  /**
   * Retorna el c�digo a ejecutar para resolver la query en formato codificado.
   * 
   * @param array $conditions las condiciones a analizar �, si es null, toma las globales
   * @param int $alias_counter un numero que se a�ade al alias de cada tabla
   * @return string el SQL codificado de la instrucci�n
   */
  public function build()
  {
    // Bouml preserved body begin 000CA705
    $db = $this->orm->getDatabase();
    $conditions = $this->conditions['filters'];
    $alias_counter = 1;
    $localFilters = "";
    $conditions['alias']="t$alias_counter";
    $conditions['alias.']=$conditions['alias'].".";
    $groups = array();
    $sorts  = array();
    $keyDef = $this->definition->getFieldDefinition('id');
	$associatedFilters = array();

    //Creo el select principal con los locals
    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $attribute => $filters) {
            $isRelation = false;
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            //Se han implementado algunas comprobaciones complejas a nivel de colección; a futuro se podrían hacer
            //más cosas sobre las colecciones:
            if ($fieldDef === false) {//Si no es un atributo directo, busco si es una relación.
                $fieldDef = $this->definition->getRelationDefinition($attribute);
                $isRelation = true;
            }

            if ($fieldDef!==false) {
                foreach($filters as $operator => $valueSets) {
                    foreach($valueSets as $valueSet) {
                        list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
						$makedCondition=null;
						//if (is_a($value, 'Attribute') || is_a($other, 'Attribute')) {
							/*
							 * TODO aca hay que hacer algo para determinar los alias.
							 * A) Si se pudieran determinar ANTES, entonces en vez de seguir un elseif volveria a poner un if y pisaría
							 * los valores de $value y/o $other con el alias.alias.field.
							 * 
							 * B) Sino trato de patear esta sentencia para más adelante y que el puntero a null nos ampare.
							 * 
							 * Uh!! se me ocurrió el "C":
							 * C) ¿que pasa si meto los valores directamente de los attributes y los voy "guardando" hasta que estan definidos?
							 * La idea es implementar un método toString() en el Attribute y aprovechar que dicho valor no va a ser "traducido"
							 * a su representación string hasta que se lo quiera concatenar con un string.
							 * De esta forma se podrían ir "completando" los alias y llegar al final con los alias puestos antes de que los
							 * usen en una consulta.
							 * Problema: igualmente sé que hay ejecuciones intermedias, por ende...
							 */
						//} 
						
						if (!$isRelation) {
                            switch ($operator) {
                                case 'BETWEEN':
                                case 'NOT BETWEEN':
                                    $makedCondition = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                                    break;
								case 'NOT IN':
								case 'IN':
									if (($value !== null) && (count($value)!= 0)) {
										if (!is_array($value)) {
											$value = array($value);
										}
										foreach($value as $innerValue) {
											$innerValues[]=$db->php2db($innerValue, $fieldDef['type']);
										}
										$makedCondition = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $innerValues, $connector);
									}
									break;
                                default:
                                    $makedCondition = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                            }//switch
                        } else {
                            switch($operator) {
                                //Implementación del IS NULL de las relaciones 1..1, 1..N, N..M lo que implica que
                                //el "id" de la entidad actual NO ESTE "IN" los "id" de la contraparte.
                                //Por ahora se aprovecha de una posibilidad de inyectar SQL (solo puede hacerlo el programador) (bajisimo riesgo).
                                case 'IS NULL':
                                    //Debería funcionar con todas las relaciones =)
                                    $makedCondition = $db->makeCondition('NOT IN', $conditions['alias.'].$keyDef['fieldName'], array('SELECT '.$fieldDef['associatedFieldName'].' FROM '.$fieldDef['table']), $connector);
                                    break;
                                default:
                                    throw new Exception("No se permite la comparación $attribute [$value, $operator, $other, $connector]");
                                    break;
                            }//switch						
                        }//if (!$isRelation)
						
						if ($makedCondition !== null) {
							if ($associationId === null) {
								$localFilters[] = $makedCondition;
							} else {
								$associatedFilters[$associationId]['joinType']=$associationConnector;
								$associatedFilters[$associationId]['conditions'][]=$makedCondition;
							}
						}
						
                    }//foreach value
                }//foreach
             } else {//If ($fieldDef!==false)
				 throw new ORMException('No se ha podido identificar el atributo "'.$attribute.'", chequee que esté bien escrito y que se haya definido correctamente en el define() de la clase '.$this->pivotClass);
			 }
        }//foreach
    }//if
	$extraAttSorts = array();
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute=>$position) {
            $attParts = explode(" ", $attribute);
            $fieldDef = $this->definition->getFieldDefinition($attParts[0]);
            $sorts[$position]=$conditions['alias.'].$fieldDef['fieldName'].((isset($attParts[1]))?' '.$attParts[1]:'');
			//Chequeo si la columna se encuentra entre las columnas a retornar
			if (isset($conditions['columns'])) {
				$attExists = false;
				foreach($conditions['columns'] as $attName) {
					if ($attParts[0] == substr($attName['column'], 0, strlen($attParts[0]))) {
						$attExists = true;
						break;
					}
				}
			} else {
				$attExists = false;
			}
			if (!$attExists)
				$extraAttSorts[$fieldDef['fieldName']] = true;
        }
    }
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

	$key_exist = false;
    if ($rewriteColumns=!isset($conditions['columns'])) {
//		if ($this->orm->getDebug())
//			krumo('ORMQuery::build() $db->sql_filter() - NoRewrite',$keyDef['table'], array($keyDef['fieldName']), $conditions['alias'], $localFilters);
		try {
			$extraAttSorts[$keyDef['fieldName']]=true;
			if (self::$debug)
				print_r($localFilters);
			$query = $db->sql_filter($keyDef['table'], array_keys($extraAttSorts), $conditions['alias'], $localFilters);
			if (self::$debug)
				print_r ($this->decrypt ($query));
		} catch (Exception $e) {
			$conditions = '';$connector = '';
			foreach ($conditions['locals'] as $attribute => $filters) {
				foreach($filters as $operator => $valueSets) {
					foreach($valueSets as $valueSet) {
						$conditions .= $connector;
						list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
						 switch ($operator) {
                                case 'BETWEEN':
                                case 'NOT BETWEEN':
                                    $conditions = " $attribute $operator $value and $other ";
                                    break;
								case 'NOT IN':
								case 'IN':
									$conditions = " $attribute $operator (".  implode(',', $value).") ";
									break;
								default:
									$conditions = " $attribute $operator $value ";
									break;
						 }
					}
				}
			}
			throw new ORMException("Error en la definición de alguno de los filtros:\n$conditions\n [".$e->getCode()."] ".$e->getMessage());
		}
//		if ($this->orm->getDebug())
//			krumo($this->decrypt ($query));
    } else {
        foreach($conditions['columns'] as $att) {
//			if ($this->orm->getDebug())
//				krumo('Atributo ',$att);
            $splitted = preg_split('/ as /i', $att['column']);
            $name = $splitted[0];
            $colDef = $this->definition->getFieldDefinition($name);
            
			if ($att['aggregate']) {
				$fieldAndAggregate = $att['aggregate'].'('.$colDef['fieldName'].')';
			} else {
				$fieldAndAggregate = $colDef['fieldName'];
			}
			
            //Si hay un alias definido (un "AS")
            if (isset($splitted[1])) {
                $colDefs[]=$fieldAndAggregate.' AS "'.$splitted[1].'"';
            } else {
                $colDefs[]=$fieldAndAggregate." AS \"$name\"";
            }
			if ($keyDef['fieldName'] == $colDef['fieldName']) {
				$key_exist = true;
			}
        }
//		if ($this->orm->getDebug())
//			krumo('ORMQuery::build() $db->sql_filter()',$keyDef['table'], $colDefs, $conditions['alias'], $localFilters);
		try {
			$query = $db->sql_filter($keyDef['table'], $colDefs, $conditions['alias'], $localFilters);
		} catch (Exception $e) {
			$conditionsTmp = '';$connector = '';
			foreach ($conditions['locals'] as $attribute => $filters) {
				foreach($filters as $operator => $valueSets) {
					foreach($valueSets as $valueSet) {
						$conditionsTmp .= $connector;
						list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
						 switch ($operator) {
                                case 'BETWEEN':
                                case 'NOT BETWEEN':
                                    $conditionsTmp = " $attribute $operator $value and $other ";
                                    break;
								case 'NOT IN':
								case 'IN':
									$conditionsTmp = " $attribute $operator (".  implode(',', $value).") ";
									break;
								default:
									$conditionsTmp = " $attribute $operator $value ";
									break;
						 }
					}
				}
			}
			throw new ORMException("Error en la definición de alguno de los filtros:\n$conditionsTmp\n o en los nombres de atributos de retorno:\n".  implode(',', $colDefs)."[".$e->getCode()."] ".$e->getMessage());
		}
//		if ($this->orm->getDebug())
//			krumo($this->decrypt ($query));
    }

	//Creo los subquerys
    $alias_counter++;
    if (isset($conditions['subquerys'])) {
        foreach($conditions['subquerys'] as $attribute => $subconditions) {
//			if ($this->orm->getDebug())
//				krumo('ORMQuery::build() build_subquery()',$query, $this->definition, $attribute, $conditions['subquerys'][$attribute], $conditions['alias'], $attribute, $alias_counter, $rewriteColumns, $associatedFilters, $sorts);
            list($query, $subSorts, $subGroups) = $this->build_subquery($query, $this->definition, $attribute, $conditions['subquerys'][$attribute], $conditions['alias'], $attribute, $alias_counter, $rewriteColumns, $associatedFilters, $sorts);
			if (self::$debug) {
				echo "$attribute => \n";
				print_r ($this->decrypt ($query));
			}
            $groups = array_merge($groups, $subGroups);
//			if ($this->orm->getDebug())
//				krumo($this->decrypt ($query));
        }
    }
	if ((count($sorts)==0) && (($key_exist && $this->uses_custom_columns) || !$this->uses_custom_columns)) { //Si no se ordena por nada, se ordena entonces por ID de la entidad principal.
		//$sorts = null;
		if (count($groups)==0) {
			$sorts[]=$conditions['alias.'].$keyDef['fieldName'];
		} else {
			$sorts=$groups;
		}
	}
    if (count($groups)==0) {
		$groups = null;
	}/* else {
        $query = $db->sql_set_columns($query, $groups, false); //Reemplazo las columnas del SELECT
    }*/

    if ($this->distinct) {
//		if ($this->orm->getDebug())
//			krumo('ORMQuery::build() $db->sql_set_columns() - Para distinct',$query, null, false, true);
        $query = $db->sql_set_columns($query, null, false, true);
		if (self::$debug)
				print_r ($this->decrypt ($query));
//		if ($this->orm->getDebug())
//			krumo($this->decrypt ($query));
    }
	
	if (count($associatedFilters)>0) {
		$start = array($db->makeCondition('(', null, null));
		$endOR   = $db->makeCondition(')', null, null, 'OR');
		$endAND  = $db->makeCondition(')', null, null, 'AND');
		foreach($associatedFilters as $key=>$filters) {
			$associatedConditions = array_merge($start, $filters['conditions']);
			if ($filters['joinType']=='OR') {
				$associatedConditions[] = $endOR;
			} else {
				$associatedConditions[] = $endAND;
			}
//			if ($this->orm->getDebug())
//				krumo('ORMQuery::build() $db->sql_add_where()',$query, $associatedConditions);
			$query = $db->sql_add_where($query, $associatedConditions);
			if (self::$debug) {
				echo "Associated $key => \n";
				print_r ($this->decrypt ($query));
			}
//			if ($this->orm->getDebug())
//				krumo($this->decrypt ($query));
		}
	}
    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA705
  }

  public function build_subquery(&$query, $pivotDefinition, $attribute, &$conditions, $parent_alias, $scrumb, &$alias_counter, &$overrideColumns, &$associatedGroups, &$sorts)
  {
    // Bouml preserved body begin 000CA885
    $db = $this->orm->getDatabase();
	/* @var $db OracleDatabase */
    $myAliasCounter = $alias_counter++;
    $conditions['alias'] = "t$myAliasCounter";
    $conditions['alias.'] = $conditions['alias'].".";
    $groups = array();
    $join_conditions = null;
	$where_conditions = null;
    $isEntity = false;
    
    //Recupero la definición de la clave primaria del pivote que me pasaron y la definición de la relación que tengo que realizar.
    //Generalmente la definición va a ser una relación pero podría ser un campo entidad, por ello se intenta una forma y si falla la otra.
    $keyDef = $pivotDefinition->getFieldDefinition('id');
    $relDefinition = $pivotDefinition->getRelationDefinition($attribute);
    if ($relDefinition === false) {
        $isEntity = true;
        $relDefinition = $pivotDefinition->getFieldDefinition($attribute);
    }

    //Recupero la definición de la clase a la que estoy apuntando y la definición de su id
    $newPivotDefinition = $this->orm->getDefinition($relDefinition['class']);
    $newPivotKeyField = $newPivotDefinition->getFieldDefinition('id');

    //Si la relación es N:M entonces tengo que enlazar los objetos a través de una tabla intermedia, reservando un alias y demás...
    //Si es 1:1 o 1:N directamente la tabla que busco es la que está en la definición de la relación previamente obtenida
    //con la salvedad de que los campos a joinear dependen de si son entidad o relación ;)
    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $NxM_alias = $myAliasCounter;
        $myAliasCounter = $alias_counter++;
        $conditions['alias'] = "t$myAliasCounter";
        $conditions['alias.'] = $conditions['alias'].".";
        $theOtherSideKey = $newPivotDefinition->getFieldDefinition('id');
        $join_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$relDefinition['associatedFieldName'], $conditions['alias.'].$theOtherSideKey['fieldName']);
    } elseif(!$isEntity) {
        $join_conditions[] = $db->makeCondition('=', $conditions['alias.']. $relDefinition['fieldName'], "$parent_alias.".$keyDef['fieldName']);
    } else {
        $join_conditions[] = $db->makeCondition('=', $conditions['alias.']. $newPivotKeyField['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
    }

    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $att => $filters) {
            $isRelation=false;
            $fieldDef = $newPivotDefinition->getFieldDefinition($att);
            //Se han implementado algunas comprobaciones complejas a nivel de colección; a futuro se podrían hacer
            //más cosas sobre las colecciones:
            if ($fieldDef === false) {//No es un atributo directo sino una relación?
                $fieldDef = $newPivotDefinition->getRelationDefinition($att);
                $isRelation = true;
            }
            foreach($filters as $operator => $valueSets) {
                foreach($valueSets as $valueSet) {
                    list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
					$makedCondition = null;
                    if (!$isRelation) {
                        switch ($operator) {
                            case 'BETWEEN':
							case 'NOT BETWEEN':
                                $makedCondition = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                                break;
							case 'IN':
							case 'NOT IN':
								if (($value !== null) && (count($value)!= 0)) {
									if (!is_array($value)) {
										$value = array($value);
									}
									foreach($value as $innerValue) {
										$innerValues[]=$db->php2db($innerValue, $fieldDef['type']);
									}
									$makedCondition = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $innerValues, $connector);
								}
								break;
                            default:
                                $makedCondition = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                        }//switch
                    } else {
                        switch($operator) {
                            case 'IS NULL':
                                ////Debería funcionar con todas las relaciones =)
//                                $otherSideId = $pivotDefinition->getFieldDefinition('id');
                                $makedCondition = $db->makeCondition('NOT IN', $conditions['alias.'].$newPivotKeyField['fieldName'], array('SELECT '.$fieldDef['associatedFieldName'].' FROM '.$fieldDef['table']), $connector);
                                break;
                            default:
                                throw new Exception("No se permite la comparación $attribute [$value, $operator, $other, $connector]");
                                break;
                        }//switch
                    }//if
					if ($makedCondition!== null) {
						if ($associationId === null) {
							$where_conditions[] = $makedCondition;
						} else {
							$associatedGroups[$associationId]['joinType']=$associationConnector;
							$associatedGroups[$associationId]['conditions'][]=$makedCondition;
						}
					}
                }//foreach
            }//foreach
        }//foreach
    }//if

    //Si hay columnas de ordenamiento dejo en $sorts los fields
	$extraAttSorts = array();
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute=>$position) {
            $attParts = explode(" ", $attribute);
            $fieldDef = $newPivotDefinition->getFieldDefinition($attParts[0]);
            $sorts[$position]=$conditions['alias.'].$fieldDef['fieldName'].((isset($attParts[1]))?' '.$attParts[1]:'');
			//Chequeo si la columna se encuentra entre las columnas a retornar
			if (isset($conditions['columns'])) {
				$attExists = false;
				foreach($conditions['columns'] as $attName) {
					if ($attParts[0] == substr($attName['column'], 0, strlen($attParts[0])))
						$attExists = true;
				}
			} else {
				$attExists = false;
			}
			if (!$attExists)
				$extraAttSorts[$fieldDef['fieldName']] = true;
        }
    }

    //Si hay groups deja los fields en $groups
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

    //Si hay subquerys llama recursivamente antes de agregar estas condiciones
    //Nota: se debe recordar que los llamados a sql_xxx() agregan las sentencias inmediatamente luego del FROM
    //por lo cual la consutrucción debe ser hecha al revés. Si quedan dudas de por qué es esto...
    //    ...inténten hacerlo de la forma "natural" y me cuentan, jejeje...
    //Nota: al final se suman los $sorts y $groups a los que devolvió el llamado recursivo.
    if (isset($conditions['subquerys']))
        foreach($conditions['subquerys'] as $att => $subconditions) {
//			if ($this->orm->getDebug())
//				krumo('ORMQuery::build_subquery() $this->build_subquery()',$query, $newPivotDefinition, $att, $conditions['subquerys'][$att], $conditions['alias'], "$scrumb.$att",$alias_counter, $overrideColumns, $associatedGroups, $sorts);

            list($query, $subSorts, $subGroups) = $this->build_subquery($query, $newPivotDefinition, $att, $conditions['subquerys'][$att], $conditions['alias'], "$scrumb.$att",$alias_counter, $overrideColumns, $associatedGroups, $sorts);
            $groups = array_merge($groups, $subGroups);
//			if ($this->orm->getDebug())
//				krumo($this->decrypt ($query));
        }

    //Finalmente si tengo que devolver ciertos atributos, los traduzco (pongo su FieldName) en la colección $colDefs
    if (isset($conditions['columns']))
        foreach($conditions['columns'] as $att) {
            $splitted = preg_split('/ as /i', $att['column']);
            $name=$splitted[0];
            $colDef = $newPivotDefinition->getFieldDefinition($name);
			
			if ($att['aggregate']) {
				$fieldAndAggregate = $att['aggregate'].'('.$colDef['fieldName'].')';//$conditions['alias.'].
			} else {
				$fieldAndAggregate = $colDef['fieldName'];//$conditions['alias.'].
			}
			
            if (isset($splitted[1])) {
                $colDefs[]=$fieldAndAggregate.' AS "'.$splitted[1].'"';
            } else {
                $colDefs[]=$fieldAndAggregate." AS \"$scrumb.$name\"";
            }
        }

    //Ahora sí, por fin, se hace el JOIN sobre la tabla de la relación, con las columnas (si existen) a devolver y
    //las condiciones especificadas por el usuario.
    //A ver... sí, es una linea sola pero ésta es LA LINEA importante!
//	krumo($join_conditions, $where_conditions);
//	if ($this->orm->getDebug())
//		krumo('ORMQuery::build_subquery() $db->sql_join_table()',$query, $newPivotKeyField['table'], $conditions['alias'], (isset($conditions['columns'])?$colDefs:null), $join_conditions, (isset($conditions['join'])?$conditions['join']:'INNER'));
	try {
		if ($this->uses_custom_columns) {
			$query = $db->sql_join_table($query, $newPivotKeyField['table'], $conditions['alias'], (isset($conditions['columns'])?$colDefs:null), $join_conditions, (isset($conditions['join'])?$conditions['join']:'INNER') );
		} else {
			/*if (isset($conditions['columns']))
				foreach($colDefs as $colTmp)
					$extraAttSorts[$colTmp]=true;*/
			if (count($extraAttSorts)==0) $extraAttSorts = null;
			$query = $db->sql_join_table($query, $newPivotKeyField['table'], $conditions['alias'], $extraAttSorts, $join_conditions, (isset($conditions['join'])?$conditions['join']:'INNER') );
		}
	} catch (Exception $e) {
		$conditionsTmp = '';$connector = '';
		foreach ($conditionsTmp['locals'] as $attribute => $filters) {
			foreach($filters as $operator => $valueSets) {
				foreach($valueSets as $valueSet) {
					$conditionsTmp .= $connector;
					list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
					 switch ($operator) {
							case 'BETWEEN':
							case 'NOT BETWEEN':
								$conditionsTmp = " $attribute $operator $value and $other ";
								break;
							case 'NOT IN':
							case 'IN':
								$conditionsTmp = " $attribute $operator (".  implode(',', $value).") ";
								break;
							default:
								$conditionsTmp = " $attribute $operator $value ";
								break;
					 }
				}
			}
		}
		throw new ORMException("Error en la definición de la relación con ".$relDefinition['class']."\n$conditionsTmp\no alguno de los filtros:\$conditionsTmp\n o en los nombres de atributos de retorno:\n".  implode(',', $colDefs)."[".$e->getCode()."] ".$e->getMessage());
	}
//	if ($this->orm->getDebug())
//		krumo($this->decrypt ($query));

	if (!is_null($where_conditions)) {
//      $debug_sql = $this->decrypt($query);
//		krumo($debug_sql[0]['SALIDA']);
//		if ($this->orm->getDebug())
//			krumo('ORMQuery::build_subquery() $db->sql_add_where()',$query, $where_conditions);
		try {
			$query = $db->sql_add_where($query, $where_conditions);
		} catch(Exception $e) {
			$conditions = '';$connector = '';
			foreach ($conditions['locals'] as $attribute => $filters) {
				foreach($filters as $operator => $valueSets) {
					foreach($valueSets as $valueSet) {
						$conditions .= $connector;
						list($value, $other, $connector, $associationId, $associationConnector) = $valueSet;
						 switch ($operator) {
								case 'BETWEEN':
								case 'NOT BETWEEN':
									$conditions = " $attribute $operator $value and $other ";
									break;
								case 'NOT IN':
								case 'IN':
									$conditions = " $attribute $operator (".  implode(',', $value).") ";
									break;
								default:
									$conditions = " $attribute $operator $value ";
									break;
						 }
					}
				}
			}
			throw new ORMException("Error en la definición de la relación con ".$relDefinition['class']."\no alguno de los filtros:\n$conditions\n o en los nombres de atributos de retorno:\n[".$e->getCode()."] ".$e->getMessage());
		}
	}
    //Y luego, si las columnas que venian hasta ahora eran para reemplazarse por cualquier definición subsiguiente,
    //chequeo el caso y sobreescribo si tengo columnas con lo cual hacerlo
    if ($overrideColumns && isset($conditions['columns'])) {
        //Reemplazo las columnas que están y pongo en false el flag para que los subsiguientes llamados no hagan lo mismo
		try {
			/* Parche para corregir error de alias en selects anidados, puesto feo xq sino segmentfaultea */
			$aliaspunto = $conditions['alias.'];
			$query = $db->sql_set_columns($query, array_map(function($elem) use ($aliaspunto) {return $aliaspunto . $elem;} , $colDefs));
		} catch (Exception $e) {
			throw new ORMException("Error al establecer los atributos a devolver de ".$relDefinition['class']."\n".  implode(',', $colDefs)."[".$e->getCode()."] ".$e->getMessage());
		}
        $overrideColumns=false;
    }

    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $NxM_conditions = array();
        $NxM_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$keyDef['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
//		if ($this->orm->getDebug())
//			krumo('ORMQuery::build_subquery() $db->sql_join_table()',$query, $relDefinition['table'], "t$NxM_alias", null, $NxM_conditions, (isset($conditions['join'])?$conditions['join']:'INNER'));
		try {
			$query = $db->sql_join_table($query, $relDefinition['table'], "t$NxM_alias", null, $NxM_conditions, (isset($conditions['join'])?$conditions['join']:'INNER'));
		} catch (Exception $e) {
			throw new ORMException("Error al vincular la relación NxM ".$pivotDefinition->getClass()."::$attribute con las instancias de la clase ".$relDefinition['class']);
		}
//		if ($this->orm->getDebug())
//			krumo($this->decrypt ($query));

    }


    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA885
  }

  /**
   * Recupera el conjunto de resultados seg�n los par�metros configurados en el el objeto.
   * 
   * @param int $from indica desde qu� elemento se deben recuperar los resultados.
   * @param int $count indica cu�ntos elementos de resultado se deben recuperar.
   * @return array Un arreglo con todos los elementos que respondieron positivamente a las condiciones indicadas.
   */
  public function find($from = null, $count = null)
  {
    // Bouml preserved body begin 000CA785
    $db = $this->orm->getDatabase();
    $f = (!is_null($from))  ? $from : $this->from;
    $c = (!is_null($count)) ? $count: $this->count;
	list ($query, $sorts, $groups) = $this->build();
	ksort($sorts);
//    if ($this->orm->getDebug()) krumo($this->decrypt ($query), $sorts, $groups);
	try {
		$result = $db->sql_execute2($query, $groups, $sorts, $f, $c, $this->maxFetchRows);
	} catch (Exception $e) {
		throw new ORMException('Error al ejecutar la consulta. Por favor, verifique que las definiciones de sus clases sean correctas (condigan con las tablas) y que los atributos involucrados en la consulta estén bien escritos');
	}
//    if ($this->orm->getDebug()) krumo($result);
    if ($result !== false) {
        if ($this->uses_custom_columns) {
            return $result;
        } else {
            $keyDef = $this->definition->getFieldDefinition('id');
			$objs=null;
            foreach($result as $res) {
                $objs[] = $this->orm->load($this->pivotClass, $res[$keyDef['fieldName']]);
            }
            return $objs;
        }
    } else {
        return false;
    }
    // Bouml preserved body end 000CA785
  }

  /**
   * Define qu� atributos recuperar del conjunto de resultados.
   * 
   * @param string $attribute un atributo a ser devueltos por la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function attributes($attribute)
  {
    // Bouml preserved body begin 000CA805
	if ((func_num_args()==1) && (is_array($attribute)))
		$listOfAttributes = $attribute;
	else
		$listOfAttributes = func_get_args();
	
	foreach($listOfAttributes as $attribute) {
		//Me fijo si es una función de agregado
		if (preg_match('/^.+\(.+\)/', $attribute)==1) {
			$_pos = strpos($attribute,'(');
			$aggregate = strtoupper(trim(substr($attribute, 0, $_pos)));
			if (!in_array($aggregate, array('COUNT', 'SUM', 'AVG', 'MIN', 'MAX'))) {
				throw new ORMException("No se reconoce la función de agregado \"$aggregate\" en la declaración $attribute de la consulta sobre ".$this->pivotClass);
			}
			$_pos2 = strpos($attribute,')');
			$_pos3 = stripos($attribute, ' AS ', $_pos2);
			$alias = substr($attribute, $_pos3);
			$attribute = substr($attribute, $_pos+1, $_pos2-$_pos-1).$alias;
		} else {
			$aggregate = false;
		}

        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['columns'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
		   $currentSlot[]=array('column'=>$att, 'aggregate'=>$aggregate);
        } else {
          $this->conditions['filters']['columns'][]=array('column'=>$attribute, 'aggregate'=>$aggregate);
        }
    }
    $this->uses_custom_columns = true;
    return $this;
    // Bouml preserved body end 000CA805
  }

  /**
   * Indica que los datos se traigan a�n en los casos donde el atributo sea null o la relaci�n no exista.
   * @param string $attribute el nombre del atributo, puede anidarse con "." y se pueden pasar varios o un array de strings tambi�n.
   * @return ORMQuery La consulta con el atributo modificado.
   */
  public function dontWorryAbout($attribute)
  {
    // Bouml preserved body begin 00136E05
	if ((func_num_args()==1) && (is_array($attribute)))
		$listOfAttributes = $attribute;
	else
		$listOfAttributes = func_get_args();
	
	foreach($listOfAttributes as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
			   if (isset($currentSlot['subquerys']) && isset($currentSlot['subquerys'][$att])) {
					$currentSlot = &$currentSlot['subquerys'][$att];
			   } else {
				   throw new ORMException("ORMQuery: Error, se ha llamado a dontWorryAbout($attribute) pero no se detectó que exista $att como relación válida.");
			   }
           }
           $currentSlot['join']='LEFT';
        } else {
          $this->conditions['filters']['subquerys'][$attribute]['join']='LEFT';
        }
    }
    return $this;
    // Bouml preserved body end 00136E05
  }

  /**
   * Anula una orden dada previamente con dontWorryAbout().
   * @param string $attribute el nombre del atributo, puede anidarse con "." y se pueden pasar varios o un array de strings tambi�n.
   * @return ORMQuery La consulta con el atributo modificado.
   */
  public function worryAbout($attribute)
  {
    // Bouml preserved body begin 00142D85
	if ((func_num_args()==1) && (is_array($attribute)))
		$listOfAttributes = $attribute;
	else
		$listOfAttributes = func_get_args();
	foreach($listOfAttributes as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
			   if (isset($currentSlot['subquerys']) && isset($currentSlot['subquerys'][$att])) {
					$currentSlot = &$currentSlot['subquerys'][$att];
			   } else {
				   throw new ORMException("ORMQuery: Error, se ha llamado a worryAbout($attribute) pero no se detectó que exista $att como relación válida.");
			   }
           }
           $currentSlot['join']='INNER';
        } else {
          $this->conditions['filters']['subquerys'][$attribute]['join']='INNER';
        }
    }
    return $this;
    // Bouml preserved body end 00142D85
  }

  /**
   * Define qu� atributos deben ser agrupados una vez obtenido el conjunto de resultados masivo.
   * 
   * @param string $attribute un atributo por el cual se debe agrupar el resultado de la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function groupBy($attribute)
  {
    // Bouml preserved body begin 000CC285
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['groups'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['groups'][]=$attribute;
        }
    }
    $this->uses_groups = true;
    $this->uses_custom_columns = true; //Esto se pone porque si se agrupa por X criterio no tiene sentido devolver objetos: ¿cuál devuelvo?
    return $this;
    // Bouml preserved body end 000CC285
  }

  /**
   * Retorna la cantidad de resultados del query.
   * 
   * @return int La cantidad de filas obtenidas por la consulta.
   */
  public function count()
  {
    // Bouml preserved body begin 000CDD05
    $db = $this->orm->getDatabase();
    list ($query, $sorts, $groups) = $this->build();
    return $db->count($query);
    // Bouml preserved body end 000CDD05
  }

  /**
   * Indica si se desea filtrar resultados distintos.
   * 
   * @param booolean $enable un valor TRUE para indicar que haga la selecci�n filtrando valores repetidos y FALSE para que traiga todo.
   * 
   * @return ORMQuery la consulta con el cambio aplicado
   */
  public function distinct($enable = true)
  {
    // Bouml preserved body begin 000DE485
    $this->distinct=$enable;
    return $this;
    // Bouml preserved body end 000DE485
  }

  /**
   * Recupera el n�mero m�ximo n�mero de filas simult�neas que se recuperan por llamado a la base de datos.
   * 
   * @return int La cantidad de filas que se recuperan de una vez.
   */
  final public function getMaxFetchRows()
  {
    return $this->maxFetchRows;
  }

  /**
   * Configura la cantidad de filas que se pueden traer en simult�neo en un s�lo llamaod a fetch de la base de datos.
   * 
   * @param int $value el n�mero de filas m�ximo a recuperar.
   * 
   * @return ORMQuery la consulta modificada.
   */
  public function setMaxFetchRows($value)
  {
    $this->maxFetchRows = $value;
    return $this;
  }

  /**
   * Involucra una nueva clase en la consulta. Es obligatorio especificarle el alias.
   * 
   * @param string $class el nombre de la clase contra la cual se quieren relacionar las entidades
   * @param string $alias un nombre genérico para referenciar tal conjunto de entidades.
   * 
   * @return ORMQuery la consulta con el nuevo conjunto de entidades vinculado.
   */
  public function involve($class, $alias)
  {
    // Bouml preserved body begin 0017EB85
    // Bouml preserved body end 0017EB85
  }

  /**
   * Configura un valor predeterminado en el caso que el atributo devuelto sea null. Ésta función sólo puede ser utilizada en conjunto con ->attributes(), de lo contrario devolverá un error (es decir, no pisa valores de objetos, sólo establece valores predeterminados en consultas planas, sino se rompería la consistencia del ORM).
   * 
   * @param string $attribute el atributo del cual se quiere verificar el valor
   * @param string $value el valor predeterminado. Por ahora es un string que se pondrá fijo.
   * 
   * @return ORMQuery La consulta con la nueva cláusula agregada.
   */
  public function isNull($attribute, $value)
  {
    // Bouml preserved body begin 001A0185
    // Bouml preserved body end 001A0185
  }

  /**
   * Imprime la consulta SQL de la query.
   * 
   * @return ORMQuery la misma query
   */
  public function debugQuery()
  {
    // Bouml preserved body begin 001B2605
	echo '<pre>'.Utils::debugQuery($this).'</pre>';
	return $this;
    // Bouml preserved body end 001B2605
  }

}
?>
