<?php
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORM_TYPES.php';
require_once 'iplan/orm/ORM_LOAD_STYLE.php';
require_once 'iplan/orm/ORM_RELATION_TYPE.php';
require_once 'iplan/orm/PrivateAccesor.php';
require_once 'iplan/orm/exceptions/ORMDefinitionError.php';
require_once 'iplan/orm/exceptions/UnknowAttribute.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Hay un array que almacena la forma de relacionar los atributos de una clase con los de la/s tablas con las que se corresponde:
 * ORMData['fields'][$attribute]= {'fieldName', 'type',
 *                                                     'length', 'isNulleable', 'precision', 'primaryKey', 'default', 'table', 'loadStyle'}
 *                si type = ORMDefinition::ORM_ENTITY  'class', 'isNulleable', 'default', 'table, 'loadStyle'}
 * ORMData['fieldsByTable'][$table][$attribute]->&ORMData['fields'][$attribute]
 * ORMData['relations'][$attribute]= {'fieldName', 'type', 'class', 'table', 'loadStyle', 'associatedFieldName'}
 * 
 *  Ms adelante puede implementarse una versin ms objetosa haciendo que haya cosas como DBTable y DBField
 * 
 * Hay 3 diferentes tipos de definiciones:
 *  - Campos simples: son atributos que representan un valor puntual como int, string, date.
 *  - Entidades: es la materializacin de la clave fornea en la entidad "hijo". Es decir cuando un id de otra tabla aparece en la tabla actual. Ese "id" no es un atributo comn, sino que representa a toda una entidad entera asociada por su id. Son relaciones 0..1 o 1..1
 *  - Relaciones: parecido al caso anterior pero para relaciones 1..N y N..M. El atributo en particular se implementa como un "array" de instancias de la clase asociada. Para el caso N..M la tabla se asume que tiene una clave conformada por los 2 IDs que se relacionan.
 * 
 */
class ORMDefinition extends PrivateAccesor implements ORM_TYPES, ORM_LOAD_STYLE, ORM_RELATION_TYPE {
  protected $ORMData;

  private $currentTable;

  /**
   * @var ORM la instancia del ORM al que pertenece la definición
   */
  private $orm;

  /**
   * @var string el nombre de la clase que se est mapeando
   */
  private $class;

  /**
   * @var string el lenguaje predeterminado de las definiciones
   */
  private $language = 'EN';

  /**
   * @var array una caché que se va formando en la medida que a la clase se le consultan los plurales de atributos en singular de los métodos add/del
   */
  private $plurals = array();

  /**
   * Constructor de la clase. Prepara las estructuras básicas necesarias para el mapeo.
   * 
   * @param ORM $orm el objeto que maneja las definiciones, de aquí también se saca el lenguaje predeterminado.
   * 
   * @return ORMDefinition una instancia inicializada de ORMDefinition
   */
  public function __construct($orm = null)
  {
    // Bouml preserved body begin 00084705
    $this->ORMData['fields']=null;
    $this->ORMData['fieldsByTable']=null;
    $this->ORMData['relations']=null;
    $this->orm=$orm;
	if ($orm !== null) {
		$this->language = $orm->getLanguage();
	}
    return $this;
    // Bouml preserved body end 00084705
  }

  /**
   * Establece una tabla objetivo para futuros llamados a addField(), addInstance() y addRelation().
   * 
   * @param string $table el nombre de la tabla
   * @return ORMDefinition la propia definicin modificada
   * 
   * @see ORMDefinition::addField()
   * @see ORMDefinition::addInstance()
   * @see ORMDefinition::addRelation()
   */
  public function setTable($tableName)
  {
    // Bouml preserved body begin 00043285
    $this->currentTable = $tableName;
    return $this;
    // Bouml preserved body end 00043285
  }

  /**
   * Indica como traducir un atributo del objeto en su campo respectivo de la base de datos.
   * 
   * @param string $attribute el nombre del atributo que queremos definir (el del objeto)
   * @param string $fieldName el nombre del campo de la base de datos donde se almacena el valor
   * @param int $type el tipo de valor, las constantes se obtienen de ORM_TYPES. En el caso que el atributo sea un objeto entero, usar @see addInstance().
   * @param int $length la longitud del campo de la base de datos. Si se quiere omitir esto indicar -1
   * @param int $precision en el caso de numricos se debe indicar la precisin para futuros redondeos. Para omitir -1
   * @param boolean $isNulleable indica si el atributo/campo puede adquirir valor NULL
   * @param string $defaultValue el valor predeterminado del campo
   * @param int $loadStyle se puede indicar si se desea que el atributo sea recuperado instantneamente al cargar el objeto o si se prefiere diferir la carga hasta que el valor del atributo sea utilizado por primera vez. Recomendable usar esta ltima opcin si el campo es una imagen o un algn valor pesado. Las constantes estn en @see ORM_LOAD_STYLE.
   * @param string $table el nombde de la tabla donde se encuentra el atributo, generalmente se especifica el mismo con un llamado previo a @see setTable() y se asume ese valor dejando ""
   * @param boolean $primaryKey poner true si el campo es la clave primaria del objeto
   * 
   * @return ORMDefinition Retorna la propia definicin modificada, para seguirla manipulando en cadena si se desea.
   */
  public function addField($attribute, $fieldName, $type, $length = -1, $precision = -1, $isNulleable = true, $defaultValue = "", $loadStyle = ORMDefinition::FORCE_LOAD, $table = "", $primaryKey = false)
  {
    // Bouml preserved body begin 00026305
    if ($table == '') {
        if ($this->currentTable!= '') {
            $table = $this->currentTable;
        } else {
            throw new ORMDefinitionError("Error al definir el atributo '$attribute': no hay tabla especificada");
        }
    }
    $this->ORMData['fields'][$attribute]=array('fieldName'=>$fieldName, 'type'=>$type, 'length'=>$length, 'isNulleable'=>$isNulleable, 'precision'=>$precision, 'primaryKey'=>$primaryKey,'default'=>$defaultValue , 'table'=>$table, 'loadStyle'=>$loadStyle, 'language'=>$this->language);
    $this->ORMData['fieldsByTable'][$table][$attribute]=&$this->ORMData['fields'][$attribute];

    switch($type) {
        case ORM_TYPES::BIGINT:
        case ORM_TYPES::INTEGER:
        case ORM_TYPES::SMALLINT:
			$validado = false;
			if (($length !== null) && is_int($length) && ($length > -1)) {
				$this->addValidation ($attribute, "precision $length");
				$validado = true;
			}
			if (($precision !== null) && is_int($precision) && ($precision > -1)) {
				$this->addValidation ($attribute, "scale $precision");
				$validado = true;
			}
			if (!$validado)
				$this->addValidation($attribute, 'numeric');
            break;
        case ORM_TYPES::DECIMAL:
			if (($length !== null) && is_int($length) && ($length > -1)) {
				$this->addValidation ($attribute, "precision $length");
				$validado = true;
			}
			if (($precision !== null) && is_int($precision) && ($precision > -1)) {
				$this->addValidation ($attribute, "scale $precision");
				$validado = true;
			}
			if (!$validado)
				$this->addValidation($attribute, 'numeric');
            break;
        case ORM_TYPES::STRING:
            $this->addValidation($attribute, 'text');
            if ($length != -1) $this->addValidation ($attribute, "max $length");
            break;
    }
    if ($isNulleable === false) $this->addValidation($attribute, 'required');
    return $this;
    // Bouml preserved body end 00026305
  }

  /**
   * Define un atributo como una instancia de otra clase.
   * 
   * @param string $attribute el nombre del atributo que queremos definir (el del objeto)
   * @param string $fieldName el nombre del campo de la base de datos donde se almacena la clave que identifica la instancia puntal de la otra clase.
   * @param string $class la clase del objeto que va en el atributo.
   * @param boolean $isNulleable indica si el atributo puede adquirir valor NULL
   * @param string $defaultValue el valor predeterminado del campo
   * @param int $loadStyle se puede indicar si se desea que el atributo sea recuperado instantneamente al cargar el objeto o si se prefiere diferir la carga hasta que el valor del atributo sea utilizado por primera vez. Recomendable usar esta ltima opcin si el campo es una imagen o un algn valor pesado. Las constantes estn en @see ORM_LOAD_STYLE.
   * @param string $table el nombde de la tabla donde se encuentra el atributo, generalmente se especifica el mismo con un llamado previo a @see setTable() y se asume ese valor dejando ""
   * @param boolean $primaryKey poner true si el campo es la clave primaria del objeto
   * 
   * @return ORMDefinition Retorna la propia definicin modificada, para seguirla manipulando en cadena si se desea.
   */
  public function addInstance($attribute, $fieldName, $class, $isNulleable = true, $defaultValue = null, $loadStyle = ORMDefinition::LAZY_LOAD, $table = "")
  {
    // Bouml preserved body begin 0006EF85
    if ($table == '') {
        if ($this->currentTable!= '') {
            $table = $this->currentTable;
        } else {
            throw new ORMDefinitionError("Error al definir el atributo(instancia) '$attribute': no hay tabla especificada");
        }
    }
	if ($isNulleable === false) $this->addValidation($attribute, 'required');
    $this->ORMData['fields'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORMDefinition::ORM_ENTITY, 'class'=>$class, 'isNulleable'=>$isNulleable, 'default'=>$defaultValue , 'table'=>$table, 'loadStyle'=>$loadStyle, 'language'=>$this->language);
    $this->ORMData['fieldsByTable'][$table][$attribute]=&$this->ORMData['fields'][$attribute];
    return $this;
    // Bouml preserved body end 0006EF85
  }

  /**
   * Agrega un discriminador para identificar la clase.
   */
  public function addDiscriminator($fieldname, $value, $type, $table)
  {
    // Bouml preserved body begin 0018DC85
    // Bouml preserved body end 0018DC85
  }

  /**
   * Retorna el registro crudo de los campos, tal como se le dieron de alta.
   * @param string $attribute el nombre del atributo del cual queremos recibir la informacin de mapeo o nada para obtener todas las definiciones
   * @param boolean $findByField indica si el atributo pasado por parmetro es en realidad el nombre del campo de la base de datos y lo busca por dicho criterio.
   * 
   * @return array|false Retorna las definiciones en un arreglo de la forma o false si el atributo no existe:
   * [$attribute] = {'fieldName', 'type', 'length', 'isNulleable', 'precision', 'primaryKey', 'default', 'table', 'loadStyle'}
   */
  public function getFieldDefinition($attribute = "", $findByField = false)
  {
    // Bouml preserved body begin 00063305
    if ($attribute != "") {
        if ($findByField) {
            foreach($this->ORMData['fields'] as $attribute2=>$definition)
                 if ($definition['fieldName']==$attribute)
                     return array_merge($definition, array('attribute'=>$attribute2));
        } else {
            if (isset($this->ORMData['fields'][$attribute]))
                 return $this->ORMData['fields'][$attribute];
            else return false;
        }
    } else {
        return $this->ORMData['fields'];
    }
    // Bouml preserved body end 00063305
  }

  /**
   * Retorna la definicin de una relacin.
   * 
   * @param string $attribute el nombre del atributo cuyo mapeo se quiere obtener o nada para recuperar todas las definiciones de relaciones.
   * @param boolean $findByTable indica si el atributo pasado por parmetro es en realidad el nombre del campo de la base de datos y lo busca por dicho criterio.
   * 
   * @return array un arreglo con la informacin de la/s definicin/es con el siguiente formato o false si la relacin no existe:
   * [$attribute] = {'type', 'class', 'table', 'loadStyle'}
   * 
   * En el caso que se busque por tabla retorna la primer relacin definida que acte sobre la tabla indicada. Esta funcin est orientada a detectar el atributo que mapea las relaciones MxN sobre una tabla dada. NO SIRVE para las relaciones 1:1 y 1:N.
   */
  public function getRelationDefinition($attribute = "", $findByTable = false)
  {
    // Bouml preserved body begin 00081205
    if ($attribute != "") {
        if ($findByTable) {
			if (is_array($this->ORMData['relations']) && (count($this->ORMData['relations']) > 0)) {
				foreach($this->ORMData['relations'] as $attributeName=>$definition) {
					if ($definition['table']==$attribute) {
						return array_merge($definition, array('attribute'=>$attributeName));
					}
				}
			}
        } else {
            if (isset($this->ORMData['relations'][$attribute]))
                 return $this->ORMData['relations'][$attribute];
            else return false;
        }
    } else {
        return $this->ORMData['relations'];
    }
    // Bouml preserved body end 00081205
  }

  /**
   * Recupera los discriminadores aplicados a la entidad. Si se pasa un nombre de campo (OJO, NO ES UN ATRIBUTO) devuelve uno específico, sino recupera todos.
   * 
   * @param string $fieldname el campo del discriminador que se desea conocer o nada si se desean obtener todos.
   * 
   * @return array un arreglo con la definición del discriminador o un arreglo de definiciones.
   */
  public function getDiscriminatorDefinition($fieldname = '')
  {
    // Bouml preserved body begin 0018DD05
    // Bouml preserved body end 0018DD05
  }

  /**
   * Retorna la informacin de los campos que pertenecen a una tabla o a todas las tablas de la definicin si se pasa ""
   */
  public function getFieldsByTable($table = "")
  {
    // Bouml preserved body begin 00061905
      if ($table=="") {
          return $this->ORMData['fieldsByTable'];
      } else {
          return $this->ORMData['fieldsByTable'][$table];
      }
    // Bouml preserved body end 00061905
  }

  /**
   * Retorna la lista de tablas que participan de la definicin.
   */
  public function getTableList()
  {
    // Bouml preserved body begin 00061805
    return array_keys($this->ORMData['fieldsByTable']);
    // Bouml preserved body end 00061805
  }

  /**
   * Configura la clase a la que se corresponde la definicin
   * 
   * @param string $class el nombre de la clase que se est definiendo
   * 
   * @return ORMDefinition la definicin modificada.
   */
  public function setClass($class)
  {
    // Bouml preserved body begin 000DFE05
    $this->class = $class;
    return $this;
    // Bouml preserved body end 000DFE05
  }

  /**
   * Establece el atributo indicado como clave del objeto. Tambin se le puede pasar un arreglo de atributos.
   * Es igual que indicar en addField() isNulleable en FALSE y primaryKey en TRUE. Se da esta funcin para facilitar su definicin y mover dichos atributos al final de los parmetros en addField().
   * 
   * @param string|array $attribute el atributo o un array de atributos que sern identificados como clave primaria del objeto.
   * 
   * @return ORMDefinition La propia definicin modificada. IMPORTANTE: llamar a esta funcin LUEGO de haber definido el mapeo del atributo.
   */
  public function setKey($attribute)
  {
    // Bouml preserved body begin 00086285
	$attributes = $attribute;
	if (!is_array($attribute)) {
		$attributes = array($attribute);
	} 
	foreach ($attributes as $att) {
		if (isset ($this->ORMData['fields'][$att])) {
			$this->ORMData['fields'][$att]['isNulleable']=false;
			$this->ORMData['fields'][$att]['primaryKey']=true;
			$this->ORMData['keys'][]=$att;
		} else {
			throw new ORMDefinitionError("Error en ORMDefinition::setKey()\nSe intentó establecer como clave primaria el atributo $this->class::$att, pero no está previamente definido. Use ->addField($att,...) antes de invocar a esta función.");
		}
	}
    return $this;
    // Bouml preserved body end 00086285
  }

  /**
   * Configura el lenguaje predeterminado de las definiciones subsecuentes.
   * 
   * @param string $language el idioma que se va a emplear en posteriores definiciones.
   * 
   * @return ORMDefinition La definición con el lenguaje establecido.
   */
  public function setLanguage($language)
  {
    // Bouml preserved body begin 0018C305
	$this->language = $language;
	return $this;
    // Bouml preserved body end 0018C305
  }

  /**
   * @todo METODO PARA BORRAR!!! ES PARA DEBUG NOMAS
   */
  public function getORMData()
  {
    // Bouml preserved body begin 00084885
    return $this->ORMData;
    // Bouml preserved body end 00084885
  }

  /**
   * Indica que hay una relacin 1:N entre esta entidad y las instancias de la otra clase. En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr el array de instancias
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param int $loadStyle determina si la relacin se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta ltima es la accin predeterminada). Las constantes estn en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definicin
   */
  public function addRelation1xN($attribute, $fieldName, $class, $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087E05
    //Manejo la recursión primero
    if ($this->class == $class) {
        $fieldId = $this->getFieldDefinition('id');
    } else {  
        if (is_null($this->orm)) {
            $params = array();
            $definition = $this->invokeStaticMethod($class, 'define', $params);
			if (!is_a($definition, 'ORMDefinition')) throw new ORMException("Atención: '$class::define()' no retorna un valor correcto, verifiquelo.");
            $fieldId = $definition->getFieldDefinition('id');
        } else {
            $definition = $this->orm->getDefinition($class);
			if (!is_a($definition, 'ORMDefinition')) throw new ORMException("Atención: '$class::define()' no retorna un valor correcto, verifiquelo.");
            $fieldId = $definition->getFieldDefinition('id');
        }
    }
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::OneToMany, 'class'=>$class, 'loadStyle'=>$loadStyle, 'table'=>$fieldId['table'], 'associatedFieldName'=>$fieldName, 'language'=>$this->language);
    return $this;
    // Bouml preserved body end 00087E05
  }

  /**
   * Indica que hay una relacin 1:1 entre esta entidad y la instancia de la otra clase. En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr el array de instancias
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param int $loadStyle determina si la relacin se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta ltima es la accin predeterminada). Las constantes estn en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definicin
   */
  public function addRelation1x1($attribute, $fieldName, $class, $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087E85
    //Manejo la recursión primero
    if ($this->class == $class) {
        $fieldId = $this->getFieldDefinition('id');
    } else {
        if (is_null($this->orm)) {
            $params = array();
            $definition = $this->invokeStaticMethod($class, 'define', $params);
			if (!is_a($definition, 'ORMDefinition')) throw new ORMException("Atención: '$class::define()' no retorna un valor correcto, verifiquelo.");
            $fieldId = $definition->getFieldDefinition('id');
        } else {
            $definition = $this->orm->getDefinition($class);
			if (!is_a($definition, 'ORMDefinition')) throw new ORMException("Atención: '$class::define()' no retorna un valor correcto, verifiquelo.");
            $fieldId = $definition->getFieldDefinition('id');
        }
    }
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::OneToOne, 'class'=>$class, 'loadStyle'=>$loadStyle, 'table'=>$fieldId['table'], 'associatedFieldName'=>$fieldName, 'language'=>$this->language);
    return $this;
    // Bouml preserved body end 00087E85
  }

  /**
   * Indica que hay una relacin entre esta entidad y las instancias de otra clase. Las relaciones que soportan son 1:N, N:M y 1:1. En el caso de la N:M debe indicarse la tabla intermedia.
   * En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr el array de instancias
   * @param string $table slo se usa en relaciones N:M e indica el nombre de la tabla intermedia
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param string $asociatedFieldName el nombre del atributo donde se mapea el identificador de la clase asociada
   * @param int $loadStyle determina si la relacin se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta ltima es la accin predeterminada). Las constantes estn en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definicin
   */
  public function addRelationNxM($attribute, $table, $fieldName, $class, $asociatedFieldName = "", $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087F05
    if ($asociatedFieldName=="") {
        //Manejo primero la recursion
        if ($this->class == $class) {
            $relatedDefinition = $this->getFieldDefinition('id');
        } else {
            if (isset ($this->orm)) {
                $relatedDefinition= $this->orm->getDefinition($class)->getFieldDefinition('id');
            } else {
                //Este moco es para "compatibilizar que el PHP no soporte el "$class::" para invocar métodos estáticos
                $params = array();
                $relatedDefinition= $this->invokeStaticMethod($class, "define", $params)->getFieldDefinition('id');
                //$relatedDefinition= $class::define()->getFieldDefinition('id');No es compatible con PHP del server
            }
        }
        $asociatedFieldName=$relatedDefinition['fieldName'];
    }
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::ManyToMany, 'class'=>$class, 'table'=>$table, 'loadStyle'=>$loadStyle, 'associatedFieldName'=>$asociatedFieldName, 'language'=>$this->language);
    return $this;
    // Bouml preserved body end 00087F05
  }

  /**
   * Instruye al ORM para que valide el valor del atributo antes de guardarlo.
   * 
   * @param string $attribute el nombre del atributo. Debe haber sido previamente definido.
   * @param string $validations una lista de strings con las versiones textuales de la validación a efectuar. Las mismas pueden ser: <ul><li><b>required</b>, indica que el campo es requerido</li>
   * <li><b>alpha</b>, el valor debe contener sólo letras y espacios</li>
   * <li><b>numeric</b>, se verifica que el valor sea numérico entero (soporta +/-)</li>
   * <li><b>alphanumeric</b>, combina numeric con alpha</li>
   * <li><b>mail</b>, verifica que el valor sea una dirección de mail correctamente escrita</li>
   * <li><b>text</b>, cualquier cosa valida este filtro</li>
   * <li><b>regexp</b>, verifica que el valor cumpla una expresión regular dada</li>
   * <li><b>min</b>, chequea que el valor no sea menor al número dado</li>
   * <li><b>max</b>, ídem pero que no sea mayor</li>
   * <li><b>float</b>, valida números enteros y/o decimales (soporta +/- y . o , para los decimales)</li>
   * <li><b>precision</b>, se usa en combinación con "float" para indicar que la parte entera no exceda de una cierta cantidad de dígitos</li>
   * <li><b>scale</b>, se usa en combinación con "float" para indicar que la parte decimal no exceda de una cierta cantidad de dígitos</li>
   * <li><b>mac</b>, chequea que el texto corresponda a una MAC Address (por compatibilidad con 10Fold es posible no usar separación entre los pares)</li>
   * <li><b>ip</b>, valida que el texto se corresponda con una IP válida</li>
   * <li><b>url</b>, verifica que el texto sea una url. Soporta: protocolo+usuario+subdominio+dominio+puerto+directorio+query+anchor</li>
   * <li><b>maxlength</b>, cuenta los caracteres de un texto para que no exceda cierto valor</li>
   * </ul>
   * 
   * @return ORMDefinition La definición con la validación agregada.
   * 
   * Modo de uso:
   * <code>
   * $definition->addValidation('atributo', 'required');//Agrega una sola validación simple
   * $definition->addValidation('atributo', 'required', 'alphanumeric', 'url');//Agrega varias validaciones simples a un atributo
   * $defintion->addValidation('atributo', 'min 5', 'max 10');//Agrega validaciones complejas
   * </code>
   */
  public function addValidation($attribute, $validations)
  {
    // Bouml preserved body begin 000E1785

    for($i=1; $i < func_num_args(); $i++) {
        $validation = func_get_arg($i);
        $space = strpos($validation,' ');
        //Si no hay espacio es un filtro preestablecido
        if ($space!==FALSE) {
            $expression = substr($validation, $space+1);
            $validation = substr($validation, 0, $space);
        } else {
            $expression = "";
        }
        if (in_array(strtolower($validation), Validation::getRegisteredValidations())) {
            $this->ORMData['validations'][$attribute][$validation][]=$expression;
        } else {
            throw new ORMDefinitionError("La validación [$validation] para el atributo $attribute es desconocida");
        }
    }
	return $this;
    // Bouml preserved body end 000E1785
  }

  /**
   * Recupera las validaciones que se realizan sobre el atributo indicado , si no se especifica dicho atributo, recupera todas las validaciones.
   * 
   * @param string $attribute El atributo del cual se desean recuperar las validaciones
   * 
   * @return array El arreglo con las validaciones. Dicho arreglo tiene la forma de
   * [validacion] => array {
   *                    0 => expresion 1
   *                    1 => expresion 2
   *                    }
   *  (si se omite indicar el atributo)
   * [atributo][validacion] => array {
   *                    0 => expresion 1
   *                    1 => expresion 2
   *                    }
   */
  public function getValidation($attribute = null)
  {
    // Bouml preserved body begin 000E1D85
	if (isset($this->ORMData['validations'])) {
		if (!is_null($attribute)) {
			if(isset($this->ORMData['validations'][$attribute])) {
				return $this->ORMData['validations'][$attribute];
			} else {
				return false;
			}
		} else {
			return $this->ORMData['validations'];
		}
	} else {
		return false;
	}
    // Bouml preserved body end 000E1D85
  }

  /**
   * Toma un atributo y elabora su plural en base a un idioma de referencia.
   * 
   * @param string $attribute el nombre del que deseamos obtener el plural
   * 
   * @return string el nombre de atributo pluralizado.
   */
  public function pluralize($attribute)
  {
    // Bouml preserved body begin 001AD305
	if (isset($this->plurals[$attribute])) return $this->plurals[$attribute];
	
	$languages = array('EN', 'ES');
	foreach($languages as $language) {
		$att2 = self::pluralize2($attribute, $language);
		if (isset($this->ORMData['relations'][$att2]) && ($this->ORMData['relations'][$att2]['language']==$language)) {
			return $att2;
		}
	}
	throw new ORMException("No se pudo encontrar un plural para el atributo $this->class::$attribute. Verifique que el idioma esté bien configurado en $this->class::define()");
    // Bouml preserved body end 001AD305
  }

  protected static function pluralize2($attribute, $language)
  {
    // Bouml preserved body begin 001B0C85
	switch ($language) {
		case 'EN':
				//Veo si es irregular: http://oyoko.org/English_plural_exceptions
				$irregulars = array('man'=>'men', 'woman'=>'women', 'child'=>'children', 'mouse'=>'mice', 'goose'=>'geese',
									'ox'=>'oxen', 'tooth'=>'teeth', 'foot'=>'feet', 'deer'=>'deer', 'sheep'=>'sheep');
				if (isset($irregulars[$attribute])) return $irregulars[$attribute];


				//Ver reglas en http://www.aprende-gratis.com/ingles/curso.php?lec=plural
				$lastTwo = substr($attribute, strlen($attribute)-2);
				$last = $lastTwo[1];

				//Si termina en s, x, ch o sh se agrega "es"
				//Lo mismo si termina en "o"
				if (in_array($lastTwo, array("sh", "ch")) || in_array($last, array('s', 'z', 'x')) || ($last == 'o')) {
					return $attribute . "es";
				}

				//Si termina en f o fe se reemplaza por v y se agrega es
				if ($last=='f') return substr($attribute, 0, strlen($attribute)-1) . "ves";
				if ($lastTwo=='fe') return substr($attribute, 0, strlen($attribute)-2) . "ves";


				//Si termina en vocal o y se agrega "s", salvo que antes de la y haya una consonante
				//en cuyo caso se reemplaza la y por ies
				$vowels = array('a', 'e', 'i', 'o', 'u');
				if (in_array($last, $vowels)) {
					return $attribute . "s";
				}
				if (($last == 'y') && (!in_array($lastTwo[0], $vowels))) {
					return substr($attribute, 0, strlen($attribute)-1) . "ies";
				}

				//En forma predeterminada agrego una "s" al final
				return $attribute . "s";
			break;
		case 'ES':
			$last = substr($attribute, strlen($attribute)-1);
			$vowels = array('a', 'e', 'i', 'o', 'u');

			if (in_array($last, $vowels)) {
				return $attribute."s";
			} else {
				return $attribute."es";
			}

			break;
		default:
			throw new Exception("Error: no se ha especificado un lenguaje válido al intentar pluralizar $this->class::$attribute");
	}
    // Bouml preserved body end 001B0C85
  }

}
?>