<?php
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORMObject.php';
require_once 'iplan/orm/ORMQuery.php';
require_once 'iplan/orm/ORMQueryMTWrapper.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
class MigrationToolkit {
  /**
   * @const string una constante que simboliza que la clase está pendiente de migración
   */
  const PENDING = 'PENDING';

  /**
   * @const string una constante que simboliza que la clase está en proceso de migración
   */
  const WORKING = 'WORKING';

  /**
   * @const string una constante que simboliza que la clase  ya ha sido migrada
   */
  const MIGRATED = 'MIGRATED';

  /**
   * @const string constante que identifica que una operación se realizó con éxito
   */
  const ST_MIGRATED = 'MIGRATED';

  /**
   * @const string constante que identifica que un atributo esta expliscitamente excluído
   */
  const ST_EXCLUDED = 'EXCLUDED';

  /**
   * @const string constante que identifica que un atributo ha sido ignorado dado que no se contaba con la definición
   */
  const ST_IGNORED = 'IGNORED';

  /**
   * @const string constante que identifica que no se pudo encontrar un objeto equivalente
   */
  const ST_NOTFOUND = 'NOTFOUND';

  /**
   * @const string constante que identifica un estado erroneo por causas desconocidas
   */
  const ST_FAILED = 'FAILED';

  /**
   * @const string constante que identifica que un objeto tiene muchas instancias equivalentes en el destino. Seguramente la clave candidata esté mal expresada
   */
  const ST_MANY = 'MANY';

  /**
   * @const string constante que identifica que un objeto fue expliscitamente intercambiado por las definiciones del usuario
   */
  const ST_EXCHANGED = 'EXCHANGED';

  /**
   * @const string constante que identifica que se encontró un objeto equivalente
   */
  const ST_FOUND = 'FOUND';

  /**
   * @var ORM el ORM origen sobre el cual se realizará la búsqueda de entidades a migrar
   */
  private $from;

  /**
   * @var ORM el ORM destino hacia el cual se pasarán o actualizarán las entidades encontradas en el origen
   */
  private $to;

  /**
   * @var array una matriz multidimensional donde se guardan las configuraciones
   */
  private $config;

  /**
   * @var DOMDocument la instancia de log
   */
  private $log;

  /**
   * @var boolean un flag que indica si se quiere imprimir mensajes de debug o no
   */
  private $debug = false;

  /**
   * @const string constante que indica el modo de previsualización de cambios
   */
  const PREVIEW_MODE = 'PREVIEW_MODE';

  /**
   * @const string constante que indica el modo de migración normal
   */
  const NORMAL_MODE = 'NORMAL_MODE';

  /**
   * @var string el modo en el cual se ejecuta la migración, actualmente hay dos modos: PREVIEW_MODE que genera un XML sin realizar los cambios en la base destino y EXPORT_MODE
   */
  private $mode = self::PREVIEW_MODE;

  public function __construct()
  {
    // Bouml preserved body begin 001D3E85
    // Bouml preserved body end 001D3E85
  }

  /**
   * Crea un nuevo elemento y lo agrega al documento o al parent indicado.
   * 
   * @param string $name el nombre del elemento
   * @param string $value el valor del elemento
   * @param DOMElement $parent en caso de indicarse, se agrega el elemento a este parent
   * 
   * @return DOMElement Retorna el elemento recién creado
   */
  public function addElement($name, $value = null, $parent = null)
  {
    // Bouml preserved body begin 001D3F05
	if ($parent === null)
		$parent = $this->log;
	if ($value) {
		$parent->appendChild($child = $this->log->createElement($name, $value));
	} else {
		$parent->appendChild($child = $this->log->createElement($name));
	}
	return $child;
    // Bouml preserved body end 001D3F05
  }

  /**
   * Crea un atributo y lo agrega al elemento indicado.
   * 
   * @var DOMElement $element el elemento al cual desearmos agregarle una tributo
   * @var string $name el nombre del atributo
   * @var string $value el valor del atributo
   * 
   * @return DOMAttr Devuelve el atributo que se acaba de crear.
   */
  public function addAttribute($element, $name, $value)
  {
    // Bouml preserved body begin 001D3F85
	$element->appendChild($att = $this->log->createAttribute($name));
	$att->value=$value;
	return $att;
    // Bouml preserved body end 001D3F85
  }

  /**
   * Configura el ORM fuente de datos.
   * 
   * @param ORM $from el ORM origen desde el cual se realizará la migración
   * 
   * @return MigrationToolkit la herramienta con el origen configurado
   */
  public function setOrigin(&$from)
  {
    // Bouml preserved body begin 001D2285
	$this->from = $from;
	return $this;
    // Bouml preserved body end 001D2285
  }

  /**
   * Configura el ORM destino de datos.
   * 
   * @param ORM $to el ORM hacia el cual se realizará la migración
   * 
   * @return MigrationToolkit la herramienta con el destino configurado
   */
  public function setDestination(&$to)
  {
    // Bouml preserved body begin 001D2305
	$this->to = $to;
	return $this;
    // Bouml preserved body end 001D2305
  }

  /**
   * Define los parámetros de migración de las entidades pertenecientes a una clase.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param array $identifiers los atributos candidatos que pueden considerarse irrepetibles entre las bases
   * @param array $filters un array de tuplas que representan un filtro tal cual como si fueran parámetros de ORMQuery
   * @param array $exclude una lista de atributos que de los cuales la herramienta no debe preocuparse
   * 
   * @return MigrationToolkit la herramienta de migración con la configuración agregada.
   */
  public function addClass($class, $identifiers, $filters = null, $exclude = array())
  {
    // Bouml preserved body begin 001D2205
	if (is_string($identifiers)) {
		if (strpos($identifiers, ',') === false) {
			$identifiers = array($identifiers);
		} else {
			$identifiers = array_map('trim', explode(',', $identifiers));
		}
	}
	
	foreach($identifiers as $identifier)
		if (strpos($identifier, '.')!==false)
			throw new ORMException("No se permiten atributos anidados para la clave candidata de un objeto. Error en \"$identifier\" de la clase $class");
			
	if (is_string($exclude)) {
		if (strpos($exclude, ',') === false) {
			$exclude = array($exclude);
		} else {
			$exclude = array_map('trim', explode(',', $exclude));
		}
	}
	
	$this->config[$class]['identifiers']=$identifiers;
	$this->config[$class]['query']['filters']=$filters;
	$this->config[$class]['exclude']=$exclude;
	$this->config[$class]['migrate']=true;
	$this->config[$class]['execute']='SIMPLE';
	$this->config[$class]['status']=self::PENDING;
	return $this;
    // Bouml preserved body end 001D2205
  }

  /**
   * Indica a la herramienta cómo reconocer una cierta clase en las bases, pero no realiza migración sobre ella.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param array $identifiers los atributos candidatos que pueden considerarse irrepetibles entre las bases
   * 
   * @return MigrationToolkit la herramienta de migración con la configuración agregada.
   */
  public function defineClass($class, $identifiers)
  {
    // Bouml preserved body begin 001D5905
	if (is_string($identifiers)) {
		if (strpos($identifiers, ',') === false) {
			$identifiers = array($identifiers);
		} else {
			$identifiers = array_map('trim', explode(',', $identifiers));
		}
	}
	$this->config[$class]['identifiers']=$identifiers;
	$this->config[$class]['migrate']=false;
	$this->config[$class]['exclude']=array();
	return $this;
    // Bouml preserved body end 001D5905
  }

  /**
   * Declara que una entidad en particular de la base de origen debe considerarse como "igual" a otra cierta entidad en el destino. Esto permite redireccionar asociaciones y valores.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param int $idFrom el identificador de la entidad en el origen
   * @param int $idTo el identificador de la entidad en el destino
   * 
   * @return MigrationToolkit la herramienta de migración con la configuración agregada.
   */
  public function exchange($class, $idFrom, $idTo)
  {
    // Bouml preserved body begin 001D5985
	if (isset ($this->config[$class])) {
		$this->config[$class]['exchange']["$idFrom"] = $idTo;
	} else {
		throw new Exception("Error al llamar a exchange('$class', $idFrom, $idTo) => Antes de agregar excepciones asegúrese de haber indicado información de identificadores mediante addClass() o addDefinition()");
	}
	return $this;
    // Bouml preserved body end 001D5985
  }

  /**
   * Función auxiliar de migración
   */
  public function migrate($class, $root)
  {
    // Bouml preserved body begin 001DB485
	if ($this->debug) echo "\n================================================================\n";
	if ($class === null) return false;//Fake para ayudar a terminar el execute
	if ($this->debug) echo "Migrando clase $class\n";
	$config = &$this->config;
	
	if (!isset($config[$class])) {
		throw new ORMException("Error al llamar a MigrationToolkit::migrate($class, \$root) => No hay configuración para la clase");
		return true;//Si no está seteada ninguna configuración, asumimos que está todo migrado.
	}
	
	if ($config[$class]['status'] == self::WORKING)
		return true;
	
	$to = $this->to;
	$from = $this->from;
	
	//Busco objetos desde el origen que cumplan con las condiciones de la configuración
	//...pero ahora chequeo si la query es "manual" o si ya fue preparada por el usuario.
	/*echo "Clase $class (".$config[$class]['execute']."):\n";*/
	if ($config[$class]['execute'] == 'SIMPLE') {
		$query = $from->query($class);
		foreach($config[$class]['query']['filters'] as $filter)
			call_user_func_array(array($query, 'filterBy'), $filter);
		$objs = $query->distinct()->find();
	} else {//Asumo COMPLEX
		$query = $config[$class]['wrapper'];
		$objs = $query->distinct()->find();
	}
	if ($this->debug) echo Utils::debugQuery ($query) . "\n";
	//Si hay algo que migrar
	if ($objs) {
		if ($this->debug) echo "$class Encontrados ".count($objs). " instancias.\n";
		$domClass = $this->addElement('class', null, $root);
		$this->addAttribute($domClass, 'name', $class);
		$config[$class]['status']=self::WORKING;
		foreach($objs as $objFrom) {
			if ($this->debug) echo "    * (".$objFrom->getId().") ";
			$instance = $this->addElement('instance',null, $domClass);
			$keys = $this->addElement('keys', null, $instance);
			$migration = $this->addElement('migration', null, $instance);
			list($objTo, $st) = $this->findObject($objFrom, $keys);
			$this->addAttribute($migration, 'found', $st);
			if ($st == self::ST_MANY)
				continue;
			if ($objTo == null) {
				$objTo = new $class($to);
				if ($this->debug) echo "\n == Nuevo ==\n";
			} else {
				if ($this->debug) echo "\n == Existente ==\n";
			}
			$definition = $from->getDefinition($class);
			$fields = $definition->getFieldDefinition();
			$domAttributes = $this->addElement('attributes', null, $migration);
			foreach($fields as $attribute=>$defField) {
				if ($this->debug) echo "\tAtributo: $class::$attribute-> ";
				if ($attribute == 'id') {
					if ($this->debug) echo "\n";
					$this->addAttribute($instance, 'idFrom', $objFrom->getId());
				} else {
					$domAttribute = $this->addElement('attribute', null, $domAttributes);
					$this->addAttribute($domAttribute, 'name', $attribute);
					$this->addAttribute($domAttribute, 'type', $defField['type']);
					if (!in_array($attribute, $config[$class]['exclude'])) {
						$getter = 'get'.ucfirst($attribute);
						$setter = 'set'.ucfirst($attribute);
						switch($defField['type']) {
							case ORMDefinition::ORM_ENTITY:
								$attClass = $defField['class'];
								$this->addAttribute($domAttribute, 'class', $attClass);
								if (isset($config[$attClass])) {
									if (($config[$attClass]['migrate']) && ($config[$attClass]['status']==self::PENDING))
										$this->migrate($attClass, $root);
									if ($objFrom->$getter() != null) {
										$var = $objFrom->$getter();
										list($toEntity, $stAtt) = $this->findObject($var);
										if ($this->debug) echo "$stAtt\n";
										switch($stAtt) {
											case self::ST_EXCHANGED:
												$this->addAttribute($domAttribute, 'value', $toEntity->getId());
												if ($objTo->$getter() != null) {
													$this->addAttribute($domAttribute, 'oldValue', $objTo->$getter()->getId());
												} else {
													$this->addAttribute($domAttribute, 'oldValue', '');
												}
												$objTo->$setter($toEntity);
												$this->addAttribute($domAttribute, 'status', 'EXCHANGED');
												break;
											case self::ST_MANY:
												$this->addAttribute($domAttribute, 'status', 'MANY');
												break;
											case self::ST_NOTFOUND:
												$this->addAttribute($domAttribute, 'status', 'NOTFOUND');
												break;
											case self::ST_FOUND:
												$this->addAttribute($domAttribute, 'value', $toEntity->getId());
												if ($objTo->$getter() != null) {
													$this->addAttribute($domAttribute, 'oldValue', $objTo->$getter()->getId());
												} else {
													$this->addAttribute($domAttribute, 'oldValue', '');
												}
												$objTo->$setter($toEntity);
												$this->addAttribute($domAttribute, 'status', 'MIGRATED');
												break;
											default:
												throw new ORMException("Error en MigrationToolkit::migration($class, \$root) => Resultado inesperado de MigrationToolkit::find($attClass [".$objFrom->$getter()->getId()."]) = (xxx, $st)");
										}
									} else {
										$objTo->$setter(null);
										$this->addAttribute($domAttribute, 'value', null);
										$this->addAttribute($domAttribute, 'oldValue', null);
										$this->addAttribute($domAttribute, 'status', 'MIGRATED');
									}
								} else {
									//no está definida la clase, por ende no se migra
									$this->addAttribute($domAttribute, 'status', 'IGNORED');
								}
								break;
							default:
								$this->addAttribute($domAttribute, 'value', $objFrom->$getter());
								if ($objTo->$getter() != null) {
									$this->addAttribute($domAttribute, 'oldValue', $objTo->$getter());
									if ($this->debug) echo $objTo->$getter()."\n";
								} else {
									$this->addAttribute($domAttribute, 'oldValue', '');
									if ($this->debug) echo "NULL\n";
								}
								$objTo->$setter($objFrom->$getter());
								$this->addAttribute($domAttribute, 'status', 'MIGRATED');
						}
					} else {
						$this->addAttribute($domAttribute, 'status', 'EXCLUDED');
					}
				}
			}
			//Ahora migro las relaciones
			if ($this->debug) echo "\nCargando Relaciones:\n";
			$relations = $definition->getRelationDefinition();
			$domRelations = $this->addElement('relations', null, $migration);
			if ($relations) {
				foreach($relations as $attribute => $relDef) {
					if ($this->debug) echo "\t$attribute => ";
					$relClass = $relDef['class'];
					$domRelation = $this->addElement('relation', null, $domRelations);
					$this->addAttribute($domRelation, 'name', $attribute);
					$this->addAttribute($domRelation, 'type', $relDef['type']);
					$this->addAttribute($domRelation, 'class', $relClass);
					if (!in_array($attribute, $config[$class]['exclude'])) {//Chequeo que no esté excluída la relación
						if ($this->debug) echo "\n";
						if (isset($config[$relClass])) {//Si no está configurada para migrarse la excluyo también
							if (($config[$relClass]['migrate']) && ($config[$relClass]['status']==self::PENDING))//Si está pendiente la migro primero
								$this->migrate($relClass, $root);
							$getter = 'get'.ucfirst($attribute);
							$adder  = 'add'.ucfirst($attribute);
							$relObs = $objFrom->$getter();
							if ($relObs != null) {
								foreach($relObs as $relObj) {
									//Agrego el <id>
									$domId = $this->addElement('instance', null, $domRelation);
									$this->addAttribute($domId, 'fromId', $relObj->getId());
									list($toRelation, $stRel) = $this->findObject($relObj);
									switch($stRel) {
										case self::ST_EXCHANGED:
											$this->addAttribute($domId, 'toId', $toRelation->getId());
											if ($objTo->$getter() != null) {
												$count = $objTo->$getter()->count();
											} else {
												$count = 0;
											}
											$objTo->$adder($toRelation);
											if ($count < $objTo->$getter()->count()) {
												$this->addAttribute($domId, 'action', 'ADDED');
												$this->addAttribute($domId, 'status', 'EXCHANGED');
											} else {
												$this->addAttribute($domId, 'action', 'NONE');
												$this->addAttribute($domId, 'status', 'EXCHANGED');
											}
											break;
										case self::ST_MANY:
											$this->addAttribute($domId, 'toId', '');
											$this->addAttribute($domId, 'action', 'NONE');
											$this->addAttribute($domId, 'status', 'MANY');
											break;
										case self::ST_NOTFOUND:
											$this->addAttribute($domId, 'toId', '');
											$this->addAttribute($domId, 'action', 'NONE');
											$this->addAttribute($domId, 'status', 'NOTFOUND');
											break;
										case self::ST_FOUND:
											$this->addAttribute($domId, 'toId', $toRelation->getId());
											if ($objTo->$getter() != null) {
												$count = $objTo->$getter()->count();
											} else {
												$count = 0;
											}
											$objTo->$adder($toRelation);
											if ($count < $objTo->$getter()->count()) {
												$this->addAttribute($domId, 'action', 'ADDED');
												$this->addAttribute($domId, 'status', 'MIGRATED');
											} else {
												$this->addAttribute($domId, 'action', 'NONE');
												$this->addAttribute($domId, 'status', 'MIGRATED');
											}
											break;
										default:
									}
								}
							}
							$this->addAttribute($domRelation, 'status', 'MIGRATED');
						} else {
							//no está configurada, por ende no se migra
							$this->addAttribute($domRelation, 'status', 'IGNORED');
						}
					} else {//else está excluído
						$this->addAttribute($domRelation, 'status', 'EXCLUDED');
						if ($this->debug) echo "excluído\n";
					}
				}
			}
			$objTo->save();
			$this->addAttribute($instance, 'toId', $objTo->getId());
		}
		$config[$class]['status'] = self::MIGRATED;
		return true;
	} else {
		if ($this->debug) echo "No se encontraron elementos de la clase\n";
		$domClass = $this->addElement('class', 'No se encontraron elementos en el origen', $root);
		$this->addAttribute($domClass, 'name', $class);

		$config[$class]['status'] = self::MIGRATED;
		return true;
	}
    // Bouml preserved body end 001DB485
  }

  /**
   * Busca el objeto equivalente al dado en el destino.
   * 
   * @param ORMObject $object el objeto cuyo equivalente se desea encontrar
   * @param DOMElement $keys un objeto DOM al cual se le desea agregar la lista de claves empleadas para encontrar el objeto.
   * 
   * @return ORMObject|null Devuelve el objeto equivalente en la base destino o null si no se encuentra.
   */
  public function findObject(&$object, &$keys = null)
  {
    // Bouml preserved body begin 001DB405
	$config = &$this->config;
	$class = get_class($object);
	
	if ($object == null) throw new ORMException("Error interno en MigrationToolkit::findObject() => Por alguna razón se pasó null como parámetro");
	
	if (isset($config[$class])) {
		if (isset($config[$class]['exchange']) && isset($config[$class]['exchange'][''.$object->getId()])) {
			//Hay un exchange definido
			$eqObject = $this->to->retrieve($class, $config[$class]['exchange'][''.$object->getId()]);
			if (!$eqObject)
				$eqObject = $this->to->load($class, $config[$class]['exchange'][''.$object->getId()]);
			if ($keys != null) {
				$key = $this->addElement ('key', null, $keys);
				$this->addAttribute($key, 'idFrom', $object->getId());
				$this->addAttribute($key, 'idTo', $config[$class]['exchange'][''.$object->getId()]);
				$this->addAttribute($key, 'status', self::ST_EXCHANGED);
			}
			return array($eqObject, self::ST_EXCHANGED);
		} else {
			//Hay que buscar en base a las claves candidatas configuradas
			if (isset($config[$class]['identifiers']) && (count($config[$class]['identifiers'])>0)) {
				$query = $this->to->query($class);
				foreach($config[$class]['identifiers'] as $identifier) {
					$this->completeQuery($query, $object, $identifier, null, $keys);
				}
				if ($this->debug) echo Utils::debugQuery ($query)."\n";
				$eqObject = $query->find();
				$st = null;
				switch(true) {
					case (count($eqObject) > 1):
						$st = self::ST_MANY;
						break;
					case ($eqObject === false) || ($eqObject === null):
						$st = self::ST_NOTFOUND;
						break;
					default:
						$st = self::ST_FOUND;
						$eqObject = $eqObject[0];
				}
				return array($eqObject, $st);
			} else {
				//No hay identifiers!
				throw new ORMException("Error al invocar a MigrationToolkit::findObject($class [".$object->getId()."]]) => No se han definido claves candidatas para la clase");
			}
		}
	} else {
		//No existe definición!
		throw new ORMException("Error al invocar a MigrationToolkit::findObject($class [".$object->getId()."]]) => No existe configuración de la clase que indique cómo buscar un objeto equivalente en el destino.");
	}
    // Bouml preserved body end 001DB405
  }

  /**
   * Agrega filtros a una consulta, en base a las configuraciones, el path indicado y el prefix, tomando como base al objeto $object.
   * 
   * @param ORMQuery $query la consulta a completar
   * @param ORMObject $object el objeto del cual se toman los valores a comparar
   * @param string $path la clave o claves (anidadas por ".") que se desean filtrar
   * @param string $prefix al completar la query, a veces se pasa un objeto que es fruto de acceder ciertos atributos a traves de n "getters", si el path se pone en la query sólo el atributo filtrado no podría ser ubicado. Con el $prefix se complementa el camino recorrido para obtener $object.
   * @param DOMElement $keys un objeto DOMElement al cual adjuntar las claves por las cuales se realiza la búsqueda.
   * 
   * @return 
   */
  public function completeQuery(&$query, &$object, $path, $prefix = null, &$keys = null)
  {
    // Bouml preserved body begin 001DB385
	$config = $this->config;
	$class = get_class($object);
	$finalPath = $path;
	$this->counter++;
	if ($this->debug) echo str_pad("\ncompleteQuery(\$query, $class [".$object->getId()."], $path, $prefix)", $this->counter, " ", STR_PAD_LEFT);
	if (strpos($path, '.') === false) {
		$getter = 'get'.  ucfirst($path);
		$objTmp = $object->$getter();
		$finalPath = (($prefix != null)?"$prefix.$path":$path);
		if ($this->debug) echo "\n1 => $finalPath";
	} else {
		$parts = explode('.', $path);
		$objTmp = $object;
		for($i = 0; $i < count($parts);$i++) {
			$getterTmp='get'.ucfirst($parts[$i]);
			$objTmp = $objTmp->$getterTmp();
			if ($objTmp === null) {
				$copyParts = $parts;
				$nullPath = implode('.', array_slice($copyParts, 0, $i+1));
				break;
			}
		}
		if (!isset($nullPath)) $nullPath = $path;
		$finalPath = (($prefix != null)?"$prefix.$nullPath":$nullPath);
		if ($this->debug) echo "\n2 => $finalPath";
	}
	switch(true) {
		case is_a($objTmp, 'ORMObject'):
			$attClass = get_class($objTmp);
			if (isset($config[$attClass])) {
				if (isset($config[$attClass]['exchange']) && isset($config[$attClass]['exchange'][''.$objTmp->getId()])) {
					$query->filterBy("$finalPath.id", '=', $config[$attClass]['exchange'][''.$objTmp->getId()]);
					if ($keys != null) {
						$key = $this->addElement('key', null, $keys);
						$this->addAttribute($key, 'path', $finalPath);
						$this->addAttribute($key, 'idFrom', $objTmp->getId());
						$this->addAttribute($key, 'idTo', $config[$attClass]['exchange'][''.$objTmp->getId()]);
						$this->addAttribute($key, 'status', self::ST_EXCHANGED);
					}
					if ($this->debug) echo "\n$finalPath.id = ".$config[$attClass]['exchange'][''.$objTmp->getId()]." <=> EXCHANGED!\n";
				} else {
					if (isset($config[$attClass]['identifiers']) && (count($config[$attClass]['identifiers'])>0)) {
						foreach($config[$attClass]['identifiers'] as $attIdentifier) {
							//$attGetter = 'get'.ucfirst($attIdentifier);
							//$var = $objTmp->$attGetter();
							$this->completeQuery($query, $objTmp, $attIdentifier, $finalPath, $keys);
						}
					} else {
						//No hay identifiers!
						throw new ORMException("Error al invocar a MigrationToolkit::completeQuery(\$query, $class [".$object->getId()."], $path, $prefix) => No se han definido claves candidatas para la clase $attClass");
					}
				}
			} else {
				//No existe definición!
				throw new ORMException("Error al invocar a MigrationToolkit::completeQuery(\$query, $class [".$object->getId()."], $path, $prefix) => No existe configuración de la clase $attClass que indique cómo buscar un objeto equivalente en el destino.");
			}
			break;
		case $objTmp == null:
			if ($this->debug) echo "\n$finalPath IS NULL";
			$query->filterBy($finalPath, 'IS NULL');
			if ($keys != null) {
				$key = $this->addElement('key', null, $keys);
				$this->addAttribute($key, 'path', $finalPath);
				$this->addAttribute($key, 'operator', 'IS NULL');
			}
			break;
		default:
			if ($this->debug) echo "\n$finalPath = $objTmp";
			$query->filterBy($finalPath, '=', $objTmp);
			if ($keys != null) {
				$key = $this->addElement('key', null, $keys);
				$this->addAttribute($key, 'path', $finalPath);
				$this->addAttribute($key, 'operator', '=');
				$this->addAttribute($key, 'value', $objTmp);
			}
	}
	$this->counter--;
    // Bouml preserved body end 001DB385
  }

  /**
   * Habilita o deshabilita el modo debug.
   * 
   * @param boolean $enable si se pasa un true se habilita el modo debug (acción por default) y si se pasa false, se deshabilita.
   * 
   * @return MigrationToolkit la herramienta con el modo debug configurado
   */
  public function setDebug($enable = true)
  {
    // Bouml preserved body begin 001DB605
	$this->debug = $enable;
	return $this;
    // Bouml preserved body end 001DB605
  }

  /**
   * Función auxiliar
   */
  public function logFiltersAndGroups(&$node, &$root)
  {
    // Bouml preserved body begin 001DD205
	if (isset($node['filters'])) {
		foreach($node['filters'] as $filter) {
			$f = $this->addElement('filter', null, $root);
			$this->addAttribute($f, 'attribute', $filter[0]);
			$this->addAttribute($f, 'operator', $filter[1]);
			if (isset($filter[2]))
				$this->addAttribute($f, 'value1', $filter[2]);
			if (isset($filter[3]))
				$this->addAttribute($f, 'value2', $filter[3]);
			if (isset($filter[4]))
				$this->addAttribute($f, 'union', $filter[4]);
		}
	}
	if (isset($node['groups'])) {
		foreach($node['groups'] as $group) {
			$g = $this->addElement('group', null, $root);
			$this->logFiltersAndGroups($group, $g);
		}
	}
    // Bouml preserved body end 001DD205
  }

  /**
   * Realiza la migración de entidades en base a las configuraciones cargadas.
   * 
   * @param string $logfile la ruta y nombre de archivo donde quedará el log de la operación
   * 
   * @return boolean Devuelve true si todo funciona bien o arroja una excepción con el error en caso que algo no funcione bien.
   */
  public function execute($logfile)
  {
    // Bouml preserved body begin 001D2385
	$dom = $this->log = new DOMDocument('1.0', 'utf-8');
	$log = $this->addElement('log');
	$settings = $this->addElement('connections', null, $log);
		$from = $this->addElement('from',null,$settings);
		$to = $this->addElement('to',null,$settings);
	$configuration = $this->addElement('configuration', null, $log);
	if (isset($this->config) && is_array($this->config)) {
		foreach($this->config as $class=>$conf) {
			//if ($conf['migrate']) {
				$classConfig = $this->addElement('class', null, $configuration);
				$this->addAttribute($classConfig, 'name', $class);
				$this->addAttribute($classConfig, 'migrate', ($conf['migrate']?'yes':'no'));
				$this->addElement('identifiers', implode(',', $conf['identifiers']), $classConfig);
				if (isset($conf['exclude']) && is_array($conf['exclude']))
					$this->addElement('exclude',  implode(',', $conf['exclude']), $classConfig);
				if (isset($conf['query'])) {
					$query = $this->addElement('query', null, $classConfig);
					if ($conf['execute'] == 'COMPLEX') {
						$this->addAttribute($query, 'type', 'COMPLEX');
						if (isset($conf['query']['limits'])) {
							$limits = $this->addElement('limits', null, $query);
							if (isset($conf['query']['limits']['from']))
								$this->addAttribute ($limits, 'from', $conf['query']['limits']['from']);
							if (isset($conf['query']['limits']['to']))
								$this->addAttribute ($limits, 'to', $conf['query']['limits']['to']);
							if (isset($conf['query']['limits']['top']))
								$this->addAttribute ($limits, 'top', $conf['query']['limits']['top']);
						}
						if (isset($conf['query']['orderBy'])) {
							$orderBy = $this->addElement('orderBy', null, $query);
							$this->addAttribute ($orderBy, 'attributes', implode (',', $conf['query']['orderBy']));
						}
						if (isset($conf['query']['dontWorryAbout'])) {
							$dontWorryAbout = $this->addElement('dontWorryAbout', null, $query);
							$this->addAttribute ($dontWorryAbout, 'attributes', implode (',', $conf['query']['dontWorryAbout']));
						}
						if (isset($conf['query']['worryAbout'])) {
							$worryAbout = $this->addElement('worryAbout', null, $query);
							$this->addAttribute ($worryAbout, 'attributes', implode (',', $conf['query']['worryAbout']));
						}
						if (isset($conf['query']['groupBy'])) {
							$groupBy = $this->addElement('groupBy', null, $query);
							$this->addAttribute ($groupBy, 'attributes', implode (',', $conf['query']['groupBy']));
						}
					} else {
						$this->addAttribute($query, 'type', 'SIMPLE');
					}
					$this->logFiltersAndGroups($conf['query'], $query);
				}
				if (isset($conf['exchange']) && is_array($conf['exchange'])) {
					foreach($conf['exchange'] as $idFrom => $idTo) {
						$exchanges = $this->addElement('exchanges', null, $classConfig);
						$this->addAttribute($exchanges, 'from', $idFrom);
						$this->addAttribute($exchanges, 'to', $idTo);
					}
				}
			//}
		}
		//Iniciamos la migración
		try {
			//Elijo las clases que son migrables
			$classes = array();
			foreach($this->config as $class => $config) {
				if ($config['migrate']) $classes[]=$class;
			}
			//print_r($classes);
			while($this->migrate($class = array_shift($classes), $log)) {  }
			if ($class === null) {
				//Terminó con éxito
				if ($this->debug) echo "Terminó con éxito\n";
			} else {
				//Hubo un error
				if ($this->debug) echo "Fallo en $class\n";
			}
		} catch (Exception $e) {
			if ($this->debug) echo $e;
		} 
	}
	$str = $dom->saveXML();
	return file_put_contents($logfile, $str);
	
    // Bouml preserved body end 001D2385
  }

  /**
   * Restaura los cambios realizados por una migración anterior.
   * 
   * @return boolean Retorna true si todo sale bien o una excepción en caso que haya algún problema.
   */
  public function rollback($filepath)
  {
    // Bouml preserved body begin 001D2405
	
    // Bouml preserved body end 001D2405
  }

  /**
   * Define los parámetros de migración de las entidades pertenecientes a una clase.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param array $identifiers los atributos candidatos que pueden considerarse irrepetibles entre las bases
   * @param Closure $filterFunction una función que toma un ORMQuery y aplica los filtros necesarios
   * @param array $exclude una lista de atributos que de los cuales la herramienta no debe preocuparse
   * 
   * @return MigrationToolkit la herramienta de migración con la configuración agregada.
   */
  public function addClass2($class, $identifiers, $filterFunction = null, $exclude = array())
  {
    // Bouml preserved body begin 001D7885
	if (is_string($identifiers)) {
		if (strpos($identifiers, ',') === false) {
			$identifiers = array($identifiers);
		} else {
			$identifiers = array_map('trim', explode(',', $identifiers));
		}
	}

	foreach($identifiers as $identifier)
		if (strpos($identifier, '.')!==false)
			throw new ORMException("No se permiten atributos anidados para la clave candidata de un objeto. Error en \"$identifier\" de la clase $class");

	if (is_string($exclude)) {
		if (strpos($exclude, ',') === false) {
			$exclude = array($exclude);
		} else {
			$exclude = array_map('trim', explode(',', $exclude));
		}
	}
	$this->config[$class]['identifiers']=$identifiers;
	$this->config[$class]['query']=array();
	$this->config[$class]['exclude']=$exclude;
	$this->config[$class]['migrate']=true;
	$this->config[$class]['execute']='COMPLEX';
	$this->config[$class]['status']=self::PENDING;
	$queryWrapper = new ORMQueryMTWrapper($class, $this->from, $this->config);
	$filterFunction($queryWrapper);
	$this->config[$class]['wrapper'] = $queryWrapper;
	return $this;
    // Bouml preserved body end 001D7885
  }

  /**
   * Amplia el conjunto de objetos a migrar para una clase.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param array $filters un array de tuplas que representan un filtro tal cual como si fueran parámetros de ORMQuery
   * 
   * @return MigrationToolkit la herramienta con el set de objetos ampliado
   */
  public function addDataSet($class, $filters)
  {
    // Bouml preserved body begin 001DD285
    // Bouml preserved body end 001DD285
  }

  /**
   * Agrega un nuevo conjunto de objetos a migrar para una clase previamente declarada.
   * 
   * @param string $class el nombre de la clase a migrar
   * @param Closure $filterFunction una función que toma un ORMQuery y aplica los filtros necesarios
   * 
   * @return MigrationToolkit la herramienta con el nuevo selector de objetos agregado.
   */
  public function addDataSet2($class, $function)
  {
    // Bouml preserved body begin 001DD305
    // Bouml preserved body end 001DD305
  }

}
?>