<?php

/**
 * Constante que define donde se va a hacer el GET de confirmación.
 */
define("LINK_CONFIRM_RESET", "http://autoprovisioning.iplantech.com.ar/confirmReset.php");

/**
 * Constante que define donde se va a hacer el GET de confirmación para los teléfonos de IPLAN
 */
define("LINK_CONFIRM_RESET_IPLAN", "http://gadget.centralvirtual.iplantech.com.ar/confirmReset.php");

/**
 * Función de reseteo de Password SIP con Confirmación 
 * 
 * @param string $linea La línea
 * @param string $mail El Mail de confirmación
 * @return string
 */
function resetPasswordSIPConfirm($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$log = LogMessages::GetInstance();
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPasswordSIPConfirm, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Linea [$linea] Mail [$mail]", null, $proceso_id);

	return resetPassword_SIP_WEB_Confirm($linea, $mail, "SIP");
}

/**
 * Función de reseteo de Password WEB con Confirmación 
 * 
 * @param string $linea La línea
 * @param string $mail El Mail de confirmación
 * @return string
 */
function resetPasswordPortalConfirm($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$log = LogMessages::GetInstance();
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPasswordPortalConfirm, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Linea [$linea] Mail [$mail]", null, $proceso_id);

	
	return resetPassword_SIP_WEB_Confirm($linea, $mail, "WEB");
}

/**
 * Función de reseteo de Password del Administrador con Confirmación 
 * 
 * @param string $linea La línea
 * @param string $mail El Mail de confirmación
 * @return string
 */
function resetPasswordGroupAdminConfirm($linea, $mail){
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPasswordGroupAdminConfirm, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Linea [$linea] Mail [$mail]", null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id);
	$userGroup = $serviceCentralIUNI->SearchDNByPhoneNumber($linea);
	
	if (!$userGroup) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "No existe el usuario", null, $proceso_id); 
		return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
	}
	
	$aDatosGrupo = $serviceCentralIUNI->getGroup($userGroup["groupId"], $userGroup["serviceProviderId"]);
	if (!(isset($aDatosGrupo["contact"]) && isset($aDatosGrupo["contact"]["contactEmail"]) && ($aDatosGrupo["contact"]["contactEmail"] == $mail))) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "El mail del perfil del grupo no coincide con el enviado.", null, $proceso_id); 
		return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
	}
	
	/* Busco todos los adminitradores del grupo */
	$aAdminUsers = array_keys($serviceCentralIUNI->SearchGroupAdmins($userGroup["serviceProviderId"], $userGroup["groupId"]));
	
	/* Administrador que crea el AP */
	$admin_default = "admin_". array_shift(explode("-", $userGroup["groupId"]));

	/* Busco en la lista de administradores el default de AP */
	if (array_search($admin_default, $aAdminUsers) === false) {
		//Admin no esta
		if (count($aAdminUsers)) {
			$selected_admin = array_shift($aAdminUsers);
		} else {
			//El grupo no tiene administradores
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "El grupo en Broadsoft no tiene administradores o el grupo no existe", null, $proceso_id); 
			return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
		}
	} else {
		//Tiene el usuario
		$selected_admin = $admin_default;
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 0, "Seleccionado administrador [$selected_admin]", null, $proceso_id); 
	}

	/* Genero el hash para el pedido */
	$hash = generarHash($mail);
	
	/* Creo el Confirm File */
	if (!crearConfirmFile($mail, $linea, $hash, "ADMIN")) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "No se pudo crear el confirm file", null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - NO SE PUDO CREAR EL CONFIRM FILE");
	} else {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 0, "Se creó el confirm file del hash [$hash]", null, $proceso_id); 
	}
	
	/* Envío el Mail de Confirmación */
	if (!enviarMailConfirm($selected_admin, $mail, $hash, 'ADMIN', $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		return new SoapFault("Server", "MAIL: ERROR - MAIL NO ENVIADO");
	} else {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id); 
	}

	return "ESTADO: OK - CONFIRMACION ENVIADA";
}

/**
 * Función de seteo de Voicemail para el usuario
 * @param type $usuario
 * @param type $voicemail
 * @param type $enabled
 * @return type
 */
function setUserVoicemail($usuario, $voicemail, $enabled) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: setUserVoicemail, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Usuario [$usuario] Voicemail [$voicemail] Enabled [$enabled] ", null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	$respuesta = $serviceCentralIUNI->SetUserVoicemail($usuario, $voicemail, $enabled);

	switch ($respuesta["estado"]) {
		case "no disponible":
			return new SoapFault("Server", "IUNI: No pudo contectarse al servicio");
			break;
		case "error":
			return new SoapFault("Server", "IUNI: Fallo al actualizar el voicemail");
			break;
		case "ok";
			return "VOICEMAIL: OK - VOICEMAIL MODIFICADO";
			break;
	}
	
}

/**
 * Devuelve el Voicemail del usuario
 * 
 * @param string $usuario
 * @return string
 */
function getUserVoicemail($usuario){

	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];

	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: getUserVoicemail, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Usuario [$usuario]", null, $proceso_id);

	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);

	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	$respuesta = $serviceCentralIUNI->GetUserVoicemail($usuario);

	if (!$respuesta) {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Voicemail [$usuario] No encontrado", null, $proceso_id);
		return new SoapFault("Server", "VOICEMAIL: No encontrado");
	}

	$res = array("voiceMessageDeliveryEmailAddress" =>	$respuesta["voiceMessageDeliveryEmailAddress"], "isActive" => $respuesta["isActive"]);

	return $res;
}

function getUserByMail($mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];

	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: getUserByMail, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Mail [$mail]", null, $proceso_id);
	
	$respuesta = getUserByCriterio("mail", $mail);
	
	if ($respuesta) {
		return array("estado" => "200", "mensaje" => $respuesta);
	} else {
		return array("estado" => "404", "mensaje" => "No se registran líneas asignadas a esa dirección de correo.");
	}
}

function getUserByLastName($apellido) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];

	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: getUserByLastName, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: LastName [$apellido]", null, $proceso_id);
	
	return getUserByCriterio("last_name", $apellido);
	
}

function getUserByExtension($extension) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];

	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: getUserByExtension, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Extension [$extension]", null, $proceso_id);
	
	return getUserByCriterio("extension", $extension);
	
}

function getUserByCriterio($tipo_dato, $dato) {

	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];

	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);

	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);

	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	
	$respuesta = $serviceCentralIUNI->SearchUsersByCritero("Iplan Corporativo", "iplan_internos", $tipo_dato, $dato);

	if (!$respuesta) {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Usuario con el criterio [$tipo_dato] [$dato] No encontrado", null, $proceso_id);
		return false;
	}
	
	return $respuesta;
}

/**
 * Devuelve si es una linea autoaprovisionada
 * 
 * @param string $linea La línea
 * @param string $mail No se usa, puesto por compatibilidad
 * @return string|SoapFault
 */
function isAutoprovisionedLine($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$context = $application->getContext();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: isAutoprovisionedLine, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Línea [$linea] Mail [$mail]", null, $proceso_id);

	$sql = "SELECT L.LINEA linea
			FROM LINEAS L, CAP_SERVICEORDERTASKS S, SITIO SI, CLIENTE CLI
			WHERE L.SUBSCRIPTION = S.SUBSCRIPTIONID
				AND L.SITIO_ID = SI.SITIO_ID
				AND SI.CLIENTE_ID = CLI.CLIENTE_ID
				AND L.ESTADO = 1
				AND CLI.ESTADO = 1
				AND S.ESTADO_ID = 7
				AND (S.TASKCODE = 'MC')
				AND LINEA = :li";
	
	$ostmt = $dbconn->prepareStatement($sql);
	
	if ($linea) $bindeo = $dbconn->bind($ostmt, ":li", $linea);
	
	$dbconn->prefetch($ostmt, $context->getProvider()->getMaxFetchRows());
	$orst = $dbconn->execute($ostmt);
	$tmp = $dbconn->fetch($ostmt, Connection::M_ASSOCIATIVE);
	
	if( $tmp && $tmp['LINEA'] == $linea)
		return "LINEA: OK - LINEA AUTOAPROVISIONADA";

	return new SoapFault("Server", "LINEA: ERROR - LINEA NO AUTOPROVISIONADA");
}

/**
 * Agrega al usuario el default de CallNotify
 * 
 * @param string $usuario
 * @return string
 */
function addDefaultCallNotifycation($usuario) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: addDefaultCallNotifycation, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Usuario [$usuario]", null, $proceso_id);

	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	$respuesta = $serviceCentralIUNI->AddUserCallNotify($usuario);

	switch ($respuesta["estado"]) {
		case "no disponible":
			return new SoapFault("Server", "IUNI: No pudo contectarse al servicio");
			break;
		case "error":
			return new SoapFault("Server", "IUNI: Fallo al agregar el CallNotify");
			break;
		case "ok":
			return "CALLNOTIFYCATION: OK - CRITERIO DEFAULT AGREGADO";
			break;
	}
}

/**
 * Función general de Reseteo de Password
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @param string $tipo El tipo de mail que es: [ADMIN | SIP | WEB]
 * @return SoapFault
 */
function resetPassword($linea, $mail, $tipo) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$log = LogMessages::GetInstance();
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPassword, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Linea [$linea] Mail [$mail] Tipo [$tipo]", null, $proceso_id);

	switch ($tipo) {
		case "WEB":
			$respuesta = resetPasswordPortal($linea, $mail);
			break;
		case "SIP":
			$respuesta = resetPasswordSIP($linea, $mail);
			break;
		case "ADMIN":
			$respuesta = resetPasswordGroupAdmin($linea, $mail);
			break;
		default:
			return new SoapFault("Server", "ERROR - Funcion no soportada");
			break;
	}

	return $respuesta;
}

/**
 * Función de Reseteo de Password para internos de IPLAN
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @param string $tipo El tipo de mail que es: [ADMIN | SIP | WEB]
 * @return SoapFault
 */
function resetPassword_iplan($linea, $mail, $tipo) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$log = LogMessages::GetInstance();
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPassword, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Linea [$linea] Mail [$mail] Tipo [$tipo]", null, $proceso_id);

	switch ($tipo) {
		case "WEB":
			$respuesta = resetPasswordPortal($linea, $mail);
			break;
		case "SIP":
			$respuesta = resetPasswordSIP_iplan($linea, $mail);
			break;
		default:
			return new SoapFault("Server", "ERROR - Funcion no soportada");
			break;
	}

	return $respuesta;
}

/**
 * Función para resetear password portal
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @return string|SoapFault
 */
function resetPasswordPortal($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	$respuesta = "";
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	$lineas = $serviceCentralIUNI->SearchUsersByMail($mail);

	/* Genera el nuevo password */
	$new_pass = ProvisioningPlataformas::generarPassword();

	$mensajes = array(	"LOG" => array(	"OK" => " LOG: OK - Registro de ejecución almacenado",
										"ERROR" => " LOG: ERROR - Registro de ejecución NO guardado" ),
						"MAIL" => array("OK" => " MAIL: OK - Correo enviado",
										"ERROR" => " MAIL: ERROR - Correo NO enviado"),
						"ESTADO" => array(	"OK" => " ESTADO: OK - Pass cambiada",
											"ERROR" => " ESTADO: ERROR - Pass NO cambiada")
					);

	$userIds = array_keys($lineas);
	array_walk($userIds, function ($elem) {list($elem, ) = explode("@", $elem);});
	if (array_search($linea, $userIds) === false) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "El la línea [$linea] no pertenece al correo [$mail]", null, $proceso_id);
		return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"]);
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se verifica que la  línea [$linea] pertenece al correo [$mail]", null, $proceso_id);
	}
	
	$res = $serviceCentralIUNI->SetUserPasswordWeb($linea, $new_pass);
	switch ($res["estado"]) {
		case "no disponible":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Broadsoft no disponible, no pudo contectarse al servicio.", null, $proceso_id); 
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: No pudo contectarse al servicio"*/);
			break;
		case "error":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Fallo al actualizar el password.", null, $proceso_id);
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: Fallo al actualizar el password"*/);
			break;
		case "ok":
			//return "CALLNOTIFYCATION: OK - CRITERIO DEFAULT AGREGADO";
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Password cambiada correctamente.", null, $proceso_id);
			$respuesta .= $mensajes["ESTADO"]["OK"];
			break;
	}

	// Se envía mail a la direccion asociada a la línea informando del blanqueo de la PASS
	if (!enviarMail($linea, $mail, $new_pass, "WEB", $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["ERROR"];
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["OK"];
	}
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: $respuesta", null, $proceso_id);
	return $respuesta;
}

/**
 * Función para resetear password del administrador de grupo
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @return string|SoapFault
 */
function resetPasswordGroupAdmin($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	$respuesta = "";
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id);
	$userGroup = $serviceCentralIUNI->SearchDNByPhoneNumber($linea);
	
	/* Busco todos los adminitradores del grupo */
	$aAdminUsers = array_keys($serviceCentralIUNI->SearchGroupAdmins($userGroup["serviceProviderId"], $userGroup["groupId"]));
	
	/* Administrador que crea el AP */
	$admin_default = "admin_". array_shift(explode("-", $userGroup["groupId"]));
	
	
	$mensajes = array(	"LOG" => array(	"OK" => " LOG: OK - Registro de ejecución almacenado",
										"ERROR" => " LOG: ERROR - Registro de ejecución NO guardado" ),
						"MAIL" => array("OK" => " MAIL: OK - Correo enviado",
										"ERROR" => " MAIL: ERROR - Correo NO enviado"),
						"ESTADO" => array(	"OK" => " ESTADO: OK - Pass cambiada",
											"ERROR" => " ESTADO: ERROR - Pass NO cambiada")
					);

	$new_pass = ProvisioningPlataformas::generarPassword();
	
	/* Busco en la lista de administradores el default de AP */
	if (array_search($admin_default, $aAdminUsers) === false) {
		//Admin no esta
		if (count($aAdminUsers)) {
			$selected_admin = array_shift($aAdminUsers);
		} else {
			//El grupo no tiene administradores
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"]);
		}
	} else {
		//Tiene el usuario
		$selected_admin = $admin_default;
	}
	
	$res = $serviceCentralIUNI->SetUserPasswordAdmin($selected_admin, $new_pass);
	switch ($res["estado"]) {
		case "no disponible":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Broadsoft no disponible, no pudo contectarse al servicio.", null, $proceso_id); 
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: No pudo contectarse al servicio"*/);
			break;
		case "error":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Fallo al actualizar el password.", null, $proceso_id);
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: Fallo al actualizar el password"*/);
			break;
		case "ok":
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Password cambiada correctamente.", null, $proceso_id);
			$respuesta .= $mensajes["ESTADO"]["OK"];
			break;
	}
	
	// Se envía mail a la direccion asociada a la línea informando del blanqueo de la PASS
	if (!enviarMail($selected_admin, $mail, $new_pass, "ADMIN", $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["ERROR"];
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["OK"];
	}
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: $respuesta", null, $proceso_id);
	return $respuesta;

}

/**
 * Función para resetear password sip de la línea
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @return string|SoapFault
 */
function resetPasswordSIP($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	$respuesta = "";
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$servicePlataformas = new ProvisioningPlataformas($dbconn, $log, $user_ejecuta, $proceso_id, null, null);
	$servicePlataformas->setServiceProviderId(10);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id);
	$lineas = $serviceCentralIUNI->SearchUsersByMail($mail);
	
	$mensajes = array ( "LOG" => array(   "OK"  => " LOG: OK - Registro de ejecución almacenado",
				"ERROR" => " LOG: ERROR - Registro de ejecución NO guardado" )
			, "MAIL" => array(  "OK"  => " MAIL: OK - Correo enviado",
				"ERROR" => " MAIL: ERROR - Correo NO enviado"  )
			, "ESTADO" => array("OK"  => " ESTADO: OK - Pass cambiada",
				"ERROR" => " ESTADO: ERROR - Pass NO cambiada")
			, "TFTP"   => array("OK"  => " TFTP: OK - template generado",
				"ERROR" => " TFTP: ERROR - template NO generado")
			, "SharedCallAppearance"   => array("OK"  => " SharedCallAppearance: OK - SharedCallAppearance habilitado",
				"ERROR" => " SharedCallAppearance: ERROR - No se pudo habilitar el SharedCallAppearance")
			);
	
	$userIds = array_keys($lineas);
	
	/* Guardo las líneas originales */
	$lineas_orginal = $userIds;
	
	/* Reemplazo la respuesta */
	array_walk($userIds, function ($elem) {list($elem, ) = explode("@", $elem);});
	$index = array_search($linea, $userIds);
	if ($index === false) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "El la línea [$linea] no pertenece al correo [$mail]", null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
	} else {
		$userId = $lineas_orginal[$index];
	}
	
	try {
		$oLine = $orm->load("Line", $linea);
		$nro_cliente = $oLine->getSite()->getClient()->getClientNumber();
	} catch (Exception $e) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "Linea no registrada en Autoprovisioning - " . $e->getMessage(), null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - NO SE PUEDE PROCESAR LA LINEA");
	}
	
	
	/* Genero un nuevo password para la línea */
	$new_pass = ProvisioningPlataformas::generarPassword();
	
	/* Cambio el password en Broadsoft */
	$resBS = $serviceCentralIUNI->SetUserPasswordSIP($userId, $new_pass);
	switch ($resBS["estado"]) {
		case "no disponible":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Broadsoft no disponible, no pudo contectarse al servicio.", null, $proceso_id); 
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: No pudo contectarse al servicio"*/);
			break;
		case "error":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Fallo al actualizar el password.", null, $proceso_id);
			return new SoapFault("Server", $mensajes["ESTADO"]["ERROR"] /*"IUNI: Fallo al actualizar el password"*/);
			break;
		case "ok":
			//return "CALLNOTIFYCATION: OK - CRITERIO DEFAULT AGREGADO";
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Password cambiada correctamente.", null, $proceso_id);
			$respuesta .= $mensajes["ESTADO"]["OK"];
			break;
	}
	
	/* Guardo el password en sac */
	$resSAC = $servicePlataformas->modServiceSAC($nro_cliente, "LineaIUNI", $linea, array("PasswordSIP" => $new_pass));
	if (!$resSAC) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "SAC no disponible, no pudo contectarse al servicio.", null, $proceso_id); 
		$respuesta .= " ESTADO: OK - Pass NO guardada";
	} else {
		switch ($resSAC["codigo"]) {
			case "0":
				//return "CALLNOTIFYCATION: OK - CRITERIO DEFAULT AGREGADO";
				$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Password guardada en SAC correctamente.", null, $proceso_id);
				$respuesta .= " ESTADO: OK - Pass guardada";
				break;
			default:
				$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Fallo al guardar el password en SAC. Código [" . $resSAC["codigo"] . "] Descripción [" . $resSAC["descripcion"] . "]", null, $proceso_id);
				$respuesta .= " ESTADO: OK - Pass NO guardada";
				break;
		}
	}
	
	/* Cosas de Template */
	try {
		if ($oLine->getMac()) {
			$generacionTemplate = AsignarHardware::generateHardwareTemplate($orm, $oLine->getMac());
		} else {
			throw new Exception("La línea no es autoaprovisionada, no se puede generar template.");
		}
	} catch (Exception $e) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "Error al generar el template - " . $e->getMessage(), null, $proceso_id);
		$respuesta .= $mensajes["TFTP"]["ERROR"];
	}
	
	// Se envía mail a la direccion asociada a la línea informando del blanqueo de la PASS
	if (!enviarMail($linea, $mail, $new_pass, "SIP", $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["ERROR"];
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["OK"];
	}
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: $respuesta", null, $proceso_id);
	return $respuesta;
}

/**
 * Función para resetear password sip de una línea de IPLAN
 * 
 * @param string $linea La línea a cambiar el password
 * @param string $mail El mail correspondiente a la línea
 * @return string|SoapFault
 */
function resetPasswordSIP_iplan($linea, $mail) {
	
	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	$respuesta = "";
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$servicePlataformas = new ProvisioningPlataformas($dbconn, $log, $user_ejecuta, $proceso_id, null, null);
	$servicePlataformas->setServiceProviderId(10);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id);
	$lineas = $serviceCentralIUNI->SearchUsersByMail($mail);
	
	$mensajes = array ( "LOG" => array(   "OK"  => " LOG: OK - Registro de ejecución almacenado.",
				"ERROR" => " LOG: ERROR - Registro de ejecución NO guardado." )
			, "MAIL" => array(  "OK"  => " MAIL: OK - Correo enviado.",
				"ERROR" => " MAIL: ERROR - Correo NO enviado."  )
			, "ESTADO" => array("OK"  => " ESTADO: OK - Pass cambiada.",
				"ERROR" => " ESTADO: ERROR - Pass NO cambiada.")
			, "TFTP"   => array("OK"  => " TFTP: OK - template generado.",
				"ERROR" => " TFTP: ERROR - template NO generado.")
			, "SharedCallAppearance"   => array("OK"  => " SharedCallAppearance: OK - SharedCallAppearance habilitado.",
				"ERROR" => " SharedCallAppearance: ERROR - No se pudo habilitar el SharedCallAppearance.")
			);
	
	$userIds = array_keys($lineas);
	
	/* Guardo las líneas originales */
	$lineas_orginal = $userIds;
	
	/* Reemplazo la respuesta */
	array_walk($userIds, function ($elem) {list($elem, ) = explode("@", $elem);});
	$index = array_search($linea, $userIds);
	if ($index === false) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "El la línea [$linea] no pertenece al correo [$mail]", null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
	} else {
		$userId = $lineas_orginal[$index];
	}
	
	/* Genero un nuevo password para la línea */
	$new_pass = ProvisioningPlataformas::generarPassword();
	
	/* Cambio el password en Broadsoft */
	$resBS = $serviceCentralIUNI->SetUserPasswordSIP($userId, $new_pass);
	switch ($resBS["estado"]) {
		case "no disponible":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Broadsoft no disponible, no pudo contectarse al servicio (Pass Change).", null, $proceso_id); 
			$respuesta .= $mensajes["ESTADO"]["ERROR"];
			break;
		case "error":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Fallo al actualizar el password.", null, $proceso_id);
			$respuesta .= $mensajes["ESTADO"]["ERROR"];
			break;
		case "ok":
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Password cambiada correctamente.", null, $proceso_id);
			$respuesta .= $mensajes["ESTADO"]["OK"];
			break;
	}
	
	/* Creo la instancia de SCA */
	$usuario = array_shift(explode("@", $mail));
	$resSCA = $serviceCentralIUNI->AddSharedCallAppearanceInternosIPLAN($linea, $usuario);
	switch ($resSCA["estado"]) {
		case "no disponible":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, "Broadsoft no disponible, no pudo contectarse al servicio (SCA Verify).", null, $proceso_id); 
			$respuesta .= $mensajes["SharedCallAppearance"]["ERROR"];
			break;
		case "error":
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 6, " Fallo al agregar el SCA del interno.", null, $proceso_id);
			$respuesta .= $mensajes["SharedCallAppearance"]["ERROR"];
			break;
		case "ok":
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "SCA creado o existente.", null, $proceso_id);
			$respuesta .= $mensajes["SharedCallAppearance"]["OK"];
			break;
	}
	
	
	/* Cosas de Template */
	try {
		generarTemplateTFTP_iplan($linea, $new_pass);
	} catch (Exception $e) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "Error al generar el template - " . $e->getMessage(), null, $proceso_id);
		$respuesta .= $mensajes["TFTP"]["ERROR"];
	}
	
	// Se envía mail a la direccion asociada a la línea informando del blanqueo de la PASS
	if (!enviarMail($linea, $mail, $new_pass, "SIP", $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["ERROR"];
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id);
		$respuesta .= $mensajes["MAIL"]["OK"];
	}
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: $respuesta", null, $proceso_id);
	return $respuesta;
}

/**
 * Función de auxiliar de reseteo de Password SIP/WEB con Confirmación 
 * 
 * @param string $mail El Mail de las líneas y de confirmación
 * @return string
 */
function resetPassword_SIP_WEB_Confirm_iplan($mail, $tipo) {

	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 26, "Operation: resetPassword_SIP_WEB_Confirm_iplan, IP: " . getIp(), null, $proceso_id);
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Parámetros: Tipo [$tipo] Mail [$mail]", null, $proceso_id);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	
	$validacion = Validation::getValidation("mail");
	if (!$validacion->validate($mail)) {
		return false;
	}
	
	$lineas = $serviceCentralIUNI->SearchUsersByMail($mail);
	if (!count($lineas)) {
		throw new Exception("No existen lineas asociadas al correo [$mail]");
	}
	
	/* UserIds */
	$userIds = array_keys($lineas);
	$aFailHash = array();
	$aFailMail = array();
	foreach ($userIds as $userId) {
		/* Genero el hash para el pedido */
		$hash = generarHash($mail);
		
		/* Creo el Confirm File */
		if (!crearConfirmFile($mail, array_shift(explode("@", $userId)), $hash, $tipo)) {
			$creado = false;
			$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "No se pudo crear el confirm file del hash [$hash]", null, $proceso_id);
			$aFailHash[] = $userId;
		} else {
			$creado = true;
			$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se creó el confirm file del hash [$hash]", null, $proceso_id);
		}
		
		if ($creado) {
			/* Envío el Mail de Confirmación */
			$linea = array_shift(explode("@", $userId));
			if (!enviarMailConfirm($linea, $mail, $hash, $tipo, $application, true)) {
				$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo a [$mail] de la línea [$linea]", null, $proceso_id);
				$aFailMail[] = $linea;
			} else {
				$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail] de la línea [$linea]", null, $proceso_id);
			}
		}
	}
	
	$aDesc = array();
	if ($aFailMail)
		$aDesc[] = "Ocurrió un fallo al enviar el/los mail/s de la/s línea/s [" . implode(", ", $aFailMail) . "]";
	if ($aFailHash)
		$aDesc[] = "Ocurrió crear el/los file confirm de la/s línea/s [" . implode(", ", $aFailHash) . "]";
	
	if ($aDesc)
		throw new Exception (implode(", ", $aDesc));
		
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: [" . implode(", ", $userIds) . "]", null, $proceso_id);
	return $userIds;
}

/**
 * Función de auxiliar de reseteo de Password SIP/WEB con Confirmación 
 * 
 * @param string $linea La línea
 * @param string $mail El Mail de confirmación
 * @return string
 */
function resetPassword_SIP_WEB_Confirm($linea, $mail, $tipo) {

	$proceso_id = $_SESSION["WS_PROCESO_ID"];
	$user_ejecuta = $_SESSION["WS_USER_EJECUTA_PACKAGES"];
	
	/* Traigo los datos de Instancias */
	$application = Application::getInstance();
	$orm = $application->getORM();
	$dbconn = $orm->getDatabase()->getConnection(); //Traigo la Conexion
	
	$log = LogMessages::GetInstance();
	
	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, getCurrentFuction(1), null, $proceso_id);
	
	$bsws = BroadsoftXMLWS::getInstance(BroadsoftXMLWS::WSDL_WS, BroadsoftXMLWS::USER_WS, BroadsoftXMLWS::PASSWORD_WS, $log);
	
	$serviceCentralIUNI = new ProvisioningIUNI($bsws, $log, $dbconn, $user_ejecuta, null, $proceso_id, null);
	$lineas = $serviceCentralIUNI->SearchUsersByMail($mail);
	
	/* Reemplazo la respuesta */
	$userIds = array_keys($lineas);
	array_walk($userIds, function ($elem) {list($elem, ) = explode("@", $elem);});
	if (array_search($linea, $userIds) === false) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 7, "El la línea [$linea] no pertenece al correo [$mail]", null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - DATOS INVALIDOS");
	}

	/* Genero el hash para el pedido */
	$hash = generarHash($mail);
	
	/* Creo el Confirm File */
	if (!crearConfirmFile($mail, $linea, $hash, $tipo)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "No se pudo crear el confirm file", null, $proceso_id);
		return new SoapFault("Server", "ESTADO: ERROR - NO SE PUDO CREAR EL CONFIRM FILE");
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se creó el confirm file del hash [$hash]", null, $proceso_id);
	}

	/* Envío el Mail de Confirmación */
	if (!enviarMailConfirm($linea, $mail, $hash, $tipo, $application)) {
		$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CA, 24, "Fallo al enviar el correo.", null, $proceso_id);
		return new SoapFault("Server", "MAIL: ERROR - MAIL NO ENVIADO");
	} else {
		$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envió el correo a [$mail]", null, $proceso_id);
	}

	$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CA, 0, "Se envía la respuesta: ESTADO: OK - CONFIRMACION ENVIADA", null, $proceso_id);
	return "ESTADO: OK - CONFIRMACION ENVIADA";
}


/**
 * Genera un hash para el ID de petición.
 * 
 * @param string $mail 
 * @return string
 */
function generarHash($mail) {
	return (substr(sha1($mail.time().microtime()), 0, 30));
}

/**
 * Crea un archivo en el temporal que guarda la petición de reset password.
 * 
 * @param string $email La cuenta de email que se le enviará el correo
 * @param string $linea La línea la cual hace referencia
 * @param string $hash El hash que se utiliza para identificar la petición
 * @param string $tipo_reset El tipo de reset, admite los valores: [SIP | ADMIN | WEB] 
 * @return boolean si se pudo crear o no.
 */
function crearConfirmFile($email, $linea, $hash, $tipo_reset) {
	$datos = "$linea;$email;$tipo_reset\n";
	if ($dh = opendir("/tmp/")) {
		while (($file = readdir($dh)) !== false) {
			if (preg_match ("/^confirm\_[a-z0-9]{30}\.session$/", $file) ) {
				@$otros_datos = file_get_contents("/tmp/" . $file);
				if (trim($otros_datos) == trim($datos)) {
					unlink("/tmp/" . $file);
				}
			}
		}
		closedir($dh);
	}
	return file_put_contents("/tmp/confirm_$hash.session", $datos);
}

/**
 * Verifica un confirm file y devuelve un array con sus datos o false en caso de que no exista
 * 
 * @param string $hash el hash del file a verificar
 * @return boolean|array false o el arreglo con los datos
 */
function verificarConfirmFile($hash) {
	//Leo los datos del archivo
	@$datos = file_get_contents("/tmp/confirm_$hash.session");
	
	if ($datos !== false) {
		$aDatos = explode(";", $datos);
		$linea = array_shift($aDatos);
		$email = array_shift($aDatos);
		$tipo_reset =  trim(array_shift($aDatos));
		$usuario_text = '';
		
		switch ($tipo_reset) {
			case "ADMIN":
				$usuario_text = 'administrador';
				break;
			case "SIP":
				$usuario_text = 'SIP';
				break;
			default:
				$usuario_text = 'WEB';
				break;
		}
		$res = array("linea" => $linea, "email" => $email, "tipo_reset" => $tipo_reset, "usuario_text" => $usuario_text);
	} else {
		$res = false;
	}
	return $res;
}

/**
 * Elimina el confirm file correspondiente al hash.
 * 
 * @param string $hash
 * @return boolean TRUE on success or FALSE on failure.
 */
function deleteConfirmFile($hash) {
	return unlink("/tmp/confirm_$hash.session");
}

/**
 * Envía un mail de Confirmación de Reseteo
 * 
 * @param string $linea La Línea que hace referencia
 * @param string $mail El Mail al cual se enviará
 * @param string $hash El hash que se utiliza para identificar la petición
 * @param string $tipo El tipo de mail que es: [ADMIN | SIP | WEB]
 * @param Application $application La aplicación que se utiliza para enviar el mail.
 * @return boolean Si se pudo enviar el mail o no.
 */
function enviarMailConfirm($linea, $mail, $hash, $tipo, $application, $isIplan = false) {
	
	$context = $application->getContext();

	
	$pagina = (($isIplan) ? LINK_CONFIRM_RESET_IPLAN : LINK_CONFIRM_RESET) . "?id=$hash";
	
	$aTextos = array(	"ADMIN" =>
						array(	"subject" => "Confirmación reseteo de password administrador de la Central Virtual (primer paso)",
								"tipo_text" => 'administrador'),
						"SIP" =>
						array(	"subject" => "Confirmación reseteo de password del usuario SIP de la Central Virtual $linea (primer paso)",
								"tipo_text" => 'SIP'),
						"WEB" =>
						array(	"subject" => "Confirmación reseteo de password WEB de la Central Virtual $linea (primer paso)",
								"tipo_text" => 'WEB'),
					);
	
	$aParametros["tipo_text"] = $aTextos[$tipo]["tipo_text"];
	$aParametros["linea"] = $linea;
	$aParametros["pagina"] = $pagina;
	$aParametros["isIplan"] = $isIplan;
	
	/* Parseo el template de mail y lo mando */
	$genManager = $context->getManager("Application");
	$template_mail = $application->parse("lib/password_reset_confirm.xhtml", $aParametros);
   $resultado_mail = $genManager->mailTo("servicios@iplantech.com.ar", $mail, utf8_decode($aTextos[$tipo]["subject"]), $template_mail);

	return $resultado_mail;
}

/**
 * Envía el mail final reseteo con los datos
 * 
 * @param string $linea La Línea que hace referencia
 * @param string $mail El Mail al cual se enviará
 * @param string $pass El password nuevo
 * @param string $tipo El tipo de mail que es: [ADMIN | SIP | WEB]
 * @param Application $application La aplicación que se utiliza para enviar el mail.
 * @return boolean Si se pudo enviar el mail o no.
 */
function enviarMail($linea, $mail, $pass, $tipo, $application) {
	
	$context = $application->getContext();
	
	$aTextos = array(	"ADMIN" =>
						array(	"subject" => "Confirmación reseteo de password administrador de la Central Virtual (último paso)",
								"tipo_text" => 'administrador',
								"usuario_text" => 'Usuario',
								"ultimo_msj" => 'utilizar el servicio con los datos brindados.'
								),
						"SIP" =>
						array(	"subject" => "Confirmación reseteo de password usuario SIP de la Central Virtual $linea (último paso)",
								"tipo_text" => 'SIP',
								"usuario_text" => 'Linea',
								"ultimo_msj" => 'descargar el instructivo de configuración de tu teléfono o softphone.'
								),
						"WEB" =>
						array(	"subject" => "Confirmación reseteo de password WEB de la Central Virtual $linea (último paso)",
								"tipo_text" => 'WEB',
								"usuario_text" => 'Usuario',
								"ultimo_msj" => 'utilizar el servicio con los datos brindados.'),
					);
	
	$aParametros["tipo_text"] = $aTextos[$tipo]["tipo_text"];
	$aParametros["usuario_text"] = $aTextos[$tipo]["usuario_text"];
	$aParametros["ultimo_msj"] = $aTextos[$tipo]["ultimo_msj"];
	$aParametros["linea"] = $linea;
	$aParametros["pass"] = $pass;

	/* Parseo el template de mail y lo mando */
	$genManager = $context->getManager("Application");
	$template_mail = $application->parse("lib/password_reset.xhtml", $aParametros);
	$resultado_mail = $genManager->mailTo("servicios@iplantech.com.ar", $mail, utf8_decode($aTextos[$tipo]["subject"]), $template_mail);

	return $resultado_mail;
}

/**
 * Devuelve el nombre de la funcion que se llamó.
 * 
 * @param integer $deep El nivel de llamada de función
 * @return string El nombre de la función.
 */
function getCurrentFuction($deep = 0) {
	$dbg = debug_backtrace();
	return ($dbg[$deep + 1]['function']);
}

/**
 * Devuelve la IP del cliente
 * @return string La IP
 */
function getIp() {
	return $_SERVER['REMOTE_ADDR'];
}

/**
 * Genera un template para los teléfonos de iplan
 * 
 * @param string $linea 
 * @param string $new_pass
 * @return boolean true si pudo crearlo
 * @throws Exception Cuando no puede crearlo
 */
function generarTemplateTFTP_iplan($linea, $new_pass) {
	
	$filename = "$linea.xml";
	$path = "/home/tftp/tftpboot/$filename";
	$xml = "<flat-profile><Password_1_  ua=\"rw\">$new_pass</Password_1_></flat-profile>";

	if (file_put_contents($path, $xml)) {
		return true;
	} else {
		throw new Exception("No se pudo guardar el template");
	}
}

function habilitarSharedCallAppearance_iplan ($linea, $mail) {
	$retorno = false;
	$usuario = split("@", $mail);

	$hola = BroadsoftXMLWS::getInstance(WSDL_WS, USER_WS, PASSWORD_WS);

	$xd = new BroadsoftXMLBase();


	$Crear_device = new BroadsoftXMLGroupAccessDeviceAddRequest14();
	$Crear_device->setData("Iplan Corporativo", "iplan_internos", "Teletrabajo_$usuario[0]", "Xten Soft Client", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 0, "");
	$xd->addCommand($Crear_device->getCommand());
	$Crear_device->setCommand();
	$hola->execute($xd->getXML());
	$request = $hola->getLastRequest();
	$response = $hola->getLastResponse();

	$SharedCallAppearance = new BroadsoftXMLUserSharedCallAppearanceAddEndpointRequest14sp2();

	$SharedCallAppearance->setData($linea, array("deviceLevel" => "Group", "deviceName" => "Teletrabajo_$usuario[0]", "linePort" => $linea . "_2@hpbx.iplannetworks.net"), "true", "true", "true");
	$xd->addCommand($SharedCallAppearance->getCommand());
	$SharedCallAppearance->setCommand();
	$hola->execute($xd->getXML());
	$request = $hola->getLastRequest();
	$response = $hola->getLastResponse();

// Busco en el Response por la frase SuccessResponse que significa que se pudo habilitar el SharedCallAppearance, sino puede ser que ya lo tenia configurado
	$retorno = checkRespuestaBS($response, "SuccessResponse");

	if ($retorno) {
		return $retorno;
	} else {
// Busco en el Response por la frase "Address is already used on the system" que significa que ya lo tenia configurado el SharedCallAppearance
		$retorno = checkRespuestaBS($response, "Address is already used on the system");
	}

	return $retorno;
}


?>
