<?php
require_once 'iplan/orm/ORMObject.php';
require_once 'iplan/security/Provider.php';
require_once 'iplan/models/googleapps/GoogleTransaction.php';
require_once 'iplan/models/wholesale/Client.php';
require_once 'iplan/orm/ORMDefinition.php';
require_once 'iplan/orm/ORM.php';
require_once 'iplan/models/provisioning/ServiceOrder.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Simboliza una cuenta de GoogleApps. La misma puede ser una cuenta existente (si la instancia la creó el AutoProvisioning) o una entidad que está pendiente de creación (si se crea desde la GUI).
 * Es posible identificar ésto dado que si está creada la cuenta, el campo "created" tendrá una fecha. También se puede ver la transacción inicial y observar el usuario que la creó.
 * 
 * @method string getDomain()
 * @method GoogleAccount setDomain()
 * @method int getLicences()
 * @method bool getDelegatedDNS()
 * @method DateTime getCreated()
 * @method GoogleAccount setCreated()
 * @method DateTime getDeleted()
 * @methdo GoogleAccount setDeleted()
 * @method ORMCollection getTransactions()
 * @method GoogleAccount setTransactions()
 * @method GoogleTransaction getTransaction()
 * @method GoogleAccount addTransaction()
 * @method GoogleAccount delTransaction()
 * @method Provider getProvider()
 * @method GoogleAccount setProvider
 * @method Client getClient()
 * @method GoogleAccount setClient()
 * @method GoogleAccount setTransferToken()
 * @method string getTransferToken()
 * @method GoogleAccount setRequireTransferToken()
 * @method boolean getRequireTransferToken()
 * @method GoogleAccount setTxt()
 * @method string getTxt()
 */
class GoogleAccount extends ORMObject {
  /**
   * @var string el dominio al cual se vinculó la cuenta
   */
  protected $domain;

  /**
   * @var int la cantidad de licencias que tiene habilitadas
   */
  protected $licences;

  /**
   * @var boolean un flag que indica si la novedad además involucra la delegación de dominios
   */
  protected $delegatedDNS;

  /**
   * @var DateTime la fecha en la cual fue creado el registro
   */
  protected $created;

  /**
   * @var DateTime la fecha en la cual fue dada de baja la cuenta
   */
  protected $deleted;

  /**
   * @var Provider el provider a nombre del cual se creó la cuenta
   */
  protected $provider;

  /**
   * @var ORMCollection la lista de transacciones que se realizaron sobre la cuenta
   */
  protected $transactions;

  /**
   * @var Client el cliente dueño de la cuenta
   */
  protected $client;

  /**
   * @var boolean un flag que indica si la cuenta requiere transfertoken
   */
  protected $requireTransferToken;

  /**
   * @var string el hash de transferencia de dominio
   */
  protected $transferToken;

  /**
   * @var string el texto de validación de dominio a través de los DNS
   */
  protected $txt;

  /**
   * @var int la cantidad de licencias que variaron del anterior valor.
   */
  private $deltaLicences = 0;

  /**
   * @var boolean indica si hubo un cambio del flag de DNS
   */
  private $deltaDNS = false;

  /**
   * Retorna la definición para el ORM de la clase GoogleAccount
   * 
   * @param ORM $orm el manejador de objetos para el cual se registra la definición
   * @return ORMDefinition La definición de la clase GoogleAccount
   */
  public static function define(&$orm = null)
  {
    // Bouml preserved body begin 001A1C85
	return parent::define($orm)
			  ->setClass('GoogleAccount')
			  ->setTable('GA_ACCOUNT')
			  ->addField('id', 'ACCOUNT_ID', ORMDefinition::INTEGER)
				->setKey('id')
			  ->addField('domain', 'ACCOUNT_DOMAIN', ORMDefinition::STRING, 255, null, false)
			  ->addField('licences', 'ACCOUNT_LICENCES', ORMDefinition::INTEGER, null, null, false)
			  ->addField('delegatedDNS', 'ACCOUNT_DNS', ORMDefinition::BOOLEAN, null, null, false, 'false')
			  ->addField('created', 'ACCOUNT_CREATED', ORMDefinition::DATE, null, null, false, 'NOW()')
			  ->addField('deleted', 'ACCOUNT_DELETED', ORMDefinition::DATE, null, null)
			  ->addField('requireTransferToken', 'ACCOUNT_REQUIRETRANSFERTOKEN', ORMDefinition::BOOLEAN)
			  ->addField('transferToken', 'ACCOUNT_TRANSFERTOKEN', ORMDefinition::STRING, 16)
			  ->addValidation('transferToken', 'min 16')
			  ->addField('txt', 'ACCOUNT_TXT', ORMDefinition::STRING, 80)
			  ->addInstance('provider', 'PROVIDER_ID', 'Provider', false)
			  ->addInstance('client', 'CLIENTE_ID', 'Client')
			  ->addRelation1xN('transactions', 'ACCOUNT_ID', 'GoogleTransaction')
			;
    // Bouml preserved body end 001A1C85
  }

  /**
   * Verifica si el objeto fue modificado y lo guarda.
   * Este método permite abstraer a las clases descendientes de la necesidad de determinar si el objeto fue alterado o no y oculta la complejidad subyacente a las clases descendientes.
   * @param ORM $orm opcionalmente se debe indicar la instancia del ORM que mantiene el objeto. Esto sólo es posible si el objeto es nuevo o DETACHED. En el caso que ya estuviera bajo monitoreo de un ORM arrojará una excepción.
   * 
   * @return boolean Un true si todo anduvo bien, false (o una excepción) si algo falló.
   * 
   * Se pensó que se podría hacer el cambio de un ORM a otro, pero el ID debería limpiarse, con lo cual sólo un INSERT se podría hacer sobre el destino y los objetos relacionados estarían aún con otro ORM lo cual es complicado (manejo de objetos distribu�dos) por lo cual se deja a consideración en futuras versiones, posiblemente en otra vida =)
   */
  public function save(&$orm = null)
  {
    // Bouml preserved body begin 001B2785
	if ($orm == null) $orm = $this->orm;
	$esNuevo = $this->id == null;
	$modifieds = array();
	if (!$esNuevo) $modifieds = $this->getModifiedAttributes();
	$res = parent::save($orm);
	$context = Application::getInstance()->getContext();
	
	if ($this->getProvider() == null)
		throw new Exception("No se puede crear una GoogleAccount sin Provider");
	
	$context = Application::getInstance()->getContext();
	if ($esNuevo) {
		//Si es nuevo, entonces valido si ya existe una transacción para esta cuenta en estado Pendiente, sino, la creo.
		$transactionExist = $orm->query('GoogleTransaction')
								->filterBy('domain', '=', $this->domain)
								->filterBy('provider.id', '=', $context->getProvider()->getId())
								->filterBy('finished', 'IS NULL')//Igual a preguntar por cualquier estado pendiente
								->findOne()
								;
		if (!is_a($transactionExist, 'GoogleTransaction')) {
			$transaction = new GoogleTransaction($orm);
			$transaction->setReaded(new DateTime())
						->setDomain($this->domain)
						->setLicences($this->licences)
						->setDelegatedDNS($this->delegatedDNS)
						->setProvider($context->getProvider())
						->setUser($context->getUser())
						->setType($orm->query('GoogleTransactionType')->filterBy('description', '=','Creación de Cuenta GoogleApps')->findOne())
						->setStatus($orm->load('GoogleStatus', (is_null($this->client)?GoogleStatus::SUSPENDED:GoogleStatus::PENDING)))
						->setAccount($this)
						->setNotification($this->serviceOrder)
					;
			$res2 = $orm->save($transaction);
			return $res && $res2;
		} else {
			throw new Exception('La transacción no se puede dar de alta porque ya existe otra encolada');
		}
	} else {
		//De momento, en una cuenta solo se puede cambiar el nro. de licencias y si tiene o no DNS delegados.
		foreach($modifieds as $attr) {
			switch($attr) {
				case 'licences':
								if ($this->deltaLicences != 0) {
									$transaction = new GoogleTransaction($orm);
									$transaction->setLicences(abs($this->deltaLicences))
												->setReaded(new DateTime())
												->setDomain($this->domain)
												->setDelegatedDNS($this->delegatedDNS)
												->setProvider($context->getProvider())
												->setUser($context->getUser())
												->setStatus($orm->query('GoogleStatus')->filterBy('description', '=','Pendiente')->findOne())
												->setAccount($this)
												->setNotification($this->serviceOrder)
											;
									if ($this->deltaLicences > 0) { //Upgrade
										$transaction->setType($orm->query('GoogleTransactionType')->filterBy('description', '=', 'Upgrade de Licencias para GoogleApps')->findOne());
									} else { //Downgrade
										$transaction->setType($orm->query('GoogleTransactionType')->filterBy('description', '=','Downgrade de Licencias para GoogleApps')->findOne());
									}
									$res2 = $orm->save($transaction);
									$res = $res && $res2;
								} else {
									//Puede tratarse de un reproceso
									$transaction = $orm->query('GoogleTransaction')
													   ->filterBy('notification', '=', $this->serviceOrder->getId())//Esto debería ser suficiente por sí sólo
													   ->filterBy('status', 'IN', array(GoogleStatus::CANCELED, GoogleStatus::ABORTED))
													   ->filterBy('account.id', '=', $this->getId())
													   ->orderBy('id')
													   ->findOne();
									//Si existe una transacción cancelada o abortada con el mismo serviceordertask veo de "revivirla"
									if ($transaction) {/* @var $transaction GoogleTransaction */
										$api = new APIGoogleApps();
										$log = LogMessages::GetInstance($orm->getDatabase()->getConnection(), $context->getUser()->getId(), LogMessages::SYS_CAGA);
										$log->getMensajes();
										$api->initLog($context, $orm, $log);
										$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 23, "Se detectó al menos una transacción cancelada o abortada para la tarea $this->serviceOrder", $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());

										$loginOK = false;
										try {
											$loginOK = $api->registerInGoogleApps('admin@reseller2.iplan.com.ar', 'ty45v5234');
										} catch(Exception $e) {
											$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'Fallo inesperado al autenticar: '.$e->getMessage());
										}

										if ($loginOK) {
											$currentLicences = $api->getMaxNumberOfLicences($this->domain);
											$diff = $this->licences - $currentLicences;
											$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 23, "El dominio $this->domain tiene un máximo de $currentLicences. En AP figura con $this->licences, se detecta diferencia de $diff.", $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
											switch(true) {
												case $diff > 0://Upgrade
													if ($transaction->getType()->getId() == GoogleTransactionType::UPGRADE_ACCOUNT) {
														if ($transaction->getLicences() == $diff) {
															$transaction->setStatus($orm->load('GoogleStatus',GoogleStatus::SUSPENDED))
																		->setNotified(false)
																		->setFinished(null)
																	;
															$transaction->save();
															$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 23, "Se reactivó la transacción para upgrade de $diff licencias del dominio $this->domain. ServiceOrderTaskId $this->serviceOrder", $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
														} else {
															$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'Se detecta un cambio en el la cantidad de licencias de la tarea.', $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
														}
													} else {
														$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'Se detecta un cambio en el tipo de operación', $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
													}
													break;
												case $diff < 0://Downgrade
													if ($transaction->getType()->getId() == GoogleTransactionType::DOWNGRADE_ACCOUNT) {
														if ($transaction->getLicences() == $diff) {
															$transaction->setStatus($orm->load('GoogleStatus',GoogleStatus::SUSPENDED))
																		->setNotified(false)
																		->setFinished(null)
																	;
															$transaction->save();
															$log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 23, "Se reactivó la transacción para downgrade de $diff licencias del dominio $this->domain. ServiceOrderTaskId $this->serviceOrder", $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
														} else {
															$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'Se detecta un cambio en el la cantidad de licencias de la tarea.', $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
														}
													} else {
															$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'Se detecta un cambio en el tipo de operación', $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
													}
													break;
												default:
													$log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 20, 'No se detecta necesidad de cambio, la cuenta ya posee las licencias a las que se quiere llegar. Verificar por favor', $this->serviceOrder->getSellOrder(), $this->serviceOrder->getProcess()->getId(), $this->serviceOrder->getId());
											}
										}
									}
								}
								break;
				case 'delegatedDNS':
								if ($this->deltaDNS) {
									$transaction = new GoogleTransaction($orm);
									$transaction->setDelegatedDNS($this->delegatedDNS)
												->setReaded(new DateTime())
												->setDomain($this->domain)
												->setProvider($context->getProvider())
												->setUser($context->getUser())
												->setStatus($orm->query('GoogleStatus')->filterBy('description', '=','Pendiente')->findOne())
												->setAccount($this)
												->setNotification($this->serviceOrder)
											;
									if ($this->delegatedDNS) { //Delegar DNS Automáticamente
										$transaction->setType($orm->query('GoogleTransactionType')->filterBy('description', '=','Delegación de Dominio para GoogleApp')->findOne());
									} else { //Quitar Delegación de DNS
										$transaction->setType($orm->query('GoogleTransactionType')->filterBy('description', '=','Baja de Delegación de Dominio para GoogleApps')->findOne());
									}
									$res2 = $orm->save($transaction);
									$res = $res && $res2;
								}
								break;
				case 'txt':
					if (($this->delegatedDNS) && (!is_null($this->txt)) && ($this->txt != '')) {
						
						$creationTransaction = $orm->query('GoogleTransaction')
												   ->filterBy('type.id', '=', GoogleTransactionType::CREATE_ACCOUNT)
												   ->filterBy('account.id', '=', $this->getId())
												   ->findOne();
						$notification = null;
						if ($creationTransaction) {
							$notification = $creationTransaction->getNotification();
						}
						$transaction = new GoogleTransaction($orm);
						$transaction->setLicences($this->licences)
									->setDelegatedDNS($this->delegatedDNS)
									->setReaded(new DateTime())
									->setDomain($this->domain)
									->setProvider($context->getProvider())
									->setUser($context->getUser())
									->setNotification($notification)
									->setStatus($orm->query('GoogleStatus')->filterBy('description', '=','Pendiente')->findOne())
									->setAccount($this)
									->setType($orm->query('GoogleTransactionType')->filterBy('description', '=','Alta de Registro TXT')->findOne())
								;
						$res2 = $orm->save($transaction);
						$res = $res && $res2;						
					}
					break;
				case 'transferToken':
					$creationTransaction = $orm->query('GoogleTransaction')
											   ->filterBy('type.id', '=', GoogleTransactionType::CREATE_ACCOUNT)
											   ->filterBy('account.id', '=', $this->getId())
											   ->findOne();/*@var $creationTransaction GoogleTransaction */
					if ($creationTransaction) {
						$context = Application::getInstance()->getContext();
						$facadeGoogleApps = $context->getManager('GoogleApps');/*@var $facadeGoogleApps GoogleAppsProvisioningManager */
						$creationTransaction->setInterval($facadeGoogleApps->getGlobal('TransferTokenSet'));
						$res2 = $creationTransaction->save();
						$res = $res && $res2;
					}
					break;
				default:
						break;
			}
		}
	}
	return $res;
    // Bouml preserved body end 001B2785
  }

  /**
   * Borra f�sicamente un objeto de la base. Al borrarse el objeto adem�s se ponen a null los campos de las entidades relacionadas que lo referenciaban y se eliminan los registros de las tablas relaci�n M:N
   * 
   * @return boolean un booleando TRUE si la operaci�n fue concretada con �xito o FALSE sino.
   */
  public function delete()
  {
    // Bouml preserved body begin 001B2805
	$context = Application::getInstance()->getContext();
	$existDelete = $this->orm->query('GoogleTransaction')
							 ->filterBy('domain', '=', $this->domain)
							 ->filterBy('provider.id','=', $this->getProvider()->getId())
							 ->filterBy('type.description', '=', 'Baja de Cuenta de GoogleApps')
							 ->filterBy('finished', 'IS NULL')
							 ->findOne();
	if (is_a($existDelete, 'GoogleTransaction')) {
		throw new Exception('La transacción ya ha sido preparada para la baja automática');
	} else {
		$context = Application::getInstance()->getContext();
		$transaction = new GoogleTransaction($this->orm);
		$transaction->setLicences(abs($this->deltaLicences))
					->setReaded(new DateTime())
					->setDomain($this->domain)
					->setLicences($this->licences)
					->setProvider($context->getProvider())
					->setUser($context->getUser())
					->setType($this->orm->query('GoogleTransactionType')->filterBy('description', '=','Baja de Cuenta de GoogleApps')->findOne())
					->setStatus($this->orm->query('GoogleStatus')->filterBy('description', '=','Pendiente')->findOne())
					->setAccount($this)
					->setNotification($this->serviceOrder)
				;
		return $this->orm->save($transaction);
	}
    // Bouml preserved body end 001B2805
  }

  /**
   * Cambia la cantidad de licencias de una cuenta.
   * 
   * @param int licences la nueva cantidad de licencias con la que debe quedar la cuenta.
   * 
   * @return GoogleAccount la cuenta con las licencias cambiadas.
   */
  public function setLicences($licences)
  {
    // Bouml preserved body begin 001B2885
	if ($this->id != null) { //Solo configuro en caso que la entidad ya exista.
		$this->deltaLicences += $licences - $this->licences;
	}
	return parent::setLicences($licences);
    // Bouml preserved body end 001B2885
  }

  /**
   * Cambia el flag de delegación de dns.
   * 
   * @param boolean dns un TRUE para indicar delegación y FALSE para quitarla.
   * 
   * @return GoogleAccount la cuenta modificada.
   */
  public function setDelegatedDNS($dns)
  {
    // Bouml preserved body begin 001B5C85
	if ($this->id != null) { //Solo configuro en caso que la entidad ya exista.
		$this->deltaDNS = $dns != $this->delegatedDNS;
	}
	return parent::setDelegatedDNS($dns);
    // Bouml preserved body end 001B5C85
  }

  /**
   * Valida los atributos DIRECTOS del objeto seg�n las reglas especificadas en la propia definici�n.
   * No llama recursivo porque ser�a muy "engorroso" y factible de caer en la circunstancia de tener una recursi�n infinita.
   * Tampoco se consideran en la validaci�n aquellos atributos asignados con un LazyLoader, dado que si est�n en base se asume que est�n OK.
   * 
   * @return mixed Retorna TRUE si todo est� bien y una matriz 3 dimensional indexada por atributo, validaci�n y errores. En caso que algo falle retorna una matr�z con el siguiente formato:
   * 
   * array (
   * 	0 = > array (
   * 		attribute, array("Error 1", "Error 2")
   * 		)
   * 	...
   * )
   */
  public function validate()
  {
    // Bouml preserved body begin 001B2905
	if ($this->id == null) {
		$context = Application::getInstance()->getContext();
		$exists = $this->orm->query('GoogleAccount')
							->filterBy('domain', '=', $this->domain)
							->filterBy('deleted', 'IS NULL')
							->findOne()
							;
		if (is_a($exists, 'GoogleAccount')) throw new Exception('No se puede crear dos veces el mismo dominio');
	}
	return parent::validate();
    // Bouml preserved body end 001B2905
  }

  /**
   * @var ServiceOrder temporal para indicar cuando se trata de un cambio basado en lectura de novedad
   */
  private $serviceOrder;

  /**
   * @return GoogleAccount
   */
  public function setServiceOrder($order)
  {
    // Bouml preserved body begin 001BE505
	$this->serviceOrder = $order;
	return $this;
    // Bouml preserved body end 001BE505
  }

  /**
   * @var boolean indica si el dominio está verificado por Google
   */
  protected $verified = false;

}
?>