<?php
require_once 'iplan/security/ApplicationContext.php';
require_once 'iplan/orm/ORM.php';
require_once 'iplan/models/googleapps/GoogleTransaction.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Esta clase es el encapsulamiento de la API de GoogleApps y que flexibiliza los llamados a las distintas funciones cuando se hacen a nombre del mismo reseller.
 */
class APIGoogleApps {
  /**
   * @const URL_AUTHENTICATION_TOKEN la URL del servicio de login de GoogleApps
   */
  const URL_AUTHENTICATION_TOKEN = 'https://www.google.com/accounts/ClientLogin';

  /**
   * @const URL_DOMAIN_VERIFICATION la URL a la cual se remiten los pedidos de verificación de dominios
   */
  const URL_DOMAIN_VERIFICATION = 'https://apps-apis.google.com/a/feeds/domain/2.0/%s/accountInformation/isVerified';

  /**
   * @const URL_BASE_DOMAIN la URL que determina si un dominio pertenece o no al reseller
   */
  const URL_BASE_DOMAIN = 'https://apps-apis.google.com/a/feeds/reseller/2.0/%s/domain/%s';

  /**
   * @const URL_DOMAIN_CREATE la URL que crea un dominio (una cuenta) de cliente
   */
  const URL_DOMAIN_CREATE = 'https://apps-apis.google.com/a/feeds/reseller/2.0/%s/domain';

  /**
   * @const URL_DOMAIN_GENERAL la URL que se emplea para consultar cosas de los dominios
   */
  const URL_DOMAIN_GENERAL = 'https://apps-apis.google.com/a/feeds/domain/2.0/%s/general/';

  /**
   * @const URL_USER_QUERY la URL que crea un usuario en un dominio dado
   */
  const URL_USER_QUERY = 'https://apps-apis.google.com/a/feeds/%s/user/2.0';

  /**
   * @const URL_DOMAIN_GENERAL la URL que se emplea para modificar los usuarios de un dominio dado
   */
  const URL_DOMAIN_USER = 'https://apps-apis.google.com/a/feeds/%s/user/2.0/%s';

  /**
   * @const ATOM_NAMESPACE la URL del namespace de los pedidos y consultas basados en el estándar de Google ATOM
   */
  const ATOM_NAMESPACE = 'http://www.w3.org/2005/Atom';

  /**
   * @const GAPPS_NAMESPACE la URL del espacio de nombres empleado para algunas solicitudes de GoogleApps
   */
  const GAPPS_NAMESPACE = 'http://schemas.google.com/apps/2006';

  /**
   * @const LOGIN_HOSTED_OR_GOOGLE una constante que Google emplea como valor en ciertos pedidos
   */
  const LOGIN_HOSTED_OR_GOOGLE = 'HOSTED_OR_GOOGLE';

  /**
   * @const GAPP_SERVICE una constante que Google emplea para identificar su servicio corporativo
   */
  const GAPP_SERVICE = 'apps';

  /**
   * @const SOURCE_LOG la denominación que se ha seleccionado como nombre del sistema de aprovisionamiento para GoogleApps
   */
  const SOURCE_LOG = 'iplan-autoprovisioning2-0.1Beta';

  /**
   * @var string el ID de la sesión con los servicios de Google
   */
  private $SID = '';

  /**
   * @var string un token de autenticación de la antigua API. No se utiliza, al menos por el momento
   */
  private $LSID = '';

  /**
   * @var string el token de autorización que brinda GoogleApps, con vigencia de 24 hs.
   */
  private $AUTH = '';

  /**
   * @var boolean un flag que indica si se está o no debugeando los llamados
   */
  private $DEBUG = false;

  /**
   * @var string una variable donde dejo temporalmente los mensajes de error del último llamado
   */
  private $lastError;

  /**
   * @var string temporariamente mantiene el registro del código de error del último llamado
   */
  private $lastErrorCode;

  /**
   * @var array el mismo arreglo que devuelve el makeCall
   */
  private $lastCall;

  /**
   * @var ApplicationContext el contexto
   */
  private $context;

  /**
   * @var ORM una instancia del ORM para usar a fines de loguear información
   */
  private $orm;

  /**
   * @var LogMessages una instancia de la clase de log
   */
  private $log;

  /**
   * @var string el dominio reseller
   */
  private $reseller;

  /**
   * @var int el último código de proceso que se emplea por default en caso que la transacción no tenga una novedad asociada.
   */
  private $lastProcessId;

  /**
   * @var GoogleTransaction la transacción actual para la que se deben registrar los logs.
   */
  private $transaction;

  /**
   * Hace un llamado a una función de la API de Google.
   * 
   * @param string $url la dirección a la cual remitir el llamado
   * @param string $body el cuerpo del mensaje que debe ser enviado.
   * @param string $method el tipo de codificación empleada en el llamado, puede ser POST, GET o PUT
   * @param array $headers encabezados adicionales, en forma de una lista de strings que serán ubicados adecuadamente en el llamado a la función.
   * 
   * @return array devuelve un arreglo con información pertinente al resultado del llamado. La estructura del mismo contiene los headers de la respuesta empleando el nombre como clave (ejemplo $respuesta['Content-Type'] y además los siguientes elementos:
   * array (
   *    'protocol': el protocolo a través del cual se invocó el llamado: http, https, etc...
   *    'status': el código de estado devuelto por el servidor: 400, 200, 404, 502, etc...
   *    'description': la descripción literal del código de estado.
   *    'body': el texto del cuerpo de la respuesta
   *    'xml': un objeto de tipo DOMXPath tradicional, si se detecta que se trata de un XML
   *    'atom': un objeto de tipo DOMXPath preparado para ATOM, si se detecta que es el caso
   * )
   */
  protected function makeCall($url, $body = '', $method = 'POST', $headers = null)
  {
    // Bouml preserved body begin 001B5D05
	if ($this->DEBUG) var_dump($url, $body, $method, $headers);
	$this->lastError = false;
	$this->lastErrorCode = false;

	$urlData = parse_url($url);
	$resource  = $urlData['path'];
	if (isset ($urlData['query']))
		$query  = $urlData['query'];
	else $query='';
	$host = $urlData['host'];
	//$ipHost = gethostbyname($host);
	$defaultPorts = array('http'=>80, 'https'=>443, 'ftp'=>21);
	$defaultPort = isset($urlData['port'])? $urlData['port'] : (isset($defaultPorts[$urlData['scheme']])?$defaultPorts[$urlData['scheme']]:80);
	$tamanio = strlen($body);
	$request = "$method $resource".((($method=='GET')&&($query!=''))?"?$query":'')." HTTP/1.1\r\n"
			  ."Host: $host\r\n";
	if (($headers !== null) && (is_array($headers)))
		foreach ($headers as $name=>$value)
			$request .= "$name:$value\r\n";
	if ((!isset($headers['Content-type'])) && (!isset($headers['Content-Type']))) {
		$request  .="Content-type: application/atom+xml\r\n";
	}
	$request  .=(($this->AUTH=='')?'':'Authorization: GoogleLogin auth='.$this->AUTH."\r\n")
			  ."Content-Length: $tamanio\r\n"
			  ."Connection: close\r\n"
			  ."\r\n";
	if ($body !== '')
		$request .="$body\r\n";

	if (isset($urlData['scheme']) && (($urlData['scheme']=='https') || ($urlData['scheme']=='ssl')))
		$socketHost='ssl://';
	else $socketHost='';
	if ($this->DEBUG) echo "<pre>Conectando a $socketHost$host:$defaultPort</pre>";
	if ($this->DEBUG) echo "<pre>$request</pre>";
	
	//Logueo el request
	$this->saveFileLog($request, 2, 27);
	$errno='';
	$errstring = '';
	$socket = fsockopen($socketHost.$host,$defaultPort, $errno, $errstring); 
	if (!$socket) {
		throw new Exception("$errstring ($errno) al acceder a $url");
	} else {
		fputs($socket, $request);
		$result='';
		while (!feof($socket)) {
		  $result .= fread($socket, 8192);
		}
		fclose($socket);

		//Logueo el response
		$this->saveFileLog($result, 2, 28);

		if ($this->DEBUG) echo "Resultado:<pre>$result</pre>";
		$response = $this->parseResponse($result);
		$response['request']=$request;
		switch(true) {
			case ((isset($response['Content-Type'])) && (preg_match('/^\s*(application\/atom\+xml).*/', $response['Content-Type']))):
			case ((isset($response['Content-type'])) && (preg_match('/^\s*(application\/atom\+xml).*/', $response['Content-type']))):
				$response['atom']=$this->parseBody($response['body'], 'atom');
				//break;//Si es Atom, también es xml
			case ((isset($response['Content-type'])) && (preg_match('/^\s*(text\/xml).*/', $response['Content-type']))):
			case ((isset($response['Content-Type'])) && (preg_match('/^\s*(text\/xml).*/', $response['Content-Type']))):
				$response['xml']=$this->parseBody($response['body'], 'xml');
				break;
		}
		$this->lastCall=$response;
		return $response;
	}
    // Bouml preserved body end 001B5D05
  }

  /**
   * Función auxiliar que parsea una respuesta del servidor y devuelve un array con ciertos componentes y los headers de la respuesta.
   * 
   * @param string $response el texto plano de la respuesta del servidor.
   */
  protected function parseResponse($response)
  {
    // Bouml preserved body begin 001B5D85
	$matches = null;
	preg_match('/\\r\\n\\r\\n/', $response, $matches, PREG_OFFSET_CAPTURE);
	$pos = $matches[0][1];
	$header= substr($response, 0, $pos);
	$body  = substr($response, $pos+4);
	$headerLines = preg_split('/\\r\\n/', $header);
	$ok = preg_match('/([\S]+)\s+(\S+)\s*(.*)/', $headerLines[0], $matches);
	if ($ok) {
		list($_, $protocol, $statusCode, $statusDescription)=$matches;
		$result['protocol']=$protocol;
		$result['status']=$statusCode;
		$result['description']=$statusDescription;
	} else {
		throw new Exception('Error al parsear el status');
	}
	for($i=1;$i<count($headerLines);$i++) {
		$ok = preg_match('/([\S]+):(.*)/', $headerLines[$i], $matches);
		if ($ok) {
			$result[$matches[1]]=$matches[2];
		} else {
			throw new Exception('Error al parsear los headers, alguno tiene un formato incorrecto');
		}
	}
	$result['body']=$body;
	return $result;
    // Bouml preserved body end 001B5D85
  }

  /**
   * Función auxiliar que dado el cuerpo de una respuesta en XML retorna un objeto DOMXPath para trabajar con él.
   * 
   * @param string $text el cuerpo en texto plano.
   * @param string $type el tipo detectado del documento
   * 
   * @return DOMXPath una instancia preparada para trabajar en forma más amena con la respuesta.
   */
  protected function parseBody($text, $type)
  {
    // Bouml preserved body begin 001B5E05
	$dom = new DOMDocument();
	$dom->loadXML($text);
	$xpath = new DOMXPath($dom);
	switch($type) {
		case 'atom':
			$xpath->registerNamespace('atom', self::ATOM_NAMESPACE);
			$xpath->registerNamespace('apps', self::GAPPS_NAMESPACE);
			return $xpath;
			break;
		case 'xml':
			return $xpath;
			break;
		default:
			throw new Exception('Contenido no soportado');
	}
    // Bouml preserved body end 001B5E05
  }

  /**
   * Chequea si un dominio es factible de ser empleado por un cliente.
   * 
   * @param string $domain el dominio a verificar
   * 
   * @return string La función retorna una cadena de texto que puede ser:
   * ES_DE_IPLAN: si la cuenta existe y está administrada (comercialmente) por IPlan
   * NO_ES_DE_IPLAN: si la cuenta existe pero administrada por un tercero (posiblemente Google). Antigüamente se intentó discriminar entre administradas por Google y por otros Resellers, pero las API no permiten conocer dicha información.
   * NO_EXISTE: el dominio no tiene creada una cuenta en GoogleApps
   */
  public function validateClientDomain($domain)
  {
    // Bouml preserved body begin 001B5F85
	$data = $this->makeCall(sprintf(self::URL_BASE_DOMAIN, $this->reseller, $domain), '', 'GET');
	switch ($data['status']) {
		case 200:
			$found = $data['atom']->query('//atom:entry/apps:property');//[@name="domainName"]
			switch ($found->length) {
				case 5://Posiblemente exista y es nuestro
				case 1://Posiblemente exista y sea de otro
						$domainNames = $data['atom']->query('//atom:entry/apps:property[@name="domainName"]');
						if (($domainNames->length==1) && ($domainNames->item(0)->getAttribute('value')==$domain)){
							$editions = $data['atom']->query('//atom:entry/apps:property[@name="edition"]');
							return (($editions->length == 1)?'ES_DE_IPLAN':'NO_ES_DE_IPLAN');
						} else return 'validateClientDomain:: Respuesta no esperada [status:'.$data['status']."]\n".$data['body'];
						break;
				default:
			}
			break;
		case 400://Es estándar o no existe
				$errors = $data['xml']->query('//AppsForYourDomainErrors/error');
				if ($errors->length > 0) {
					$isStandard = true;
					for($i=0;$i<$errors->length;$i++) {
						if (($errors->item($i)->getAttribute('reason')=='EntityDoesNotExist') && 
							($errors->item($i)->getAttribute('errorCode')=='1301'))
								$isStandard=false;
								break;
					}
					return ($isStandard?'ES_STANDARD':'NO_EXISTE');
				} else return 'validateClientDomain:: Respuesta no esperada [status:'.$data['status']."]\n".$data['body'];
			break;
		default:
			$this->lastError = 'validateClientDomain::La acción de autenticación no fue permitida. Respuesta del servidor: ['.$data['status']."]\n".$data['body'];
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001B5F85
  }

  public function createDomainUser($domain, $username, $password, $name, $family, $admin)
  {
    // Bouml preserved body begin 001BAD05
	$request = 
'<?xml version="1.0" encoding="UTF-8"?>
<atom:entry xmlns:atom="http://www.w3.org/2005/Atom" xmlns:apps="http://schemas.google.com/apps/2006">
<atom:category scheme="http://schemas.google.com/g/2005#kind" term="http://schemas.google.com/apps/2006#user"/>
  <apps:login userName="'.$username.'" password="'.$password.'" admin="'.($admin?'true':'false').'"/>
  <apps:name familyName="'.$family.'" givenName="'.$name.'"/>
</atom:entry>';
	$data = $this->makeCall(sprintf(self::URL_USER_QUERY, $domain), $request, 'POST');
	switch ($data['status']) {
		case 201:
			return true;
			break;
		case 400:
			if ($data['xml']) {
				$errors = $data['xml']->query("//AppsForYourDomainErrors/error");
				if ($errors->length > 0) {
					$errorCode = $errors->item(0)->getAttribute('errorCode');
					$invalidInput = $errors->item(0)->getAttribute('invalidInput');
					$reason = $errors->item(0)->getAttribute('reason');
					switch($errorCode) {
						case '1301':
							$this->lastError = 'El dominio aún no está listo para crear usuarios';
							$this->lastErrorCode = $errorCode;
							break;
						default:
						$this->lastError = "Error en el valor \"$invalidInput\". Causa: $reason";
						$this->lastErrorCode = $errorCode;
					}
				} else {
					$this->lastError = 'Error general';
					$this->lastErrorCode = '';
				}
			} else {
				$this->lastError = 'Error general';
				$this->lastErrorCode = '';
			}
			break;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001BAD05
  }

  /**
   * Crea una cuenta de GoogleApps. Es necesario que antes de invocar a esta función se hayan realizadas las validaciones pertinentes.
   * 
   * @param string $domain el dominio que se va a emplear en la cuenta
   * @param int $licences la cantidad de licencias que se le va a asignar a la cuenta
   * @param string $countryCode el código de país
   * 
   * @return boolean Un true si la creación de cuenta fue exitosa o false en caso contrario. También es posible que arroje excepciones ante inconvenientes operativos.
   */
  public function createCustomerDomain($domain, $licences, $countryCode)
  {
    // Bouml preserved body begin 001B6005
	$request = 
'<?xml version="1.0" encoding="UTF-8"?>
<atom:entry xmlns:atom="http://www.w3.org/2005/Atom" xmlns:apps="http://schemas.google.com/apps/2006">
  <apps:property name="domainName" value="'.$domain.'"/>
  <apps:property name="edition" value="premier"/>
  <apps:property name="maximumNumberOfUsers" value="'.$licences.'"/>
  <apps:property name="countryCode" value="'.$countryCode.'"/>
</atom:entry>';
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_CREATE, $this->reseller), $request, 'POST');
	switch ($data['status']) {
		case 201:
			return true;
			break;
		case 400:
			if ($data['xml']) {
				$errors = $data['xml']->query("//AppsForYourDomainErrors/error");
				if ($errors->length > 0) {
					$errorCode = $errors->item(0)->getAttribute('errorCode');
					$invalidInput = $errors->item(0)->getAttribute('invalidInput');
					$reason = $errors->item(0)->getAttribute('reason');
					switch($errorCode) {
						case '1303':
							$this->lastError = "Error en el nombre del dominio \"$invalidInput\". Causa $reason: no se pueden consignar dominios que empleen marcas registradas de Google en su denominación";
							$this->lastErrorCode = $errorCode;
							break;
						case '1800':
							$this->lastError = "Error inesperado, Google no permite que al dominio $domain se le asigne la licencia Premier. Causa $reason. También puede ser que el reseller no soporte el tipo de licencia.";
							$this->lastErrorCode = $errorCode;
							break;
						default:
							$this->lastError = "Error en el valor \"$invalidInput\". Causa: $reason";
							$this->lastErrorCode = $errorCode;
					}
				} else {
					$this->lastError = 'Error general';
					$this->lastErrorCode = '';
				}
			} else {
				$this->lastError = 'Error general';
				$this->lastErrorCode = '';
			}
			break;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001B6005
  }

  /**
   * Dado un dominio, chequea con Google si el mismo ya fue verificado por sus sistemas.
   * 
   * @param string $domain el dominio a verificar
   * 
   * @return boolean Retorna true si el dominio ya ha sido verificado satisfactoriamente o false en caso contrario. Además, si el pedido falla arroja distintas excepciones.
   */
  public function verifyDomain($domain)
  {
    // Bouml preserved body begin 001B5F05
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_VERIFICATION, $domain), '', 'GET');
	switch($data['status']) {
		case 200:
			$found = $data['atom']->query('//atom:entry/apps:property[@name="isVerified"]');
			if ($found->length > 0) {
				return $found->item(0)->getAttribute('value') == 'true';
			} else {
				$this->lastError = 'verifyDomain:: La respuesta de google no fue la esperada';
				$this->lastErrorCode = '';
				throw new Exception('verifyDomain:: La respuesta de google no fue la esperada');
			}
			break;
		case 403:
			return false;
		default:
			$this->lastError = 'verifyDomain:: La acción de autenticación no fue permitida. Respuesta del servidor: ('.$data['status'].') '.$data['description'];
			$this->lastErrorCode = '';
			throw new Exception('verifyDomain:: La acción de autenticación no fue permitida. Respuesta del servidor: ('.$data['status'].') '.$data['description']);
	}
    // Bouml preserved body end 001B5F05
  }

  /**
   * Devuelve la cantidad de usuarios configurados en el dominio.
   * 
   * @param string $domain el dominio del cliente
   * 
   * @return int Devuelve la cantidad de usuarios del dominio indicado.
   */
  public function countDomainUsers($domain)
  {
    // Bouml preserved body begin 001C3585
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_GENERAL.'currentNumberOfUsers', $domain), '', 'GET');
	switch ($data['status']) {
		case 200:
			$found = $data['atom']->query('//atom:entry/apps:property[@name="currentNumberOfUsers"]');//
			if ($found->length >0) {
				return $found->item(0)->getAttribute('value');
			} else return -1;
			break;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return -1;
	}
    // Bouml preserved body end 001C3585
  }

  /**
   * Devuelve la capacidad máxima de usuarios contratados en el dominio.
   * 
   * @param string $domain el dominio del cliente
   * 
   * @return int Devuelve la cantidad de usuarios que puede albergar el dominio según la contratación actual.
   */
  public function getMaxNumberOfLicences($domain)
  {
    // Bouml preserved body begin 001C3605
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_GENERAL.'maximumNumberOfUsers', $domain), '', 'GET');
	switch ($data['status']) {
		case 200:
			$found = $data['atom']->query('//atom:entry/apps:property[@name="maximumNumberOfUsers"]');//
			if ($found->length >0) {
				return $found->item(0)->getAttribute('value');
			} else return -1;
			break;
		default:
			return -1;
	}
    // Bouml preserved body end 001C3605
  }

  /**
   * Valida las credenciales del reseller y prepara a la instancia para trabajar con la API de google.
   * 
   * @param string $email el mail de la cuenta administradora de la entidad reseller
   * @param string $password la contraseña del administrador
   * 
   * @return boolean devuelve true si todo estuvo bien o arroja una excepción si hubo un error.
   */
  public function registerInGoogleApps($email, $password)
  {
    // Bouml preserved body begin 001B5E85
	$val = new MailValidation();
	if ($val->validate($email)) {
		$parts = explode('@', $email);
		$this->reseller = $parts[1];
	} else {
		if ($this->log) $this->log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 17, 'El mail provisto como administrador del Reseller es inválido.');
		return false;
	}
	$body = '&Email='.urlencode($email)."&Passwd=$password&accountType=".
			   self::LOGIN_HOSTED_OR_GOOGLE.'&service='.self::GAPP_SERVICE.'&source='.self::SOURCE_LOG;
	$data = $this->makeCall(self::URL_AUTHENTICATION_TOKEN, $body, 'POST', array('Content-type'=>'application/x-www-form-urlencoded; charset=UTF-8'));
	if ($data['status']==200) {
		$matches=null;
		$ok = preg_match('/SID=(?P<SID>\S*)[\\r\\n]+LSID=(?P<LSID>\S*)[\\r\\n]+Auth=(?P<Auth>\S*)[\r\n]+/', $data['body'], $matches);
		if ($ok) {
			$this->SID =$matches['SID'];
			$this->LSID=$matches['LSID'];
			$this->AUTH=$matches['Auth'];
			return true;
		} else {
			if ($this->log) $this->log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 17, 'No se pudieron identificar correctamente los tokens de autorización');
		}
	} else {
		if ($this->log) $this->log->AddLog(LogMessages::LOG_ERROR, LogMessages::SYS_CAGA, 17, 'La acción de autenticación no fue permitida. Respuesta del servidor: ('.$data['status'].') '.$data['description']);
	}
	return false;
    // Bouml preserved body end 001B5E85
  }

  /**
   * Lista los usuarios de una cuenta. Opcionalmente se puede filtrar por aquellos usuarios que sean administradores.
   * 
   * @param string $domain el dominio del cliente
   * @param boolean $filterByAdmins un flag que indica si se desea o no que se recuperen sólo las cuentas administradoras.
   * 
   * @return array|false Devuelve un arreglo indexado donde cada posición tiene un arreglo con el nombre del usuario (username), un flag que indica si es o no administrador (admin) y el nombre completo del usuario (name y surname). Si ocurriera algún error devuelve false.
   */
  public function listDomainUsers($domain, $filterByAdmins = false)
  {
    // Bouml preserved body begin 001C3505
	$users = array();
	$next = sprintf(self::URL_USER_QUERY, $domain);
	while($next) {
		$data = $this->makeCall($next, '', 'GET');
		switch ($data['status']) {
			case 200:
				if ($filterByAdmins) {
					$found = $data['atom']->query('//atom:entry/apps:login[@admin="true"]/..');
				} else {
					$found = $data['atom']->query('//atom:entry');
				}
				if ($found->length >0) {
					for($i=0; $i<$found->length;$i++) {
						$loginNode=$data['atom']->query('apps:login', $found->item($i));
						$login= $loginNode->item(0);
						$namesNode=$data['atom']->query('apps:name', $found->item($i));
						$names= $namesNode->item(0);
						$users[]=array('username'=>$login->getAttribute('userName'), 
									   'admin'=>$login->getAttribute('admin')=='true',
									   'name'=>$names->getAttribute('givenName'),
									   'surname'=>$names->getAttribute('familyName'),
									   'suspended'=>$login->getAttribute('suspended')=='true',
									   'quota' => $data['atom']->query('apps:quota', $found->item($i))->item(0)->getAttribute('limit'));
					}
				}
				break;
			default:
				$this->lastError = 'Error general';
				$this->lastErrorCode = '';
				return false;
		}
		$next = $data['atom']->evaluate('string(//atom:link[@rel="next"]/@href)');
	}
	return $users;
    // Bouml preserved body end 001C3505
  }

  /**
   * Promociona una cuenta de GoogleApps Standard existente a Premium.
   * 
   * @param string $domain el dominio que se va a emplear en la cuenta
   * @param int $licences la cantidad de licencias que se le va a asignar a la cuenta
   * @param string $countryCode el código de país
   * @param string $transferToken el código brindado por el usuario para realizar la actualización
   * @return boolean Un true si la actualización de la cuenta fue exitosa o false en caso contrario. También es posible que arroje excepciones ante inconvenientes operativos.
   */
  public function upgradeStandardToPremium($domain, $licences, $countryCode, $transferToken)
  {
    // Bouml preserved body begin 001C3485
	$request = 
'<?xml version="1.0" encoding="UTF-8"?>
<atom:entry xmlns:atom="http://www.w3.org/2005/Atom" xmlns:apps="http://schemas.google.com/apps/2006">
  <apps:property name="domainName" value="'.$domain.'"/>
  <apps:property name="edition" value="premier"/>
  <apps:property name="maximumNumberOfUsers" value="'.$licences.'"/>
  <apps:property name="countryCode" value="'.$countryCode.'"/>
  <apps:property name="transferToken" value="'.$transferToken.'"/>
</atom:entry>';
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_CREATE, $this->reseller), $request, 'POST');
	switch ($data['status']) {
		case 201:
			return true;
			break;
		case 400:
			if (isset($data['xml'])) {
				$errors = $data['xml']->query('//AppsForYourDomainErrors/error');
				if ($errors->length > 0) {
					$error = $errors->item(0);
					$this->lastError = $error->getAttribute('reason').': '.$error->getAttribute('invalidInput');
					$this->lastErrorCode = $error->getAttribute('errorCode');
				}
			} else {
				$this->lastError = $data['body'];
				$this->lastErrorCode = '';
			}
			return false;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001C3485
  }

  /**
   * Disminuye la cantidad de cuentas en una cierta cantidad.
   * 
   * @param string $domain el dominio que se va a emplear en la cuenta
   * @param int $licences la cantidad de licencias que se le va a asignar a la cuenta
   * 
   * @return boolean Un true si la actualización de la cuenta fue exitosa o false en caso contrario. También es posible que arroje excepciones ante inconvenientes operativos.
   */
  public function setMaxNumberOfLicences($domain, $licences)
  {
    // Bouml preserved body begin 001C5405
$request = '<entry xmlns="http://www.w3.org/2005/Atom" xmlns:apps="http://schemas.google.com/apps/2006">
<apps:property name="maximumNumberOfUsers" value="'.$licences.'"/>
</entry>';
	$data = $this->makeCall(sprintf(self::URL_BASE_DOMAIN.'/maximumNumberOfUsers', $this->reseller, $domain), $request, 'PUT');
	switch ($data['status']) {
		case 200:
		case 201:
			return true;
			break;
		case 400:
			if (isset($data['xml'])) {
				$errors = $data['xml']->query('//AppsForYourDomainErrors/error');
				if ($errors->length > 0) {
					$error = $errors->item(0);
					$this->lastError = $error->getAttribute('reason').': '.$error->getAttribute('invalidInput');
					$this->lastErrorCode = $error->getAttribute('errorCode');
				}
			} else {
				$this->lastError = $data['body'];
				$this->lastErrorCode = '';
			}
			return false;
		case 403:
			if ($data['Content-Type'] == 'text/html') {
				if (strpos($data['body'], 'Invalid domain')) {
					$this->lastError = 'Dominio Inválido';
					$this->lastErrorCode = 'Error 403';					
				} else {
					$this->lastError = $data['body'];
					$this->lastErrorCode = 'Error 403';					
				}
			} else {
				$this->lastError = $data['body'];
				$this->lastErrorCode = '';
			}
			break;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001C5405
  }

  /**
   * Desvincula la cuenta del reseller.
   * 
   * @param string $domain el dominio que se va a emplear en la cuenta
   * 
   * @return boolean Un true si la actualización de la cuenta fue exitosa o false en caso contrario. También es posible que arroje excepciones ante inconvenientes operativos.
   */
  public function downgradeAccount($domain)
  {
    // Bouml preserved body begin 001C5485
$request = '<entry xmlns="http://www.w3.org/2005/Atom" xmlns:apps="http://schemas.google.com/apps/2006">
<apps:property name="edition" value="standard"/>
</entry>';
	$data = $this->makeCall(sprintf(self::URL_BASE_DOMAIN.'/edition', $this->reseller, $domain), $request, 'PUT');
	switch ($data['status']) {
		case 200:
		case 201:
			return true;
			break;
		default:
			$this->lastError = 'Error general';
			$this->lastErrorCode = '';
			return false;
	}
    // Bouml preserved body end 001C5485
  }

  /**
   * Recupera los dominios del reseller.
   * 
   * @param $from string el dominio a partir del cual se solicita el listado
   * @return array|false La lista de dominios o false si algo falló.
   */
  public function listDomains($from = '')
  {
    // Bouml preserved body begin 001C5505
	$domains = array();
	if ($from == '') {
		$next = sprintf('https://apps-apis.google.com/a/feeds/reseller/2.0/%s/domain', $this->reseller);
	} else {
		$next = sprintf('https://apps-apis.google.com/a/feeds/reseller/2.0/%s/domain/%s', $this->reseller, $from);
	}
	while($next) {
		$data = $this->makeCall($next, '', 'GET');
		//echo $data['body']."\n";
		switch ($data['status']) {
			case 200:
				$found = $data['atom']->query('//atom:entry');
				if ($found->length >0) {
					if ($next == sprintf('https://apps-apis.google.com/a/feeds/reseller/2.0/%s/domain', $this->reseller)) {
						$start = 0;
					} else {
						$start = 1;
					}
					for($i=$start; $i<$found->length;$i++) {
						$domain=$data['atom']->query('apps:property[@name="domainName"]', $found->item($i));
						$domains[]=$domain->item(0)->getAttribute('value');
					}
				}
				break;
			default:
				$this->lastError = 'Error general';
				$this->lastErrorCode = '';
				return false;
		}
		$next = $data['atom']->evaluate('string(//atom:link[@rel="next"]/@href)');
	}
	return $domains;
    // Bouml preserved body end 001C5505
  }

  /**
   * Devuelve el texto correspondiente al mensaje de error del último llamado.
   * 
   * @return string|false El mensaje de error o false si no se produjo ningún error
   */
  public function getLastError()
  {
    // Bouml preserved body begin 001C8C05
	return $this->lastError;
    // Bouml preserved body end 001C8C05
  }

  /**
   * Devuelve el texto correspondiente al código de error del último llamado.
   * 
   * @return string|false El código de error o false si no se produjo ningún error
   */
  public function getLastErrorCode()
  {
    // Bouml preserved body begin 001C8C85
	return $this->lastErrorCode;
    // Bouml preserved body end 001C8C85
  }

  /**
   * Devuelve el arreglo completo resultado del último makeCall()
   * 
   * @return array El arreglo del último makeCall()
   */
  public function getLastCall()
  {
    // Bouml preserved body begin 001D0285
	return $this->lastCall;
    // Bouml preserved body end 001D0285
  }

  /**
   * Inicializa algunas variables necesarias para integrar la API con el sistema de Logs
   */
  public function initLog(&$context = null, &$orm = null, &$log = null)
  {
    // Bouml preserved body begin 001D0385
	$this->context = $context;
	$this->log = $log;
	$this->orm = $orm;
	return $this;
    // Bouml preserved body end 001D0385
  }

  /**
   * @param boolean $enable true para activalo
   * @return APIGoogleApps
   */
  public function setDebug($enable)
  {
    // Bouml preserved body begin 001D0485
	$this->DEBUG = $enable;
	return $this;
    // Bouml preserved body end 001D0485
  }

  public function getDomainInfo($domain)
  {
    // Bouml preserved body begin 001D0505
	$data = $this->makeCall(sprintf(self::URL_BASE_DOMAIN, $this->reseller, $domain), '', 'GET');
	$result['domain']=$domain;
	$result['domain2']='';
	$result['edition']='';
	$result['licences']='';
	$result['used']='';
	$result['created']='';
	$result['country']='';
	switch ($data['status']) {
		case 200:
			$result['edition'] = $data['atom']->evaluate('string(//atom:entry/apps:property[@name="edition"]/@value)');
			$result['licences'] = $data['atom']->evaluate('string(//atom:entry/apps:property[@name="maximumNumberOfUsers"]/@value)');
			$result['created'] = $data['atom']->evaluate('string(//atom:entry/apps:property[@name="creationTime"]/@value)');
			$result['country'] = $data['atom']->evaluate('string(//atom:entry/apps:property[@name="countryCode"]/@value)');
			$result['domain2'] = $data['atom']->evaluate('string(//atom:entry/apps:property[@name="domainName"]/@value)');
			$result['used']=$this->countDomainUsers($domain);
			break;
		default:
				$this->lastError = 'Error general - Status '.$data['status'];
				$this->lastErrorCode = $data['status'];
				return false;
	}
	return $result;
    // Bouml preserved body end 001D0505
  }

  /**
   * Establece el último nro. de proceso para el caso que la transacción no haya venido de Tenfold.
   */
  public function setLastProcessId($process)
  {
    // Bouml preserved body begin 001E0685
	$this->lastProcessId = $process;
	return $this;
    // Bouml preserved body end 001E0685
  }

  /**
   * Guarda un documento como parte del log.
   * 
   * @param string|array $contents un string que representa el contenido del archivo o un arreglo de strings donde cada uno representa un contenido de archivo distinto.
   * @param int $contentType el tipo de mensaje (XML, HTTP, etc...) en el que está dado el/los string/s.
   * @param int $messageType la clasificación interna del mensaje (request/response).
   * 
   * @return boolean Devuelve True si todo anduvo bien, False en caso contrario.
   */
  public function saveFileLog($contents, $contentType, $messageType)
  {
    // Bouml preserved body begin 001D0305
	if (is_null($this->log)) return false;
	
	$serviceOrder=null;
	if ($this->transaction)
		$serviceOrder = $this->transaction->getNotification();
	$context = $this->context;
	$process = (!is_null($serviceOrder))?$serviceOrder->getProcess()->getId():$this->lastProcessId;
	$sellOrder = (!is_null($serviceOrder))?$serviceOrder->getSellOrder():'';
	$taskId = (!is_null($serviceOrder))?$serviceOrder->getId():'';
	
	if (is_null($process)) return false;
	
	$conn = $this->orm->getDatabase()->getConnection();
	
	if (is_string($contents)) {
		$contents=array($contents);
	}
	foreach($contents as $content) {
		//Agrego el contenido a la tabla de archivos
		$params = array("p_user_id" => $context->getUser()->getId(), "p_clob" => $content, "p_t_xml_id" => $messageType, 'p_t_mensaje_id' => $contentType);
		$result = null;
		$lala = $conn->executeClobPackage("PKG_CAP_MENSAJEXML.FU_MENSAJEXML_INS", $params, "p_clob", $result, Connection::T_CURSOR);
		$asd = $conn->fetch($lala);
		$conn->free($lala);
		if ( (!is_array($asd)) || (( isset($asd["SQLCODE_ERROR"])) && ($asd["SQLCODE_ERROR"] != "200")) ) {
			$desc = (is_array($asd)) ? "[SQLCODE_ERROR] => " . $asd["SQLCODE_ERROR"] . " [FUNCION] => " . $asd["FUNCION"] . " [SQLERRM_ERROR] => " . $asd["SQLERRM_ERROR"] : "";
			$this->log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 5, "PKG_CAP_MENSAJEXML.FU_MENSAJEXML_INS $desc", $sellOrder, $process, $taskId);
			trigger_error("PKG_CAP_MENSAJEXML.FU_MENSAJEXML_INS", E_USER_WARNING);
		} else {
			$content_id = $asd["LASTID"];
		}
		//Agrego la relación con la Orden de Venta
		if ($sellOrder) {
			$params = array("p_user_id" => $context->getUser()->getId(), "p_orden" => $sellOrder, "p_xml_id" => $content_id);
			$result = null;
			$lala = $conn->executeFunction("PKG_CAP_ORDENESXML.FU_ORDENESXML_INS", $params, $result, Connection::T_CURSOR);
			$asd = $conn->fetch($lala);
			$conn->free($lala);
			if ( (!is_array($asd)) || (( isset($asd["SQLCODE_ERROR"])) && ($asd["SQLCODE_ERROR"] != "200")) ) {
				$desc = (is_array($asd)) ? "[SQLCODE_ERROR] => " . $asd["SQLCODE_ERROR"] . " [FUNCION] => " . $asd["FUNCION"] . " [SQLERRM_ERROR] => " . $asd["SQLERRM_ERROR"] : "";
				$this->log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 5, "PKG_CAP_ORDENESXML.FU_ORDENESXML_INS $desc", $sellOrder, $process, $taskId);
				trigger_error("PKG_CAP_ORDENESXML.FU_ORDENESXML_INS", E_USER_ERROR);
			}
		}
		//Agrego la relación con la ServiceOrder
		$params = array("p_user_id" => (int)$context->getUser()->getId(), "p_serviceordertaskid" => (int)$taskId, "p_xml_id" => (int)$content_id);
		$result = null;
		$lala = $conn->executeFunction("PKG_CAP_SERVICEORDERTASKS.FU_SERVICEORDERTASKXML_INS", $params, $result, Connection::T_CURSOR);
		$asd = $conn->fetch($lala);
		$conn->free($lala);
		if ( (!is_array($asd)) || (( isset($asd["SQLCODE_ERROR"])) && ($asd["SQLCODE_ERROR"] != "200")) ) {
			$desc = LogMessages::erroresArray($asd) . " P: " . LogMessages::erroresArray($params);
			$this->log->AddLog(LogMessages::LOG_INFORMATION, LogMessages::SYS_CAGA, 4, "PKG_CAP_SERVICEORDERTASKS.FU_SERVICEORDERTASKXML_INS $desc", $sellOrder, $process, $taskId);
			trigger_error("PKG_CAP_SERVICEORDERTASKS.FU_SERVICEORDERTASKXML_INS", E_USER_ERROR);
		}
	}
	return true;
    // Bouml preserved body end 001D0305
  }

  /**
   * Configura la transacción en curso a fines de registrarla en los logs.
   * 
   * @param GoogleTransaction $transaction la transacción en la cual se deben registrar los logs.
   * 
   * @return APIGoogleApps La API con el valor registrado.
   */
  public function setTransaction(&$transaction)
  {
    // Bouml preserved body begin 001E0785
	$this->transaction = $transaction;
	return $this;
    // Bouml preserved body end 001E0785
  }

  /**
   * Cambia el password de un usuario perteneciente a algún dominio administrado por el reseller.
   * 
   * @param string $domain el dominio al cual pertenece el usuario
   * @param string $username el nombre de usuario que se quiere modificar
   * @param string $password el resultado de aplicar la función de hash a la clave del usuario
   * @param string $hash el algoritmo de hash empleado, por default es SHA-1
   * 
   * @return boolean La función retorna TRUE si el cambio se realizó con éxito y FALSE si no.
   */
  public function changeUserPassword($domain, $username, $password, $hash = 'SHA-1')
  {
    // Bouml preserved body begin 001ECB05
	//PUT https://apps-apis.google.com/a/feeds/domain/user/2.0/userName
$request = '<?xml version="1.0" encoding="UTF-8"?>
<atom:entry xmlns:atom="http://www.w3.org/2005/Atom"  xmlns:apps="http://schemas.google.com/apps/2006">
 <atom:category scheme="http://schemas.google.com/g/2005#kind"  term="http://schemas.google.com/apps/2006#user"/>
 <apps:login password="'.$password.'" hashFunctionName="'.$hash.'"/>
</atom:entry>';
	$data = $this->makeCall(sprintf(self::URL_DOMAIN_USER, $domain, $username), $request, 'PUT');
	switch ($data['status']) {
		case 200:
			return true;
			break;
		default:
			$this->lastError = 'Error general - Status '.$data['status'];
			$this->lastErrorCode = $data['status'];
			return false;
	}

    // Bouml preserved body end 001ECB05
  }

}
?>