<?php
require_once 'iplan/database/Connection.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Concreci�n de una conexi�n Oracle
 */
class OracleConnection extends Connection {
  /**
   * @const string Una constante que indica que la conexi�n se efectua por host:port y sid
   */
  const ORACLE_DIRECT_CONNECTION = "ORACLE_DIRECT_CONNECTION";

  /**
   * @const string Esta constante significa que hay un archivo de conexi�n previamente configurado en el server y que se puede conectar por nombre.
   */
  const ORACLE_BYNAME_CONNECTION = "ORACLE_BYNAME_CONNECTION";

  private $host;

  private $port;

  /**
   * @var string El identificador de "Servicio".
   */
  private $sid;

  private $schema;

  private $user;

  private $password;

  /**
   * Un arreglo que permite un acceso r�pido a los tipos de datos nativos
   */
  private static $oraTypes = array(
	Connection::T_BLOB => SQLT_BLOB,
	Connection::T_BINARY =>  SQLT_BIN,
	Connection::T_BINARY_FILE => SQLT_BFILEE,
	Connection::T_CBLOB => SQLT_CLOB,
	Connection::T_CHAR => SQLT_CHR,
	Connection::T_CHARACTER_FILE => SQLT_CFILEE,
	Connection::T_CURSOR => SQLT_RSET,
	Connection::T_CUSTOM => SQLT_NTY,
	Connection::T_LONG => SQLT_LNG,
	Connection::T_INT => SQLT_INT,
	Connection::T_VARCHAR=>SQLT_CHR
);

  private static $oraTransactionModes = array(Connection::TR_DEFAULT=>OCI_DEFAULT,
         Connection::TR_AUTO_COMMIT=>OCI_COMMIT_ON_SUCCESS,
	 Connection::TR_NO_AUTO_COMMIT=>OCI_DEFAULT);

  private $session_mode;

  /**
   * Almacena temporalmente la conexi�n
   */
  private $connection = null;

  /**
   * @var string El nombre de la conexi�n.
   */
  private $tnsName;

  /**
   * @var string El tipo de conexi�n que se puede realizar: "direct" o "byName"
   */
  private $connectionType;

  /**
   * Crea una instancia del objeto en base a los datos especificados
   */
  public function __construct($user, $password, $sid_tnsName, $host = "", $port = 1521, $schema = "", $session_mode = OCI_DEFAULT)
  {
    // Bouml preserved body begin 00036385
    if (!extension_loaded('oci8')) die("No se ha cargado la extensión OCI8.\n");
	
	if (func_num_args() == 3) {
		$this->connectionType = OracleConnection::ORACLE_BYNAME_CONNECTION;
		$this->user    = $user;
		$this->password= $password;
		$this->tnsName = $sid_tnsName;
	} else {
		$this->connectionType = OracleConnection::ORACLE_DIRECT_CONNECTION;
		$this->host    = $host;
		if ($port==-1) {
			$port=1521;
		}
		$this->port    = $port;
		$this->schema  = $schema;
		$this->user    = $user;
		$this->password= $password;
		$this->session_mode=$session_mode;
		$this->sid = $sid_tnsName;
	}
	
    
    return $this;
    // Bouml preserved body end 00036385
  }

  /**
   * Cierra la conexi�n antes de destruir el objeto
   */
  public function __destruct()
  {
if ($this->isConnected()) {
    $this->close();
}
  }

  /**
   * Recupera la cantidad de filas afectadas (modificadas) por la �ltima operaci�n
   */
  public function affectedRows($query)
  {
    // Bouml preserved body begin 00039805
    return oci_num_rows($query);
    // Bouml preserved body end 00039805
  }

  /**
   * Inicia una transacci�n
   */
  public function beginTransaction()
  {
    // Bouml preserved body begin 00032885
    $statement = oci_parse($this->connection,"SAVEPOINT StartTransaction");
    return oci_execute($statement, OracleConnection::$oraTransactionModes[Connection::TR_NO_AUTO_COMMIT]);
    // Bouml preserved body end 00032885
  }

  public function bind($preparedStatement, $name, &$value, $size = -1, $type = Connection::T_VARCHAR, $element_type = Connection::T_VARCHAR)
  {
    // Bouml preserved body begin 0003B205
    if ($type != Connection::T_ARRAY)
        return oci_bind_by_name($preparedStatement, $name, $value, $size, OracleConnection::$oraTypes[$type]);
    else
        return oci_bind_array_by_name($preparedStatement, $name, $value,count($value), $size, OracleConnection::$oraTypes[$element_type]);
    // Bouml preserved body end 0003B205
  }

  /**
   * Cierra la conexi�n si est� abierta
   */
  public function close()
  {
    // Bouml preserved body begin 00032985
    $result = oci_close($this->connection);
    if ($result) $this->connection=null;
    return $result;
    // Bouml preserved body end 00032985
  }

  /**
   * Hace un commit de los cambios hasta el momento
   */
  public function commit()
  {
    // Bouml preserved body begin 00032A05
    return oci_commit($this->connection);
    // Bouml preserved body end 00032A05
  }

  /**
   * Establece la conexi�n con los par�metros que posee registrados en la clase
   */
  public function connect()
  {
    // Bouml preserved body begin 00032A85
	if ($this->connectionType == OracleConnection::ORACLE_BYNAME_CONNECTION) {
		$connection= $this->connection = oci_new_connect($this->user, $this->password, $this->tnsName, 'AL32UTF8');
	} else {
		$strConnectionString ="(DESCRIPTION = (ADDRESS = (PROTOCOL = TCP)(HOST =$this->host)(PORT = $this->port))
								(CONNECT_DATA =(SERVER = DEDICATED) (SERVICE_NAME = $this->sid)))";
		$connection= $this->connection = oci_new_connect($this->user, $this->password, $strConnectionString, 'AL32UTF8');
	}
    return $connection;
    // Bouml preserved body end 00032A85
  }

  /**
   * Crea una collection basada en el tipo de datos indicado
   * 
   * @param string $param el nombre del tipo de datos asociado. Generalmente es un TYPE personalizado.
   * @param string $schema el nombre de la base de datos (su esquema)
   * @return resource|false un puntero al recurso o false en caso de problemas
   */
  public function createCollection($type, $schema = '')
  {
    // Bouml preserved body begin 0007DE05

    /* Sitios de errores
     * http://marc.info/?l=php-cvs&m=118591630928253
     * http://bugs.php.net/bug.php?id=44113
     * http://pecl.php.net/package-info.php?package=oci8&version=1.2.5
     */
    if ($schema==='') {
        return oci_new_collection($this->connection, $type, $this->schema);
	} elseif(is_null($schema)) {
		return oci_new_collection($this->connection, $type);
	} else {
        return oci_new_collection($this->connection, $type, $schema);
	}
    // Bouml preserved body end 0007DE05
  }

  /**
   * Crea un cursor sobre la conexi�n especificada
   */
  public function createCursor()
  {
    // Bouml preserved body begin 0003CC85
    return oci_new_cursor($this->connection);
    // Bouml preserved body end 0003CC85
  }

  /**
   * Ejecuta el sql indicado y retorna el statement si todo anduvo bien o false si algo falló.
   * 
   * @param resource $statement el comando a ejecutar.
   * @param int $mode indica si la operación debe consolidarse automáticamente Connection::TR_AUTO_COMMIT, si no debe hacerlo TR_NO_AUTO_COMMIT o si se deja el modo predeterminado Connection::TR_DEFAULT.
   * 
   * @return resource|boolean si la operación se realizó correctamente retorna el handler, sino devuelve false.
   */
  public function execute($statement, $mode = Connection::TR_DEFAULT)
  {
    // Bouml preserved body begin 00032B05
    $result = null;
	try {
		if (is_string($statement)) {
			$statement = oci_parse($this->connection, $statement);
		}

		if ($mode==Connection::TR_DEFAULT) {
			$result= oci_execute($statement);
		} else {
			$result = oci_execute($statement, OracleConnection::$oraTransactionModes[$mode]);
		}
	} catch (Exception $e) {
		echo $e->getMessage();
		$result = false;
	}
    return ($result?$statement:false);
    // Bouml preserved body end 00032B05
  }

  /**
   * Ejecuta una funci�n en la base de datos y retorna el valor.
   * Es posible, indicar la variable en la cual queremos bindear el resultado, su tipo y tama�o en caso de ser necesario. Si la misma no se especifica se utiliza un valor interno y se retorna una copia del valor del resultado.-
   */
  public function executeFunction($name, &$params, &$result = null, $type = Connection::T_VARCHAR, $size = -1)
  {
    // Bouml preserved body begin 0003CB85

    if ($type == Connection::T_CURSOR) {
        $result = $this->createCursor();
    }

    $strParams ="";//Preparo el string de los parámetros
    foreach($params as $key=>$value) {
        $strParams .= ":param$key, ";
    }
    $strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
    $strSQL = "begin :data := $name$strParams; end;";
    $preparedStatement = $this->prepareStatement($strSQL);
    if ($preparedStatement===false) {
		throw new Exception('Error al preparar la consulta en OracleConnection::executeFunction()');
        return false;
    }
    
    //Bindeo el resultado con lo que me pasaron o algo propio
    if (oci_bind_by_name($preparedStatement,":data",$result,$size,OracleConnection::$oraTypes[$type])) {
        //Luego los resultados
        foreach($params as $key => $value) {
            if (!oci_bind_by_name($preparedStatement, ":param$key", $params[$key], -1, (is_int($params[$key])) ? SQLT_INT : SQLT_CHR)) {
				throw new Exception('Error al preparar el bindeo del parámetro param'.$key.' en OracleConnection::executeFunction()');
                return false;
            }
        }
        if ($this->execute($preparedStatement)) {
            if ($type == Connection::T_CURSOR) {
                $this->execute($result);
            }
            return $result;
        } else {
            return false;
        }
    } else {
		throw new Exception('Error al preparar la bindear el resultado en OracleConnection::executeFunction()');
        return false;
    }
    // Bouml preserved body end 0003CB85
  }

  /**
   * Ejecuta una funci�n en la base de datos y retorna el valor.
   * Es posible, indicar la variable en la cual queremos bindear el resultado, su tipo y tama�o en caso de ser necesario. Si la misma no se especifica se utiliza un valor interno y se retorna una copia del valor del resultado.-
   */
  public function executeClobPackage($name, &$params, $clob, &$result = null, $type = Connection::T_VARCHAR, $size = -1)
  {
    // Bouml preserved body begin 0012E385
	if ($type == Connection::T_CURSOR) {
        $result = $this->createCursor();
    }

    $strParams = "";//Preparo el string de los parámetros
    foreach($params as $key => $value) {
		$strParams .= ":param$key, ";
    }
    $strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
    $strSQL = "begin :data := $name$strParams; end;";
    $preparedStatement = $this->prepareStatement($strSQL);
    if ($preparedStatement===false) {
        return false;
    }
    
	$hayClobConfiguradoBien = false;
    //Bindeo el resultado con lo que me pasaron o algo propio
    if (oci_bind_by_name($preparedStatement,":data",$result,$size,OracleConnection::$oraTypes[$type])) {
        //Luego los resultados
		$clobcin = null;
        foreach($params as $key => $value) {
			if ($key === $clob) {
				$hayClobConfiguradoBien = true;
				$clobero = oci_new_descriptor($this->connection, OCI_D_LOB);
				if (!oci_bind_by_name($preparedStatement, ":param$key", $clobero, -1, OCI_B_CLOB)) {
					return false;
				}
			} else {
				if (!oci_bind_by_name($preparedStatement, ":param$key", $params[$key], -1, (is_int($params[$key])) ? SQLT_INT : SQLT_CHR)) {
					return false;
				}
			}
        }
        if ($this->execute($preparedStatement, Connection::TR_NO_AUTO_COMMIT)) {
			if ($hayClobConfiguradoBien) {
				$clobero->write($params[$clob]);
			}
			$this->commit();
            if ($type == Connection::T_CURSOR) {
                $this->execute($result);
            }
			
            return $result;
        } else {
            return false;
        }
    } else {
        return false;
    }
    // Bouml preserved body end 0012E385
  }
  
  
  public function executeMultiClobPackage($name, &$params, $clob, &$result = null, $type = Connection::T_VARCHAR, $size = -1)
  {
	if ($type == Connection::T_CURSOR) {
		$result = $this->createCursor();
	}

	if (!is_array($clob)) {
		$clob = array($clob);
	}

	$strParams = "";//Preparo el string de los parámetros
	foreach($params as $key => $value) {
		$strParams .= ":param$key, ";
	}
	$strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
	$strSQL = "begin :data := $name$strParams; end;";
	$preparedStatement = $this->prepareStatement($strSQL);
	if ($preparedStatement===false) {
		return false;
	}

	$hayClobConfiguradoBien = false;
	//Bindeo el resultado con lo que me pasaron o algo propio
	if (oci_bind_by_name($preparedStatement,":data",$result,$size,OracleConnection::$oraTypes[$type])) {
		//Luego los resultados
		$clobcin = null;
		$clobero = array();
		foreach($params as $key => $value) {
			if (array_search($key, $clob, true) !== false) {
				$hayClobConfiguradoBien = true;
				$clobero[$key] = oci_new_descriptor($this->connection, OCI_D_LOB);
				if (!oci_bind_by_name($preparedStatement, ":param$key", $clobero[$key], -1, OCI_B_CLOB)) {
					return false;
				}
			} else {
				if (!oci_bind_by_name($preparedStatement, ":param$key", $params[$key], -1, (is_int($params[$key])) ? SQLT_INT : SQLT_CHR)) {
					return false;
				}
			}
		}
		if ($this->execute($preparedStatement, Connection::TR_NO_AUTO_COMMIT)) {
			if (count($clobero) > 0) {
				foreach ($clobero as $cind => $cvalor)
					$cvalor->write($params[$cind]);
			}
			$this->commit();
			if ($type == Connection::T_CURSOR) {
				$this->execute($result);
			}

			return $result;
		} else {
			return false;
		}
	} else {
		return false;
	}
  }


  /**
   * Ejecuta el procedimiento bindeando los valores de los par�metros en el orden en que se encuentren
   */
  public function executeProcedure($name, &$params)
  {
    // Bouml preserved body begin 00037E05
    $strParams ="";//Preparo el string de los parámetros
    foreach($params as $key=>$value) {
        $strParams .= ":param$key, ";
    }
    $strParams = "(". substr($strParams, 0, strlen($strParams)-2) .")";
    $strSQL = "begin $name$strParams; end;";
    $preparedStatement = $this->prepareStatement($strSQL);
    if ($preparedStatement===false) { return false; }
    //Bindeo los parámetros
    foreach($params as $key => $value) {
        if (!oci_bind_by_name($preparedStatement, ":param$key", $params[$key])) {
            return false;
        }
    }
    return $this->execute($preparedStatement);
    // Bouml preserved body end 00037E05
  }

  /**
   * Establece un valor recuperaci�n "anticipada" de filas al hacer un fetch. Es decir, que cuando existe un prefetch establecido de N filas, al hacer la primer operaci�n fetch sobre un resultset, internamente se recuperan no una, sino las N filas existentes.
   * 
   * @param resource $query la consulta sobre la cual se quiere aplicar la recuperaci�n "preventiva"
   * @param int $numRows la cantidad de filas a pre-recuperar.
   * 
   * @return boolean un TRUE si la conexi�n de Oracle fue modificada exitosamente, FALSE sino.
   */
  public function prefetch($query, $numRows)
  {
    // Bouml preserved body begin 00130005
	return oci_set_prefetch($query, $numRows);
    // Bouml preserved body end 00130005
  }

  /**
   * Restorna una fila del conjunto de resultados en un arreglo indexado como se indique en $mode y avanza una posici�n el cursor del $resource
   */
  public function fetch($resource, $mode = Connection::M_ASSOCIATIVE)
  {
    // Bouml preserved body begin 00032D05
    if ($mode==Connection::M_ASSOCIATIVE) {
        return oci_fetch_array($resource, OCI_ASSOC+OCI_RETURN_NULLS);
    } else {
        return oci_fetch_array($resource, OCI_NUM+OCI_RETURN_NULLS);
    }
    // Bouml preserved body end 00032D05
  }

  /**
   * Retorna un arreglo con los registros recuperados de la consulta. Se provee la facilidad de indicar que se saltee tantos registros como se indiquen en $offest y que se recuperen s�lo una cierta cantidad de ellos; se una -1 en caso de querer recuperar todas las filas disponibles desde $offset.
   * En forma predeterminada retorna un arreglo asociativo, lo cual puede cambiarse con el par�metro $mode.
   */
  public function fetchAll($resource, $mode = Connection::M_ASSOCIATIVE, $offset = 0, $count = -1)
  {
    // Bouml preserved body begin 00032D85
    if ($mode==Connection::M_ASSOCIATIVE) {
        $flags=OCI_FETCHSTATEMENT_BY_ROW+OCI_ASSOC;
    } else {
        $flags=OCI_FETCHSTATEMENT_BY_ROW+OCI_NUM;
    }
    if (oci_fetch_all($resource, $result, $offset, $count, $flags)) {
        return $result;
    } else {
        return false;
    }
    // Bouml preserved body end 00032D85
  }

  /**
   * Libera un recurso previamente asignado.
   */
  public function free(&$resource)
  {
    // Bouml preserved body begin 00032E05
    return oci_free_statement($resource);
    // Bouml preserved body end 00032E05
  }

  /**
   * Retorna true si la conexi�n est� activa o false en caso contrario
   */
  public function isConnected()
  {
    // Bouml preserved body begin 00032E85

    if (isset($this->connection)) {
        return !is_null($this->connection) && ($this->connection != false);
    } else return false;
    // Bouml preserved body end 00032E85
  }

  /**
   * Retorna un texto que describe el �ltimo error que sucedi� en la conexi�n. Si se desea se puede acompa�ar del handler que retorno la funci�n que fall� o de lo contrario se retorna el �ltimo error que se dio en la conexi�n.
   */
  public function lastError($over = -1)
  {
    // Bouml preserved body begin 00032F05
    if ($over == -1) {
        $data = oci_error($this->connection);
    } else {
        $data = oci_error($over);
    }

    if ($data == false) {
        return false;
    } else {
        return $data['message'];
    }
    // Bouml preserved body end 00032F05
  }

  /**
   * Retorna el c�digo del �ltimo error que se present� en la conexi�n o en el handler del par�metro
   */
  public function lastErrorCode($over = -1)
  {
    // Bouml preserved body begin 00032F85
    if ($over == -1) {
        $data = oci_error($this->connection);
    } else {
        $data = oci_error($over);
    }

    if ($data != false) {
        return $data['code'];
    } else {
        return false;
    }
    // Bouml preserved body end 00032F85
  }

  /**
   * Salta al siguiente conjunto de resultados
   */
  public function nextResultset($handler)
  {
    // Bouml preserved body begin 00033005
    return false;
    // Bouml preserved body end 00033005
  }

  /**
   * Retorna el n�mero de columnas del handler del query especificado
   */
  public function numColumns($query)
  {
    // Bouml preserved body begin 00033085
    return oci_num_fields($query);
    // Bouml preserved body end 00033085
  }

  /**
   * Retorna el n�mero de filas que devolvi� el query especificado
   */
  public function numRows($query)
  {
    // Bouml preserved body begin 00033105
	/*
	 * Esto establecia
    if (oci_statement_type($query)=="SELECT") {
        return 0;
    } else {
        return oci_num_rows($query);
    }*/
	return oci_num_rows($query);
    // Bouml preserved body end 00033105
  }

  /**
   * Retorna un handler a un prepared statement
   */
  public function prepareStatement($sql)
  {
    // Bouml preserved body begin 00033185
    return oci_parse($this->connection, $sql);
    // Bouml preserved body end 00033185
  }

  /**
   * Ejecuta la instrucci�n SQL y retorna el resultado.
   */
  public function query($sql)
  {
    // Bouml preserved body begin 00033205
    $statement = oci_parse($this->connection, $sql);
    oci_execute($statement, OracleConnection::$oraTransactionModes[Connection::TR_NO_AUTO_COMMIT]);
    return $statement;
    // Bouml preserved body end 00033205
  }

  /**
   * Si el motor de la base de datos lo soporta, hace un rollback de la �ltima transacci�n
   */
  public function rollback()
  {
    // Bouml preserved body begin 00033285
    //return oci_rollback($this->connection);
    $statement = oci_parse($this->connection, "ROLLBACK TO SAVEPOINT StartTransaction");
    return oci_execute($statement);
    // Bouml preserved body end 00033285
  }

  /**
   * Retorna la informaci�n de status del motor
   */
  public function status()
  {
    // Bouml preserved body begin 00033305
    return false;
    // Bouml preserved body end 00033305
  }

}
?>
