<?php
require_once 'iplan/orm/ORMDefinition.php';
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORMObject.php';
require_once 'iplan/database/Connection.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Este objeto se crea indicando cu�l es la clase "pivote", con cuyo nombre recupera la definici�n desde el ORM.
 * 
 * Las condiciones de filtrado se especifican a trav�s de llamados a filterBy() y el conjunto de resultados se ordena seg�n el orden de llamados a orderBy().
 * 
 * Es posible segmentar el conjunto de resultados usando los pares de funciones: from()/to() � from()/top(), seg�n se quieran indicar fila de inicio y fin o fila de inicio y cantidad de registros respectivamente.
 * 
 * El resultado de la consulta se obtiene llamando a find().
 */
class ORMQuery {
  /**
   * @var string el nombre de la clase pivote de la consulta
   */
  public $pivotClass;

  /**
   * @var int el n�mero de fila del primer registro del resultado.
   */
  public $from;

  /**
   * @var int la cantidad de filas a recuperar.
   */
  public $count;

  /**
   * @var ORMDefinition la definici�n de la clase pivote
   */
  private $definition;

  /**
   * @var ORM el manejador de objetos desde el cual se toman las definiciones
   */
  public $orm;

  /**
   * @var array un arreglo con las condiciones indicadas por los llamados a filterBy()
   * El formato del arreglo es:
   * $conditions['filters'][$field][$operator][0..n]=array($value, $other, $connector);
   * $conditions['subqueries'][$field][0..n]=ORMQuery();
   */
  public $conditions;

  /**
   * @var boolean determina si el usuario ha o no especificado columnas personalizadas para el resultado
   */
  public $uses_custom_columns;

  /**
   * @var boolean indica si se han especificado atributos de agrupaci�n
   */
  public $uses_groups;

  /**
   * Crea un objeto de consulta que retorna elementos del tipo de la clase indicada.
   * 
   * @param string $class la clase de objetos que retornar� la consulta.
   * @param ORM $orm el objeto ORM al que se ata la consulta.
   * @return ORMQuery un objeto con los criterios de una consulta.
   */
  public function __construct($class, &$orm)
  {
    // Bouml preserved body begin 00096C05
    $this->orm=$orm;
    $this->pivotClass=$class;
    $this->definition=$orm->getDefinition($class);
    $this->uses_custom_columns = false;
    $this->uses_groups = false;
    return $this;
    // Bouml preserved body end 00096C05
  }

  /**
   * Establece la cantidad de filas a saltear del conjunto de resultados antes de considerarla v�lida como respuesta.
   * 
   * @param int $row la fila a partir de la cual deseamos que se nos retornen resultados.
   * 
   * @return ORMQuery La consulta modificada. from() siempre debe llamarse antes que to() para calcular el diferencial.
   */
  public function from($row)
  {
    // Bouml preserved body begin 00098705
    if (isset($this->from) && isset($this->count)) {
        $tmpCount = $this->count;
        $tmpCount += ($this->from - $row);
        if ($tmpCount > 0) $this->count = $tmpCount;
    }
    $this->from = $row;
    return this;
    // Bouml preserved body end 00098705
  }

  /**
   * Establece el m�ximo n�mero de fila que quiere recuperarse. Debe ser mayor al valor prefijado en from().
   * 
   * @param int $row el n�mero de fila m�xima del conjunto de resultado.
   * 
   * @return ORMQuery La consulta con el nuevo l�mite aplicado. Debe considerarse que si no se llam� a from() previamente no se va a calcular el diferencial.
   */
  public function to($row)
  {
    // Bouml preserved body begin 00098785
    if (isset($this->from) && isset($this->count)) {
        $tmpTo = $this->from+$this->count;
        $tmpTo += ($row-$tmpTo);
        if ($tmpCount > 0) $this->count = $tmpCount;
    } else {
        $this->count = $row-$this->from;
    }
    return this;
    // Bouml preserved body end 00098785
  }

  /**
   * Establece cuantas filas se deben recuperar del conjunto de resultados.
   * 
   * @param int $limit la cantidad de filas a recuperar
   * 
   * @return ORMQuery El objeto Query modificado.
   */
  public function top($limit)
  {
    // Bouml preserved body begin 00098805
    $this->count = $limit;
    return this;
    // Bouml preserved body end 00098805
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra clase, utilizando para el emparejamiento las condiciones dadas e incorporando al conjunto de resultados los atributos indicados.
   * @param string $attribute el atributo que determina la relaci�n con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $attributes los atributos nuevos a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados
   */
  public function joinWith($attribute, $alias, $attributes, $conditions)
  {
    // Bouml preserved body begin 000C8A85
    // Bouml preserved body end 000C8A85
  }

  /**
   * Une mediante un JOIN los valores de la consulta actual con los de otra consulta. Debido a que no se pueden inferir los resultados de la consulta tan f�cilmente como con el joinWith, a esta funci�n se le deben pasar los nombres de columnas devueltas por el query para hacer el emparejamiento y para definir nuevos campos para el conjunto de resultados.
   * 
   * @param ORMQuery $query la query con la cual se ha de realizar el emparejamiento
   * @param string $alias el alias de la clase
   * @param array $fields los campos nuevos (con denominaci�n completa) a incorporar en el conjunto de resultados
   * @param array $conditions las condiciones de emparejamiento del conjunto de resultados tambi�n con denominaci�n completa
   */
  public function joinBy($query, $alias, $fields, $conditions)
  {
    // Bouml preserved body begin 000C8A05
    // Bouml preserved body end 000C8A05
  }

  /**
   * Busca el primer elemento de la consulta.
   * 
   * @return ORMObject el objeto en cuesti�n
   */
  public function findOne()
  {
    // Bouml preserved body begin 0009D785
    $res = $this->find(1, 1);
    if ($res === false)
        return false;
    else
        return $res[0];
    // Bouml preserved body end 0009D785
  }

  /**
   * @return string Imprime un cierto n�mero de espacios acorde al $level
   */
  public function echoTab()
  {
    // Bouml preserved body begin 0009F105
    echo str_pad("\n", 5 * self::$level, " ", STR_PAD_RIGHT);
    // Bouml preserved body end 0009F105
  }

  /**
   * @var int $level utilizado para fines de debug
   */
  static $level = 0;

  /**
   * @var boolean $debug indica si se debe hacer una salida de depuraci�n de los pasos de la consulta
   */
  static $debug = false;

  /**
   * Funci�n temporal s�lo para fines de debug. Decifra la cadena retornada por "build()".
   */
  public function decrypt(&$coded_sql)
  {
    // Bouml preserved body begin 000BEB05
    $db = $this->orm->getDatabase();
    $conn = $db->getConnection();
    $ps = $conn->prepareStatement("SELECT PKG_WEB_ACCESSOR.DECRYPT(:salida) as salida FROM DUAL");
    $conn->bind($ps, ":salida", $coded_sql,-1, Connection::T_BINARY);
    $resQuery = $conn->execute($ps);
    return $conn->fetchAll($ps);
    // Bouml preserved body end 000BEB05
  }

  /**
   * Establece una condici�n de b�squeda para la consulta.
   * 
   * @param string $attribute el campo sobre el cual se especifica la condici�n
   * @param int $operator la operaci�n de comparaci�n
   * @param mixed $value el valor utilizado para la comparaci�n
   * @param mixed $other otro valor en caso de ser necesario
   * @param string $connector la palabra 'AND' o 'OR' seg�n se quiera utilizar para concatenar la condici�n
   * 
   * @return ORMQuery la consulta con la condici�n incorporada.
   */
  public function filterBy($attribute, $operator, $value, $other = "", $connector = 'AND')
  {
    // Bouml preserved body begin 000CA605
    if ($pointPos = strpos($attribute, '.')) {
       $atts = explode('.', $attribute);

       $currentSlot = &$this->conditions['filters'];
       $i = 1;
       foreach($atts as $key => $att) {
           if (count($atts) == $i++) {
              $currentSlot = &$currentSlot['locals'][$att];
           } else {
              $currentSlot = &$currentSlot['subquerys'][$att];
           }
       }
       $currentSlot[strtoupper($operator)][]=array($value, $other, $connector);
    } else {
      $this->conditions['filters']['locals'][$attribute][strtoupper($operator)][]=array($value, $other, $connector);
    }
    return $this;
    // Bouml preserved body end 000CA605
  }

  /**
   * Ordena el conjunto de resultados por la columna indicada.
   * 
   * @param string $attribute el atributo por el cual se desea ordenar el resultado. No se puede ordenar por atributos relaci�n, ni por los atributos de los elementos de las relaciones o de los atributos entidad. Se puede indicar anteponiendo un espacio si es 'ASC' o 'DESC'
   * 
   * @return ORMQuery La consulta modificada como se indic�.
   */
  public function orderBy($attribute)
  {
    // Bouml preserved body begin 000CA685
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['sorts'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['sorts'][]=$attribute;
        }
    }
    return $this;
    // Bouml preserved body end 000CA685
  }

  /**
   * Retorna el c�digo a ejecutar para resolver la query en formato codificado.
   * 
   * @param array $conditions las condiciones a analizar �, si es null, toma las globales
   * @param int $alias_counter un numero que se a�ade al alias de cada tabla
   * @return string el SQL codificado de la instrucci�n
   */
  public function build()
  {
    // Bouml preserved body begin 000CA705
    $db = $this->orm->getDatabase();
    $conditions = $this->conditions['filters'];
    $alias_counter = 1;
    $localFilters = "";
    $conditions['alias']="t$alias_counter";
    $conditions['alias.']=$conditions['alias'].".";
    $groups = array();
    $sorts  = array();

    //Creo el select principal con los locals
    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $attribute => $filters) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            foreach($filters as $operator => $valueSets) {
                foreach($valueSets as $valueSet) {
                    list($value, $other, $connector) = $valueSet;
                    switch ($operator) {
                        case 'BETWEEN':
                            $localFilters[] = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                            break;
                        default:
                            $localFilters[] = $db->makeCondition($operator, $conditions['alias.'].$fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                    }

                }
            }
        }
    }
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute) {
            $attParts = explode(" ", $attribute);
            $fieldDef = $this->definition->getFieldDefinition($attParts[0]);
            $sorts[]=$conditions['alias.'].$fieldDef['fieldName'].((isset($attParts[1]))?' '.$attParts[1]:'');
        }
    }
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $this->definition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

    $keyDef = $this->definition->getFieldDefinition('id');
    if (!isset($conditions['columns'])) {
        $query = $db->sql_filter($keyDef['table'], array($keyDef['fieldName']), $conditions['alias'], $localFilters);
    } else {
        foreach($conditions['columns'] as $att) {
            $colDef = $this->definition->getFieldDefinition($att);
            $colDefs[]=$colDef['fieldName']." AS \"$att\"";
        }
        $query = $db->sql_filter($keyDef['table'], $colDefs, $conditions['alias'], $localFilters);
    }

    //Creo los subquerys
    $alias_counter++;
    if (isset($conditions['subquerys'])) {
        foreach($conditions['subquerys'] as $attribute => $subconditions) {
            list($query, $subSorts, $subGroups) = $this->build_subquery($query, $this->definition, $attribute, $conditions['subquerys'][$attribute], $conditions['alias'], $attribute, $alias_counter);
            $sorts = array_merge($sorts, $subSorts);
            $groups = array_merge($groups, $subGroups);
        }
    }
    if (count($sorts)==0)  $sorts = null;
    if (count($groups)==0) $groups = null;
    else {
        $query = $db->sql_set_from($query, $groups); //Reemplazo las columnas del SELECT
    }
    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA705
  }

  public function build_subquery(&$query, $pivotDefinition, $attribute, &$conditions, $parent_alias, $scrumb, &$alias_counter)
  {
    // Bouml preserved body begin 000CA885
    $keyDef = $pivotDefinition->getFieldDefinition('id');
    $db = $this->orm->getDatabase();

    $myAliasCounter = $alias_counter++;
    $conditions['alias'] = "t$myAliasCounter";
    $conditions['alias.'] = $conditions['alias'].".";
    $sorts = array();
    $groups = array();

    $join_conditions = null;
    $relDefinition = $pivotDefinition->getRelationDefinition($attribute);
    if ($relDefinition === false) $relDefinition = $pivotDefinition->getFieldDefinition($attribute);

    $newPivotDefinition = $this->orm->getDefinition($relDefinition['class']);

    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $NxM_alias = $myAliasCounter;
        $myAliasCounter = $alias_counter++;
        $conditions['alias'] = "t$myAliasCounter";
        $conditions['alias.'] = $conditions['alias'].".";
//        $NxM_alias = $alias_counter++;
        $theOtherSideKey = $newPivotDefinition->getFieldDefinition('id');
        $join_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$relDefinition['associatedFieldName'], $conditions['alias.'].$theOtherSideKey['fieldName']);
    } else {
        $join_conditions[] = $db->makeCondition('=', $conditions['alias.']. $relDefinition['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
    }

    if (isset($conditions['locals'])) {
        foreach ($conditions['locals'] as $att => $filters) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($att);
            foreach($filters as $operator => $valueSets) {
                foreach($valueSets as $valueSet) {
                    list($value, $other, $connector) = $valueSet;
                    switch ($operator) {
                        case 'BETWEEN':
                            $join_conditions[] = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']) . " AND " . $db->php2db($other, $fieldDef['type']) , $connector);
                            break;
                        default:
                            $join_conditions[] = $db->makeCondition($operator, $conditions['alias.']. $fieldDef['fieldName'], $db->php2db($value, $fieldDef['type']), $connector);
                    }

                }
            }
        }
    }
    if (isset($conditions['sorts'])) {
        foreach ($conditions['sorts'] as $attribute) {
            $attParts = explode(" ", $attribute);
//            $fieldDef = $this->definition->getFieldDefinition($attParts[0]);
            $fieldDef = $newPivotDefinition->getFieldDefinition($attParts[0]);
            $sorts[]=$conditions['alias.'].$fieldDef['fieldName'].((isset($attParts[1]))?' '.$attParts[1]:'');
        }
    }
    if (isset($conditions['groups'])) {
        foreach ($conditions['groups'] as $attribute) {
            $fieldDef = $newPivotDefinition->getFieldDefinition($attribute);
            $groups[]=$conditions['alias.'].$fieldDef['fieldName'];
        }
    }

    if (isset($conditions['subquerys']))
        foreach($conditions['subquerys'] as $att => $subconditions) {
            list($query, $subSorts, $subGroups) = $this->build_subquery($query, $newPivotDefinition, $att, $conditions['subquerys'][$att], $conditions['alias'], "$scrumb.$att",$alias_counter);
            $sorts = array_merge($sorts, $subSorts);
            $groups = array_merge($groups, $subGroups);
        }

    $newPivotKeyField = $newPivotDefinition->getFieldDefinition('id');
    if (isset($conditions['columns']))
        foreach($conditions['columns'] as $att) {
            $colDef = $newPivotDefinition->getFieldDefinition($att);
            $colDefs[]=$colDef['fieldName']." AS \"$scrumb.$att\"";
        }

    $query = $db->sql_join_table($query, $newPivotKeyField['table'], $conditions['alias'], (isset($conditions['columns'])?$colDefs:null), $join_conditions);

    if (isset($relDefinition['type']) && ($relDefinition['type']==ORMDefinition::ManyToMany)) {
        $newPivotKeyField = $newPivotDefinition->getFieldDefinition('id');
        $NxM_conditions = array();
        $NxM_conditions[] = $db->makeCondition('=', "t$NxM_alias.".$keyDef['fieldName'], "$parent_alias.".$relDefinition['fieldName']);
        $query = $db->sql_join_table($query, $relDefinition['table'], "t$NxM_alias", null, $NxM_conditions);

    }

    return array($query, $sorts, $groups);
    // Bouml preserved body end 000CA885
  }

  /**
   * Recupera el conjunto de resultados seg�n los par�metros configurados en el el objeto.
   * 
   * @param int $from indica desde qu� elemento se deben recuperar los resultados.
   * @param int $count indica cu�ntos elementos de resultado se deben recuperar.
   * @return array Un arreglo con todos los elementos que respondieron positivamente a las condiciones indicadas.
   */
  public function find($from = null, $count = null)
  {
    // Bouml preserved body begin 000CA785
    $db = $this->orm->getDatabase();
    $f = (!is_null($from))  ? $from : $this->from;
    $c = (!is_null($count)) ? $count: $this->count;
    list ($query, $sorts, $groups) = $this->build();
    $result = $db->sql_execute2($query, $groups, $sorts, $f, $c);
    if ($result !== false) {
        if ($this->uses_custom_columns) {
            return $result;
        } else {
            $keyDef = $this->definition->getFieldDefinition('id');
            foreach($result as $res) {
                $objs[] = $this->orm->load($this->pivotClass, $res[$keyDef['fieldName']]);
            }
            return $objs;
        }
    } else {
        return false;
    }
    // Bouml preserved body end 000CA785
  }

  /**
   * Define qu� atributos recuperar del conjunto de resultados.
   * 
   * @param string $attribute un atributo a ser devueltos por la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function attributes($attribute)
  {
    // Bouml preserved body begin 000CA805
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['columns'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['columns'][]=$attribute;
        }
    }
    $this->uses_custom_columns = true;
    return $this;
    // Bouml preserved body end 000CA805
  }

  /**
   * Define qu� atributos deben ser agrupados una vez obtenido el conjunto de resultados masivo.
   * 
   * @param string $attribute un atributo por el cual se debe agrupar el resultado de la consulta. Se pueden pasar varios uno tras otro y son "navegables" a trav�s de puntos.
   * @return ORMQuery retorna la query modificada
   */
  public function groupBy($attribute)
  {
    // Bouml preserved body begin 000CC285
    foreach(func_get_args() as $attribute) {
        if ($pointPos = strpos($attribute, '.')) {
           $atts = explode('.', $attribute);

           $currentSlot = &$this->conditions['filters'];
           $i = 1;
           foreach($atts as $key => $att) {
               if (count($atts) == $i++) {
                  $currentSlot = &$currentSlot['groups'];
               } else {
                  $currentSlot = &$currentSlot['subquerys'][$att];
               }
           }
           $currentSlot[]=$att;
        } else {
          $this->conditions['filters']['groups'][]=$attribute;
        }
    }
    $this->uses_groups = true;
    $this->uses_custom_columns = true; //Esto se pone porque si se agrupa por X criterio no tiene sentido devolver objetos: ¿cuál devuelvo?
    return $this;
    // Bouml preserved body end 000CC285
  }

  /**
   * Retorna la cantidad de resultados del query.
   * 
   * @return int La cantidad de filas obtenidas por la consulta.
   */
  public function count()
  {
    // Bouml preserved body begin 000CDD05
    $db = $this->orm->getDatabase();
    list ($query, $sorts, $groups) = $this->build();
    return $db->count($query);
    // Bouml preserved body end 000CDD05
  }

}
?>