<?php
require_once 'iplan/orm/ORM.php';
require_once 'iplan/orm/ORM_TYPES.php';
require_once 'iplan/orm/ORM_LOAD_STYLE.php';
require_once 'iplan/orm/ORM_RELATION_TYPE.php';
require_once 'iplan/orm/PrivateAccesor.php';
require_once 'iplan/orm/exceptions/ORMDefinitionError.php';
require_once 'iplan/orm/exceptions/UnknowAttribute.php';



/**
* Author: Jorge Alexis Viqueira
* 
*/
/**
 * Hay un array que almacena la forma de relacionar los atributos de una clase con los de la/s tablas con las que se corresponde:
 * ORMData['fields'][$attribute]= {'fieldName', 'type',
 *                                                     'length', 'isNulleable', 'precision', 'primaryKey', 'default', 'table', 'loadStyle'}
 *                si type = ORMDefinition::ORM_ENTITY  'class', 'isNulleable', 'default', 'table, 'loadStyle'}
 * ORMData['fieldsByTable'][$table][$attribute]->&ORMData['fields'][$attribute]
 * ORMData['relations'][$attribute]= {'fieldName', 'type', 'class', 'table', 'loadStyle', 'associatedFieldName'}
 * 
 *  M�s adelante puede implementarse una versi�n m�s objetosa haciendo que haya cosas como DBTable y DBField
 * 
 * Hay 3 diferentes tipos de definiciones:
 *  - Campos simples: son atributos que representan un valor puntual como int, string, date.
 *  - Entidades: es la materializaci�n de la clave for�nea en la entidad "hijo". Es decir cuando un id de otra tabla aparece en la tabla actual. Ese "id" no es un atributo com�n, sino que representa a toda una entidad entera asociada por su id. Son relaciones 0..1 o 1..1
 *  - Relaciones: parecido al caso anterior pero para relaciones 1..N y N..M. El atributo en particular se implementa como un "array" de instancias de la clase asociada. Para el caso N..M la tabla se asume que tiene una clave conformada por los 2 IDs que se relacionan.
 * 
 */
class ORMDefinition extends PrivateAccesor implements ORM_TYPES, ORM_LOAD_STYLE, ORM_RELATION_TYPE {
  public $ORMData;

  public $currentTable;

  /**
   * @var ORM la instancia del ORM al que pertenece la definici�n
   */
  private $orm;

  /**
   * Constructor de la clase. Prepara las estructuras b�sicas necesarias para el mapeo.
   * 
   * @param ORM $orm el objeto que maneja las definiciones
   * @return ORMDefinition una instancia inicializada de ORMDefinition
   */
  public function __construct($orm = null)
  {
    // Bouml preserved body begin 00084705
    $this->ORMData['fields']=null;
    $this->ORMData['fieldsByTable']=null;
    $this->ORMData['relations']=null;
    $this->orm=$orm;
    return $this;
    // Bouml preserved body end 00084705
  }

  /**
   * Establece una tabla objetivo para futuros llamados a addField(), addInstance() y addRelation().
   * 
   * @param string $table el nombre de la tabla
   * @return ORMDefinition la propia definici�n modificada
   * 
   * @see ORMDefinition::addField()
   * @see ORMDefinition::addInstance()
   * @see ORMDefinition::addRelation()
   */
  public function setTable($tableName)
  {
    // Bouml preserved body begin 00043285
    $this->currentTable = $tableName;
    return $this;
    // Bouml preserved body end 00043285
  }

  /**
   * Indica como traducir un atributo del objeto en su campo respectivo de la base de datos.
   * 
   * @param string $attribute el nombre del atributo que queremos definir (el del objeto)
   * @param string $fieldName el nombre del campo de la base de datos donde se almacena el valor
   * @param int $type el tipo de valor, las constantes se obtienen de ORM_TYPES. En el caso que el atributo sea un objeto entero, usar @see addInstance().
   * @param int $length la longitud del campo de la base de datos. Si se quiere omitir esto indicar -1
   * @param int $precision en el caso de num�ricos se debe indicar la precisi�n para futuros redondeos. Para omitir -1
   * @param boolean $isNulleable indica si el atributo/campo puede adquirir valor NULL
   * @param int $loadStyle se puede indicar si se desea que el atributo sea recuperado instant�neamente al cargar el objeto o si se prefiere diferir la carga hasta que el valor del atributo sea utilizado por primera vez. Recomendable usar esta �ltima opci�n si el campo es una imagen o un alg�n valor pesado. Las constantes est�n en @see ORM_LOAD_STYLE.
   * @param string $defaultValue el valor predeterminado del campo
   * @param string $table el nombde de la tabla donde se encuentra el atributo, generalmente se especifica el mismo con un llamado previo a @see setTable() y se asume ese valor dejando ""
   * @param boolean $primaryKey poner true si el campo es la clave primaria del objeto
   * 
   * @return ORMDefinition Retorna la propia definici�n modificada, para seguirla manipulando en cadena si se desea.
   */
  public function addField($attribute, $fieldName, $type, $length = -1, $precision = -1, $isNulleable = true, $loadStyle = ORMDefinition::FORCE_LOAD, $defaultValue = "", $table = "", $primaryKey = false)
  {
    // Bouml preserved body begin 00026305
    if ($table == '') {
        if ($this->currentTable!= '') {
            $table = $this->currentTable;
        } else {
            throw new ORMDefinitionError("Error al definir el atributo '$attribute': no hay tabla especificada");
        }
    }
    $this->ORMData['fields'][$attribute]=array('fieldName'=>$fieldName, 'type'=>$type, 'length'=>$length, 'isNulleable'=>$isNulleable, 'precision'=>$precision, 'primaryKey'=>$primaryKey,'default'=>$defaultValue , 'table'=>$table, 'loadStyle'=>$loadStyle);
    $this->ORMData['fieldsByTable'][$table][$attribute]=&$this->ORMData['fields'][$attribute];
    return $this;
    // Bouml preserved body end 00026305
  }

  public function addInstance($attribute, $fieldName, $class, $isNulleable = true, $table = "", $defaultValue = null, $loadStyle = ORMDefinition::FORCE_LOAD)
  {
    // Bouml preserved body begin 0006EF85
    if ($table == '') {
        if ($this->currentTable!= '') {
            $table = $this->currentTable;
        } else {
            throw new ORMDefinitionError("Error al definir el atributo(instancia) '$attribute': no hay tabla especificada");
        }
    }
    $this->ORMData['fields'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORMDefinition::ORM_ENTITY, 'class'=>$class, 'isNulleable'=>$isNulleable, 'default'=>$defaultValue , 'table'=>$table, 'loadStyle'=>$loadStyle);
    $this->ORMData['fieldsByTable'][$table][$attribute]=&$this->ORMData['fields'][$attribute];
    return $this;
    // Bouml preserved body end 0006EF85
  }

  /**
   * Indica que hay una relaci�n entre esta entidad y las instancias de otra clase. Las relaciones que soportan son 1:N, N:M y 1:1. En el caso de la N:M debe indicarse la tabla intermedia.
   * En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr� el array de instancias
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param int $relation_type el tipo de relaci�n: 1:N, N:M o 1:1. Las constantes est�n en ORM_RELATION_TYPE.
   * @param int $loadStyle determina si la relaci�n se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta �ltima es la acci�n predeterminada). Las constantes est�n en ORM_LOAD_STYLE.
   * @param string $asociatedFieldName el nombre del atributo donde se mapea el identificador de la clase asociada
   * @param string $table s�lo se usa en relaciones N:M e indica el nombre de la tabla intermedia
   * 
   * @return ORMDefinition la propia definici�n
   */
  public function addRelation($attribute, $fieldName, $class, $relation_type, $table = "", $asociatedFieldName = "", $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00066885
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>$relation_type, 'class'=>$class, 'table'=>$table, 'loadStyle'=>$loadStyle, 'associatedFieldName'=>$asociatedFieldName);
    return $this;
    // Bouml preserved body end 00066885
  }

  /**
   * Retorna el registro crudo de los campos, tal como se le dieron de alta.
   * @param string $attribute el nombre del atributo del cual queremos recibir la informaci�n de mapeo o nada para obtener todas las definiciones
   * @param boolean $findByField indica si el atributo pasado por par�metro es en realidad el nombre del campo de la base de datos y lo busca por dicho criterio.
   * 
   * @return array|false Retorna las definiciones en un arreglo de la forma o false si el atributo no existe:
   * [$attribute] = {'fieldName', 'type', 'length', 'isNulleable', 'precision', 'primaryKey', 'default', 'table', 'loadStyle'}
   */
  public function getFieldDefinition($attribute = "", $findByField = false)
  {
    // Bouml preserved body begin 00063305
    if ($attribute != "") {
        if ($findByField) {
            foreach($this->ORMData['fields'] as $definition)
                 if ($definition['fieldName']==$attribute)
                     return array_merge($definition, array('attribute'=>$attribute));
        } else {
            if (isset($this->ORMData['fields'][$attribute]))
                 return $this->ORMData['fields'][$attribute];
            else return false;
        }
    } else {
        return $this->ORMData['fields'];
    }
    // Bouml preserved body end 00063305
  }

  /**
   * Retorna la definici�n de una relaci�n.
   * 
   * @param string $attribute el nombre del atributo cuyo mapeo se quiere obtener o nada para recuperar todas las definiciones de relaciones.
   * @param boolean $findByTable indica si el atributo pasado por par�metro es en realidad el nombre del campo de la base de datos y lo busca por dicho criterio.
   * 
   * @return array un arreglo con la informaci�n de la/s definici�n/es con el siguiente formato o false si la relaci�n no existe:
   * [$attribute] = {'type', 'class', 'table', 'loadStyle'}
   * 
   * En el caso que se busque por tabla retorna la primer relaci�n definida que act�e sobre la tabla indicada. Esta funci�n est� orientada a detectar el atributo que mapea las relaciones MxN sobre una tabla dada. NO SIRVE para las relaciones 1:1 y 1:N.
   */
  public function getRelationDefinition($attribute = "", $findByTable = false)
  {
    // Bouml preserved body begin 00081205
    if ($attribute != "") {
        if ($findByTable) {
            foreach($this->ORMData['relations'] as $attributeName=>$definition) {
                if ($definition['table']==$attribute) {
                    return array_merge($definition, array('attribute'=>$attributeName));
                }
            }
        } else {
            if (isset($this->ORMData['relations'][$attribute]))
                 return $this->ORMData['relations'][$attribute];
            else return false;
        }
    } else {
        return $this->ORMData['relations'];
    }
    // Bouml preserved body end 00081205
  }

  /**
   * Retorna la informaci�n de los campos que pertenecen a una tabla o a todas las tablas de la definici�n si se pasa ""
   */
  public function getFieldsByTable($table = "")
  {
    // Bouml preserved body begin 00061905
      if ($table=="") {
          return $this->ORMData['fieldsByTable'];
      } else {
          return $this->ORMData['fieldsByTable'][$table];
      }
    // Bouml preserved body end 00061905
  }

  /**
   * Retorna la lista de tablas que participan de la definici�n.
   */
  public function getTableList()
  {
    // Bouml preserved body begin 00061805
    return array_keys($this->ORMData['fieldsByTable']);
    // Bouml preserved body end 00061805
  }

  /**
   * Establece el atributo indicado como clave del objeto.
   * Es igual que indicar en addField() isNulleable en FALSE y primaryKey en TRUE. Se da esta funci�n para facilitar su definici�n y mover dichos atributos al final de los par�metros en addField().
   * 
   * @param string $attribute el atributo que ser� identificado como clave primaria del objeto.
   * 
   * @return ORMDefinition La propia definici�n modificada. IMPORTANTE: llamar a esta funci�n LUEGO de haber definido el mapeo del atributo.
   */
  public function setKey($attribute)
  {
    // Bouml preserved body begin 00086285
    $this->ORMData['fields'][$attribute]['isNulleable']=false;
    $this->ORMData['fields'][$attribute]['primaryKey']=true;
    return $this;
    // Bouml preserved body end 00086285
  }

  /**
   * @todo METODO PARA BORRAR!!! ES PARA DEBUG NOMAS
   */
  public function getORMData()
  {
    // Bouml preserved body begin 00084885
    return $this->ORMData;
    // Bouml preserved body end 00084885
  }

  /**
   * Indica que hay una relaci�n 1:N entre esta entidad y las instancias de la otra clase. En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr� el array de instancias
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param int $loadStyle determina si la relaci�n se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta �ltima es la acci�n predeterminada). Las constantes est�n en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definici�n
   */
  public function addRelation1xN($attribute, $fieldName, $class, $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087E05
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::OneToMany, 'class'=>$class, 'loadStyle'=>$loadStyle, 'table'=>'', 'associatedFieldName'=>'');
    return $this;
    // Bouml preserved body end 00087E05
  }

  /**
   * Indica que hay una relaci�n 1:1 entre esta entidad y la instancia de la otra clase. En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr� el array de instancias
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param int $loadStyle determina si la relaci�n se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta �ltima es la acci�n predeterminada). Las constantes est�n en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definici�n
   */
  public function addRelation1x1($attribute, $fieldName, $class, $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087E85
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::OneToOne, 'class'=>$class, 'loadStyle'=>$loadStyle, 'table'=>'', 'associatedFieldName'=>'');
    return $this;
    // Bouml preserved body end 00087E85
  }

  /**
   * Indica que hay una relaci�n entre esta entidad y las instancias de otra clase. Las relaciones que soportan son 1:N, N:M y 1:1. En el caso de la N:M debe indicarse la tabla intermedia.
   * En forma predeterminada estas relaciones son cargadas como LAZY.
   * 
   * @param string $attribute el nombre del atributo que tendr� el array de instancias
   * @param string $table s�lo se usa en relaciones N:M e indica el nombre de la tabla intermedia
   * @param string $fieldName el nombre del atributo donde se mapea el identificador de la clase actual
   * @param string $class la clase asociada (la clase de las instancias relacionadas)
   * @param string $asociatedFieldName el nombre del atributo donde se mapea el identificador de la clase asociada
   * @param int $loadStyle determina si la relaci�n se debe cargar inmediamente al recuperar la clase o si se hace una carga diferida en el momento en que se use (esta �ltima es la acci�n predeterminada). Las constantes est�n en ORM_LOAD_STYLE.
   * 
   * @return ORMDefinition la propia definici�n
   */
  public function addRelationNxM($attribute, $table, $fieldName, $class, $asociatedFieldName = "", $loadStyle = ORMDefinition::LAZY_LOAD)
  {
    // Bouml preserved body begin 00087F05
    if ($asociatedFieldName=="") {
        if (isset ($this->orm)) {
            $relatedDefinition= $this->orm->getDefinition($class)->getFieldDefinition('id');
        } else {
        //Este moco es para "compatibilizar que el PHP no soporte el "$class::" para invocar métodos estáticos
        $params = array();
        $relatedDefinition= $this->invokeStaticMethod($class, "define", $params)->getFieldDefinition('id');
            //$relatedDefinition= $class::define()->getFieldDefinition('id');No es compatible con PHP del server
        }
        $asociatedFieldName=$relatedDefinition['fieldName'];
    }
    $this->ORMData['relations'][$attribute]=array('fieldName'=>$fieldName, 'type'=>ORM_RELATION_TYPE::ManyToMany, 'class'=>$class, 'table'=>$table, 'loadStyle'=>$loadStyle, 'associatedFieldName'=>$asociatedFieldName);
    return $this;
    // Bouml preserved body end 00087F05
  }

}
?>