<?php 

require_once("CLL_config.php");

/**
 * Centralized Login Library
 *
 * This is the php client's library to interact with the centralized login.
 *
 * As follows, we will show you how to use the library. Let's assume we're working in a client's authentication, so
 * we asks the client for his user and password, with this information, we need to construct a CLL_Login object
 * and ask to the centralized login server if the user and password is ok, and if the authentication is ok, we want
 * to bring back the perfil that the login has in the web application, all in one step, we don't want to asks the server
 * first if the user is ok and then ask for the metadata.
 * <code>
 * //We already have the user in $user and the pass in $password
 * $login = CLL_newLogin($user, $pass);
 *
 * //Ask to the server if the login is ok, and bring back the perfil metadata
 * $answer = Cll_Authenticate($login, array( NewMetadata("perfil", CLL_METADATA_APP_LOGIN), CLL_NewMetadata("NombreApellido", CLL_METADATA_LOGIN) );
 *
 * //Check if the server could be reached
 * if ($answer->getServerStatus() == false)
 *   die("Midway Midway, the server is down!!!");
 * 
 * if ($answer->getError() == true)
 * {
 *   //The server couldn't be reached, so let's use local authentification and log the error
 *   $error = $answer->getErrorDescription();
 *   //Here put the code to do local authentification
 * }
 * else
 * {
 *   //The server answered, let's see if it's a valid answer
 *   if ($answer->getAnswer() == true)
 *   {
 *     //The login is ok, let's get the metadata
 *     $metadata = $answer->getMetadata();
 *     //Let's get the perfil data
 *     $data = $metadata[CLL_METADATA_APP_LOGIN]["perfil"];
 *     //Store client id for future uses.
 *     $login = $answer->getLogin();
 *     $id_client = $login->getClientId(); 
 *   }
 *   else
 *   {
 *     //The login is not valid
 *     $error = $answer->getAnswerDescription();
 *   }
 * }
 * </code>
 *
 * Don't forget to add code into the file CLL_user_authentication.php
 *
 * @author Carla Ares <cares@iplan.com.ar> - Rodrigo Dominguez <rorra@rorra.com.ar>
 * @version 1.0
 * @package CentralizedLogin
 */

/**
 * Authenticates a login
 *
 * The centralized login expects the username and password, and returns whether the user can access the web application or not.
 * If you set the users's client id, then it will be used to authenticate the user.
 * Optionally, you can send an array of metadata keys, so the centralized login returns the metadata if the authentication is successful.
 * In example:
 * 
 * <code>
 * $login = CLL_NewLogin($usuario, $password);
 * $answer = Cll_Authenticate($login, array( NewMetadata("perfil", CLL_METADATA_APP_LOGIN), NewMetadata("NombreApellido", CLL_METADATA_USER) );
 * </code>
 *
 * In the answer instance:
 * 
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the login is valid or not.
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - metadata will be the metadata array with keys, types and values, if metadata keys were requested.
 *   - login will be the login instance if the user was authenticated
 *   - client will be the client instance if the login is related to a client
 *
 * @param CLL_Login $CLL_Login The login's object
 * @param array $metadata Array of metadata objects
 * @return CLL_Answer Centralized Login Answer.
 */
function CLL_Authenticate($CLL_Login, $metadata = null) 
{
  $client = CLL_Initiate_SOAP_Connection();
//print_r(CLL_EncryptCLLLogin(clone($CLL_Login)));
  $result = $client->call('Authenticate',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($CLL_Login)),
				'metadata' => CLL_EncryptCLLMetadatas($metadata)));
  //print_r(CLL_Get_Answer($result));
  //print_r($client);
  return(CLL_Get_Answer($result));
}

/**
 * Modificates a password
 *
 * The centralized login expects the username, old password and new password, and returns wheter the password 
 * could be modified or not. 
 * If you set the users's client id, then it will be used too to authenticate the user.
 *
 * In example:
 * <code>
 * $login = CLL_NewLogin($usuario);
 * $answer = Cll_ModifyPassword($login, $new_password);
 * </code>
 *
 * In the answer instance:
 * 
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the password could be modified or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - login will be the login instance if the user was authenticated
 *   - client will be the client instance if the login is related to a client
 *
 * @param CLL_Login $CLL_Login The login's object
 * @param string $new_password The new login's password.
 * @return CLL_Answer Centralized Login Answer.
 */
function CLL_ModifyPassword($CLL_Login, $new_password) 
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('ModifyPassword',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($CLL_Login)),
                                'new_password' => CLL_EncryptValue($new_password)));
  return(CLL_Get_Answer($result));
}


/**
 * Modificates metadata
 *
 * Example:
 * <code>
 * $answer = CLL_ModifyMetadata(array(CLL_NewMetadata("perfil", CLL_METADATA_APP_LOGIN, "admin", null, "juan")));
 * </code>
 *
 * Please, check CLL_NewMetadata in CLL_data_definition.php
 *
 * In the answer instance:
 * 
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the funcion was executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - metadata will be the matadata modified with its values
 *
 * NOTE: Every metadata instance has a error and errorDescription variable, because $metadata it's an array of metadata
 * and some matadata may work and other not, so you should check every error and errorVariables of the metadata instance
 *
 * @param array $metadata Arrays of metadata objects
 * @return CLL_Answer Centralized Login Answer.
 */
function CLL_ModifyMetadata($metadata = NULL) 
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('ModifyMetadata',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'metadata' => CLL_EncryptCLLMetadatas($metadata)));
  return(CLL_Get_Answer($result));
}

/**
 * Gets metadata
 *
 * Example:
 * <code>
 * $answer = CLL_GetMetadata(array(CLL_NewMetadata("perfil", CLL_METADATA_APP_LOGIN, "admin")));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the funcion was executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - metadata will be the matadata retrieved with its values
 *
 * NOTE: Every metadata instance has a error and errorDescription variable, because $metadata it's an array of metadata
 * and some matadata may work and other not, so you should check every error and errorVariables of the metadata instance
 *
 * @param array $metadata Arrays of metadata Objects. 
 * @return CLL_Answer Centralized Login Answer.
 */
function CLL_GetMetadata($metadata = NULL) 
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetMetadata',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'metadata' => CLL_EncryptCLLMetadatas($metadata)));
  return(CLL_Get_Answer($result));
}

/**
 * Deletes metadata
 *
 * Example:
 * <code>
 * $answer = CLL_DeleteMetadata(array(CLL_NewMetadata("perfil", CLL_METADATA_APP_LOGIN, "admin")));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the funcion was executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - metadata will be the matadata deleted with its values
 *
 * @param array $metadata Arrays of metadata Objects. 
 * @return CLL_Answer Centralized Login Answer.
 */
function CLL_DeleteMetadata($metadata = NULL) 
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('DeleteMetadata',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'metadata' => CLL_EncryptCLLMetadatas($metadata)));
  return(CLL_Get_Answer($result));
}

/**
 * Creates a login (never an administration login, but a sublogin, example, CGP_WhatEver)
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin("0101232_pablito", "password");
 * $answer = CLL_CreateLogin($login, array(CLL_WEB_APPLICATION_ID));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the login was created or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - login will be the login that was created
 *   - client will be the client associated to the login
 *
 * @param CLL_Login $login Login to be created.
 * @param array $metadata Arrays of metadata Objects. 
 * @param webapplications_id $webapplications_id Array of ids of the webapplications where the login has access
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_CreateLogin($login, $webapplications_id)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('CreateLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login)),
                                'webapplications_id' => $webapplications_id));
  return(CLL_Get_Answer($result));
}

/**
 * Deletes a login (actually, it just deletes the login if it has not access to other web applications, if it does
 * have access to other web applications, then it just deletes the relation between the login and this web application.
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin("010321321_juan", "password");
 * $answer = CLL_DeleteLogin($login, array(CLL_WEB_APPLICATION_ID));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the login was deleted or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - login will be the login that was deleted
 *   - client will be the client associated to the login
 *
 * @param CLL_Login $login Login to be deleted
 * @param webapplications_id web array Array of ids of the webapplications where the login has access
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_DeleteLogin($login, $webapplications_id)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('DeleteLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login)),
                                'webapplications_id' => $webapplications_id));
  return(CLL_Get_Answer($result));
}

/**
 * Add privileges to the login to access other web applications
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin("0101232_pablito", "password");
 * $answer = CLL_LoginAddWebAccesses($login, array(CLL_WEB_APPLICATION_ID));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the privileges were given or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - login will be the login instance with all of its data
 *   - client will be the client associated to the login
 *
 * @param CLL_Login $login Login to be created.
 * @param array $metadata Arrays of metadata Objects. 
 * @param webapplications_id $webapplications_id Array of ids of the webapplications where the login has access
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_LoginAddWebAccesses($login, $webapplications_id)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('LoginAddWebAccesses',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login)),
                                'webapplications_id' => $webapplications_id));
  return(CLL_Get_Answer($result));
}

/**
 * Remove privileges to the login to access other web applications
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin("0101232_pablito", "password");
 * $answer = CLL_LoginDeleteWebAccesses($login, array(CLL_WEB_APPLICATION_ID));
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the privileges were deleted or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - login will be the login instance with all of its data
 *   - client will be the client associated to the login
 *
 * @param CLL_Login $login Login to be created.
 * @param array $metadata Arrays of metadata Objects. 
 * @param webapplications_id $webapplications_id Array of ids of the webapplications where the login has access
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_LoginDeleteWebAccesses($login, $webapplications_id)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('LoginDeleteWebAccesses',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login)),
                                'webapplications_id' => $webapplications_id));
  return(CLL_Get_Answer($result));
}


/**
 * Gets a list of logins by metadata existence query
 *
 * Example:
 * <code>
 * $answer = CLL_GetLoginsByMetadata(array(CLL_NewMetadata("perfil", CLL_METADATA_APP_LOGIN, "admin")));
 * print_r($answer);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the function were executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - logins will be all the logins retrieved by the query
 *
 * @param array $metadata Arrays of metadata Objects. 
 * @param int $limit How many records to be retrieved
 * @param int $from Where to start to fetch records
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_GetLoginsByMetadata($metadata = NULL, $limit = NULL, $from = NULL)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetLoginsByMetadata',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'metadata' => CLL_EncryptCLLMetadatas($metadata),
                                'limit' => CLL_EncryptValue($limit),
                                'from' => CLL_EncryptValue($from)));
  return(CLL_Get_Answer($result));
}

/**
 * Gets a list of logins by metadata value query
 *
 * Example:
 * <code>
 * $answer = CLL_GetLoginsByMetadata(array(CLL_NewMetadata("perfil", CLL_METADATA_APP_LOGIN, "admin")));
 * print_r($answer);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the function were executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - logins will be all the logins retrieved by the query
 *
 * @param array $metadata Arrays of metadata Objects. 
 * @param int $limit How many records to be retrieved
 * @param int $from Where to start to fetch records
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_GetLoginsByMetadataExistence($metadata = NULL, $limit = NULL, $from = NULL)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetLoginsByMetadataExistence',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'metadata' => CLL_EncryptCLLMetadatas($metadata),
                                'limit' => CLL_EncryptValue($limit),
                                'from' => CLL_EncryptValue($from)));
  return(CLL_Get_Answer($result));
}

/**
 * Gets the list of web applications where the login has access
 * 
 * Example:
 * <code>
 * $login = CLL_NewLogin('cares');
 * $answer = CLL_GetWebApplicationsByLogin($login);
 * print_r($answer);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the function were executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 *   - web_applications will be all the web applications retrieved by the query
 *
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized Login Answer
 */
function CLL_GetWebApplicationsByLogin($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetWebApplicationsByLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login))));
  return(CLL_Get_Answer($result));
}

/**
 * Recovers the password for the login.
 *
 * The process is as follows, an email is sent to the login telling that someone requested to reset the password, and to click the link in the email if he wants to reset the password.
 * Once he clicks the links, he is redirected to a page in the unified login telling him that the password was reseted and that the new password has been sent by email.
 * The client will recive another email with the new password.
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin('cares');
 * $answer = CLL_RecoverPassword($login);
 * print_r($answer);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the function were executed ok or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 * 
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized login Answer
 */
function CLL_RecoverPassword($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('RecoverPassword',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login))));
  return(CLL_Get_Answer($result));
}

/**
 * Recovers the password for the login.
 *
 * The process is as follows, an email is sent to the login telling that someone requested to reset the password, and to click the link in the email if he wants to reset the password.
 * Once he clicks the links, he is redirected to a page in the unified login telling him that the password was reseted and that the new password has been sent by email.
 * The client will recive another email with the new password.
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin('cares');
 * $answer = CLL_IsFirstLogin($login);
 * print_r($answer);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the user has never login to the unified login or not
 *   - answerDescription will say if some error happens (the error description) or if everything was ok
 * 
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized login Answer
 */
function CLL_IsFirstLogin($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('IsFirstLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login))));
  return(CLL_Get_Answer($result));
}

/**
 * Gets the link for the login so he could be informed about the unified login functionality.
 * This functionality is exclusive for Iplan Control.
 * Once you check that the login has never logged in into the unified login, you should call this function and
 * rediret the user to the link.
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin('cares');
 * $answer = CLL_IsFirstLogin($login);
 * if ($answer->getAnswer() == true) {
 *   $answe2 = CLL_GetLinkForFirstLogin($login);
 *   if ($answer2->getAnswer() == true) {
 *     $link = $answer2->getAnswerDescription();
 *   }
 * }
 * print_r($answer);
 * print_r($answer2);
 * </code>
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the user has never login to the unified login or not
 *   - answerDescription will say if some error happens (the error description) or the link
 * 
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized login Answer
 */
function CLL_GetLinkForFirstLogin($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetLinkForFirstLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login))));
  return(CLL_Get_Answer($result));
}

/**
 * It just query a login
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin('cares');
 * $result = CLL_GetLogin($login);
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true or false, indicating whether the user has never login to the unified login or not
 *   - answerDescription will say if some error happens (the error description) or the link
 *   - login will be the login instance
 * 
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized login Answer
 */
function CLL_GetLogin($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('GetLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
                                'login' => CLL_EncryptCLLLogin(clone($login))));
  return(CLL_Get_Answer($result));
}

/**
 * Modifies the login and email of the login
 *
 * Example:
 * <code>
 * $login = CLL_NewLogin('0012689_test');
 * $login->setEmail('test@test.com.ar');
 * $login->setName('testeo');
 * $result = CLL_ModifyLogin($login);
 *
 * In the answer instance:
 *
 *   - serverStatus will be true or false whether the server answered or not
 *   - answer will be true if the login's data were modify, otherwise it will be false
 *   - answerDescription will say if some error happens (the error description) or the link
 *   - login will be the login instance
 * 
 * @param CLL_Login $login The login
 * @return CLL_Answer Centralized login Answer
 */
function CLL_ModifyLogin($login)
{
  $client = CLL_Initiate_SOAP_Connection();
  $result = $client->call('ModifyLogin',
                          array('webapplication_id' => CLL_WEB_APPLICATION_ID,
			  'login' => CLL_EncryptCLLLogin(clone($login))));
  //print_r($result);
  return(CLL_Get_Answer($result));
}

?>
